/*
 * Copyright [yyyy] [name of copyright owner]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


// Anonymous function start
//
(function( window, undefined )
{

// reference
var Config			= window.h5glib.Config;
var Debug			= window.h5glib.Debug;
var Command			= window.h5glib.Command;
var Task			= window.h5glib.Task;
var SoundTask		= window.h5glib.SoundTask;
var ReadyTask		= window.h5glib.ReadyTask;
var ImageAnimator	= window.h5glib.ImageAnimator;
var Message			= window.h5glib.Message;
var MessageHandler	= window.h5glib.MessageHandler;
var MessageManager	= window.h5glib.MessageManager;
var SceneStatus		= window.h5glib.SceneStatus;
var Scene			= window.h5glib.Scene;

/**
 * RenderInfo
 */
var RenderInfo =
{
	chipPixels	: 32
};

/**
 * Res
 */
var Res =
{
	String :
	{
		HTML_USAGE		: "<p>" +
						  "＜キー入力ができない場合はゲームの画面をクリックしてみてください＞<br>" +
						  "[↑]: 左上, [←]: 左下, [↓]: 右下, [→]: 右上 <br>" +
						  "</p>"
	},
	Color :
	{
		TEXTBOX			: "rgba(192, 80, 77, 0.7)",
		BORDER_TEXTBOX	: "rgb(192, 192, 192)",
		FONT_TEXTBOX	: "rgb(255, 255, 255)"
	},
	Font :
	{
		LARGE			: "bold 18px 'ＭＳ Ｐゴシック'",
		SMALL			: "bold 14px 'ＭＳ Ｐゴシック'"
	}
};

/**
 * MessageType
 */
var MessageType =
{
	NONE			: 0
};

/**
 * EventType
 */
var EventType =
{
	NONE	: "none"
};

/**
 * StageTask
 */
var StageTask = function( scene )
{
	this.command		= scene.command;	// input

	this.map			= null,
	this.getMapHeight	= function() { return this.map.length; };
	this.getMapWidth	= function() { return this.map[0].length; };

	this.scroll			= { x : 0, y : 0 };
};
StageTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.setData = function( scene, map )
	{
		this.map	= map;
	};

	/**
	 * 
	 */
	pt.updateScrollValue = function( scene )
	{
		var context		= scene.context;
		var canvas		= context.canvas;
		var resource	= scene.data.resource;
		var viewPoint	= scene.viewPoint;

		var cx	= canvas.width  / RenderInfo.chipPixels / 2;
		var cy	= canvas.height / RenderInfo.chipPixels / 2;
		var mw	= this.getMapWidth();
		var mh	= this.getMapHeight();

		// scroll.x
		if ( viewPoint.ad.x <= cx )
		{
			this.scroll.x	= 0;
		}
		else if ( cx < viewPoint.ad.x && viewPoint.ad.x < mw - cx )
		{
			this.scroll.x	= viewPoint.ad.x - cx;
		}
		else
		{
			this.scroll.x	= mw - ( cx * 2 );
		}
		// scroll.y
		if ( viewPoint.ad.y <= cy )
		{
			this.scroll.y	= 0;
		}
		else if ( cy < viewPoint.ad.y && viewPoint.ad.y < mh - cy )
		{
			this.scroll.y	= viewPoint.ad.y - cy;
		}
		else
		{
			this.scroll.y	= mh - ( cy * 2 );
		}
	};
	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var context		= scene.context;
		var canvas		= context.canvas;
		var resource	= scene.data.resource;
		var viewPoint	= scene.viewPoint;

		context.save();
		//context.clearRect( 0, 0, canvas.width, canvas.height );

		// update scroll
		this.updateScrollValue( scene );

		var cells_w	= canvas.width  / RenderInfo.chipPixels;
		var cells_h	= canvas.height / RenderInfo.chipPixels;

		var scroll_x	= Math.floor( this.scroll.x );
		var scroll_y	= Math.floor( this.scroll.y );
		var offset_x	= Math.floor( ( this.scroll.x % 1 ) * RenderInfo.chipPixels );
		var offset_y	= Math.floor( ( this.scroll.y % 1 ) * RenderInfo.chipPixels );

		var image	= resource.image.mchip.data;
		var image_tr= resource.image.mchip_tr.data;
		for ( var i = 0; i <= cells_h; i++ )
		{
			for ( var j = 0; j <= cells_w; j++ )
			{
				if ( i == cells_h && offset_y == 0 ) { break; }
				if ( j == cells_w && offset_x == 0 ) { continue; }

				// チップを描画
				var mv	= this.map[ scroll_y + i ][ scroll_x + j ];
				var mx	= mv % 10;
				var my	= Math.floor( ( mv % 100 ) / 10 );
				this.drawChip( context, image, cells_w, cells_h, i, j, scroll_x, scroll_y, offset_x, offset_y, mx, my );

				// 透明チップを描画
				var mx_tr	= Math.floor( ( mv % 1000 ) / 100 );
				var my_tr	= Math.floor( ( mv % 10000 ) / 1000 );
				if ( mx_tr != 0 || my_tr != 0 )
				{
					this.drawChip( context, image_tr, cells_w, cells_h, i, j, scroll_x, scroll_y, offset_x, offset_y, mx_tr, my_tr );
				}
			}
		}
		context.restore();
	};
	/**
	 * 
	 */
	pt.drawChip = function( context, image, cells_w, cells_h, i, j, scroll_x, scroll_y, offset_x, offset_y, mx, my )
	{
		var sx = mx * RenderInfo.chipPixels;
		var sy = my * RenderInfo.chipPixels;
		var dx = j * RenderInfo.chipPixels;
		var dy = i * RenderInfo.chipPixels;
		var fw = RenderInfo.chipPixels;
		var fh = RenderInfo.chipPixels;

		// x
		if ( j == 0 )
		{
			sx += offset_x;
			fw -= offset_x;
		}
		else
		{
			dx -= offset_x;
			if ( j == cells_w )
			{
				fw = offset_x;
			}
		}
		// y
		if ( i == 0 )
		{
			sy += offset_y;
			fh -= offset_y;
		}
		else
		{
			dy -= offset_y;
			if ( i == cells_h )
			{
				fh = offset_y;
			}
		}
		// draw
		context.drawImage( image, sx, sy, fw, fh, dx, dy, fw, fh );
	};
})( StageTask.prototype );

/**
 * ActorListTask
 */
var ActorListTask = function( scene )
{
	this.command	= scene.command;	// input
	this.visible	= [];
};
ActorListTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.setData = function( scene, actors )
	{
		this.child	= null;

		// load actorList
		for ( var i = 0; i < actors.length; i++ )
		{
			var task	= new ActorTask();
			task.init( scene, i, actors[i] );
			// add
			if ( this.child == null )
			{
				this.child = task;
			}
			else
			{
				this.child.append( task );
			}
		}
	};

	/**
	 * 
	 */
	pt.processAction = function( scene )
	{
		for ( var actor = this.child; actor != null; actor = actor.next )
		{
			actor.processAction( scene );
		}
	};
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		this.processAction( scene );

		return this.updateChildren( scene );
	};
	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		this.pickVisible( scene );

		var context		= scene.context;
		var canvas		= context.canvas;
		var stage		= scene.stageTask;

		context.save();

		for ( var i = 0; i < this.visible.length; i++ )
		{
			var actor	= this.visible[i];
			var image	= actor.animator.image;

			var x	= Math.floor( ( actor.ad.x - stage.scroll.x ) * RenderInfo.chipPixels );
			var y	= Math.floor( ( actor.ad.y - stage.scroll.y ) * RenderInfo.chipPixels );

			var sy	= actor.DIR2SY[ actor.ad.dir ] * actor.animator.fh;

			context.drawImage(
				image,
				actor.animator.fx,			// sx
				sy,							// sy
				actor.animator.fw,			// sw
				actor.animator.fh,			// sh
				x - actor.animator.fw / 2,	// dx
				y - actor.animator.fh / 2,	// dy
				actor.animator.fw,			// dw
				actor.animator.fh			// dh
			);
		}
		context.restore();
	};
	/**
	 * 
	 */
	pt.pickVisible = function( scene )
	{
		var context		= scene.context;
		var canvas		= context.canvas;
		var stage		= scene.stageTask;

		var sw	= canvas.width / RenderInfo.chipPixels;
		var sh	= canvas.height / RenderInfo.chipPixels;

		this.visible.length = 0;
		for ( var actor = this.child; actor != null; actor = actor.next )
		{
			if ( stage.scroll.x <= actor.ad.x && actor.ad.x <= stage.scroll.x + sw &&
				 stage.scroll.y <= actor.ad.y && actor.ad.y <= stage.scroll.y + sh )
			{
				this.visible.push( actor );
			}
		}
	};
})( ActorListTask.prototype );

/**
 * ActorStatus
 */
var ActorStatus =
{
	STATIC	: 0,
	ACTIVE	: 1,
	MORTAL	: 2
};

/**
 * ActorTask
 */
var ActorTask = function()
{
	this.animator	= null;
	this.id			= 0;
	// actor data
	this.ad			= null;
	// step data
	this.step		=
	{
		goal	: { x : 0, y : 0, dir : 0 },
		count	: 0,
		locked	: false
	}
};
ActorTask.prototype = new Task();

(function( pt )
{
	pt.DIR2STEP		= [ { x : 1, y : 1 }, { x : 1, y : -1 }, { x : -1, y : -1 }, { x : -1, y : 1 } ];
	pt.DIR2SY		= [ 1, 3, 2, 0 ];
	pt.STEP_VAL		= 0.4;
	pt.STEP_COUNT	= 4;

	/**
	 * 
	 */
	pt.init = function( scene, id, ad )
	{
		this.id		= id;
		this.ad		= ad;

		this.moveTo( scene.stageTask, this.ad.x, this.ad.y );
		this.command	= ( id == 0 ) ? scene.command : new AICommand();

		var image		= scene.data.resource.image.actors[ this.ad.type ].data;
		var animation	= scene.data.animation;
		this.interfaces	=
		[
			{ update : this.updateActive,	animator : new ImageAnimator( image, animation.actorAction ) },
			{ update : this.updateActive,	animator : new ImageAnimator( image, animation.actorAction ) }
		];
		this.setStatus( this.ad.status );
	};
	/**
	 * 
	 */
	pt.getGoalPos = function( val )
	{
		var step = this.DIR2STEP[ this.ad.dir ];
		var pos =
		{
			x : this.ad.x + val * step.x,
			y : this.ad.y + val / 2 * step.y
		};
		return pos;
	};
	/**
	 * 
	 */
	pt.moveTo = function( stage, x, y )
	{
		this.ad.x = x;
		this.ad.y = y;
	};
	/**
	 * 
	 */
	pt.canMoveTo = function( stage, x, y )
	{
		var arr = [ -0.3, 0.3 ];

		for ( var i = 0; i < arr.length; i++ )
		{
			var my = Math.floor( y + arr[i] );
			if ( my < 0 || stage.getMapHeight() <= my ) { return false; }

			for ( var j = 0; j < arr.length; j++ )
			{
				var mx = Math.floor( x + arr[j] );
				if ( mx < 0 || stage.getMapWidth() <= mx ) { return false; }
/*
				var value = stage.map[ my ][ mx ];
				if ( value != 0 && value != Config.actorValBase + this.id )
				{
					return flase;
				}
*/
			}
		}
		return true;
	};
	/**
	 * 
	 */
	pt.processAction = function( scene )
	{
		if ( this.status == ActorStatus.ACTIVE && !this.step.locked )
		{
			this.command.update();

			// key
			if ( this.command.tbl.up )
			{
				this.ad.dir			= 2;
				var pos = this.getGoalPos( this.STEP_VAL );
				this.step.goal.x	= pos.x;
				this.step.goal.y	= pos.y;
				this.step.count		= this.STEP_COUNT;
			}
			else if ( this.command.tbl.left )
			{
				this.ad.dir			= 3;
				var pos = this.getGoalPos( this.STEP_VAL );
				this.step.goal.x	= pos.x;
				this.step.goal.y	= pos.y;
				this.step.count		= this.STEP_COUNT;
			}
			else if ( this.command.tbl.down )
			{
				this.ad.dir			= 0;
				var pos = this.getGoalPos( this.STEP_VAL );
				this.step.goal.x	= pos.x;
				this.step.goal.y	= pos.y;
				this.step.count		= this.STEP_COUNT;
			}
			else if ( this.command.tbl.right )
			{
				this.ad.dir			= 1;
				var pos = this.getGoalPos( this.STEP_VAL );
				this.step.goal.x	= pos.x;
				this.step.goal.y	= pos.y;
				this.step.count		= this.STEP_COUNT;
			}
		}
	};
	/**
	 * 
	 */
	pt.updateActive = function( scene )
	{
		var upd = false;
		var stage	= scene.stageTask;

		// step
		if ( this.step.count != 0 )
		{
			this.step.count--;
			if ( this.step.count == 0 ) { this.step.locked = false; }
			var newX = ( this.step.count ) ? ( this.ad.x + this.step.goal.x ) / 2 : this.step.goal.x;
			var newY = ( this.step.count ) ? ( this.ad.y + this.step.goal.y ) / 2 : this.step.goal.y;

			if ( this.canMoveTo( scene.stageTask, newX, this.ad.y ) )
			{
				this.moveTo( stage, newX, this.ad.y );
				upd = true;
			}
			if ( this.canMoveTo( scene.stageTask, this.ad.x, newY ) )
			{
				this.moveTo( stage, this.ad.x, newY );
				upd = true;
			}
		}
		// animation
		if ( this.animator.isActive() )
		{
			upd = this.animator.proceed();
		}
		else
		{
			var r = Math.floor( Math.random() * 10 );
			if ( r == 1 )
			{
				this.animator.start();
				upd = true;
			}
		}
		return upd;
	};
	/**
	 * 
	 */
	pt.updateMortal = function( scene )
	{
		var upd = false;
		// animation
		if ( this.animator.isActive() )
		{
			upd = this.animator.proceed();
		}
		return upd;
	};
})( ActorTask.prototype );

/**
 * DebugTask
 */
var DebugTask = function( scene )
{
	this.command	= scene.command;	// input
	this.info		= null;
	this.lag		= 0;
};
DebugTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL	= 10;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		// lag
		if ( this.lag )
		{
			this.lag--;
		}
		else
		{
			if ( this.command.tbl.debug )
			{
				this.info	= "viewPoint.ad.x=" + scene.viewPoint.ad.x + ", viewPoint.ad.y=" + scene.viewPoint.ad.y;
				this.lag	= this.LAG_VAL;
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		// output debug info
		if ( this.info )
		{
			Debug.print( this.info );
			this.info = null;
		}
	};
})( DebugTask.prototype );


/**
 * AICommand
 */
var AICommand = function()
{
	this.tbl.up		= 0;
	this.tbl.left	= 0;
	this.tbl.down	= 0;
	this.tbl.right	= 0;
};
AICommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function()
	{
		this.clear();

		var r = Math.floor( Math.random() * 12 );
		if      ( r == 2 )	{ this.tbl.up		= 1; }
		else if ( r == 0 )	{ this.tbl.left		= 1; }
		else if ( r == 3 )	{ this.tbl.down		= 1; }
		else if ( r == 1 )	{ this.tbl.right	= 1; }
	};
})( AICommand.prototype );

/**
 * InputCommand
 */
var InputCommand = function()
{
	this.tbl.up		= 0;
	this.tbl.left	= 0;
	this.tbl.down	= 0;
	this.tbl.right	= 0;

	this.tbl.fire	= 0;
	this.tbl.seek	= 0;
	this.tbl.editor	= 0;
	this.tbl.escape	= 0;
	this.tbl.debug	= 0;
};
InputCommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function() {};
	/**
	 * 
	 */
	pt.handleSysEvent = function( event )
	{
		var type = event.type.toLowerCase();
		if ( type.substring( 0, 3 ) == "key" )
		{
			var value = 0;
			if      ( type == "keydown" ) { value = 1; }
			else if ( type == "keyup"   ) { value = 0; }

			switch ( event.keyCode )
			{
				case 38: this.tbl.up	= value; event.prevent( event.orgEvent ); break;	// ↑
				case 37: this.tbl.left	= value; event.prevent( event.orgEvent ); break;	// ←
				case 40: this.tbl.down	= value; event.prevent( event.orgEvent ); break;	// ↓
				case 39: this.tbl.right	= value; event.prevent( event.orgEvent ); break;	// →

				case 90: this.tbl.fire	= value; break;	// Z
				case 32: this.tbl.seek	= value; break;	// space;
				case 69: this.tbl.editor= value; break;	// E
				case 27: this.tbl.escape= value; break;	// ESC
				case 81: this.tbl.debug	= value; break;	// Q
			}
		}
	};
})( InputCommand.prototype );

/**
 * StageScene
 */
var StageScene = function( app, name, online )
{
	this.app			= app;
	this.name			= name;
	this.data			= null;

	this.command		= new InputCommand();
	this.viewPoint		= null;

	// create task
	this.stageTask		= new StageTask( this );
	this.actorListTask	= new ActorListTask( this );
	this.soundTask		= new SoundTask( this );
	this.debugTask		= new DebugTask( this );
	// create list
	this.stageTask.append( this.actorListTask );
	this.stageTask.append( this.soundTask );
	this.stageTask.append( this.debugTask );
	// head of task list
	this.child		= this.stageTask;
	this.setStatus( SceneStatus.READY );

	// message handler
	//this.msgManager		= new MessageManager();
};
StageScene.prototype = new Scene();

(function( pt )
{
	/**
	 * 
	 */
	pt.init = function()
	{
	};
	/**
	 * 
	 */
	pt.show = function()
	{
		this.setUsage( Res.String.HTML_USAGE );

		this.command.clear();
		this.holdContext();
		this.draw( this );
	};

	/**
	 * 
	 */
	pt.setData = function( data )
	{
		this.data	= data;

		// load stage
		this.stageTask.setData( this, data.map );
		// load actorList
		this.actorListTask.setData( this, data.actors );
		// init viewPoint
		this.viewPoint	= this.actorListTask.child;
	};
	/**
	 * 
	 */
	pt.getData = function()
	{
		return this.data;
	};
	/**
	 * 
	 */
	pt.loadData = function( data )
	{
		try
		{
			var self = this;

			// load resource
			this.app.loadResource( this.name, data.resource );
			// set data
			this.setData( data );
			// set status
			window.setTimeout( function() { self.setStatus( SceneStatus.RUNNING ); self.show(); }, Config.loadInterval + 300 );
		}
		catch ( e )
		{
			this.app.kill();
			Debug.alertError( e );
		}
	};
})( StageScene.prototype );


// Expose
if ( !window.h5glib ) { window.h5glib = {}; }
window.h5glib.StageScene	= StageScene;

// Anonymous function end
//
})( window );
