package com.kurukurupapa.tryandroid.fw.view;

import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.RectF;

/**
 * 色相環ヘルパー
 */
public class BoxHsvColorPickerHueHelper {
	// 色相環の幅
	private static final int ARC_WIDTH = 24;
	// 色相環の色と色の隙間（単位：角度）
	private final float SPACE = 2.0f;
	private final float HARF_SPACE = SPACE / 2.0f;
	// 選択色の枠幅
	private static final float SELECT_WIDTH = 2.0f;
	private static final float SELECT_HARF_WIDTH = SELECT_WIDTH / 2.0f;

	// 色数
	private int hueNum;
	// 1つの色の角度
	private double colorAngle;
	// 選択されているインデックス
	private int selectIndex;

	// 半径
	private float radius;

	private Paint paint;
	private Paint selectPaint;

	public BoxHsvColorPickerHueHelper(int hueNum) {
		paint = new Paint();
		paint.setStyle(Paint.Style.STROKE);
		paint.setStrokeWidth(ARC_WIDTH);
		paint.setAntiAlias(true);

		selectPaint = new Paint();
		selectPaint.setStyle(Paint.Style.STROKE);
		selectPaint.setStrokeWidth(ARC_WIDTH + SELECT_WIDTH);
		selectPaint.setAntiAlias(true);
		selectPaint.setColor(Color.WHITE);

		setColorNum(hueNum);
	}

	public void setSize(float radius) {
		this.radius = Math.max(radius, getArcWidth());
	}

	public void setColorNum(int hueNum) {
		this.hueNum = hueNum;
		colorAngle = 360.0 / hueNum;
	}

	public void drawOval(Canvas canvas) {
		float r = radius - ARC_WIDTH * 0.5f - SELECT_HARF_WIDTH;
		RectF rect = new RectF(-r, -r, r, r);

		for (int i = 0; i < hueNum; i++) {
			// 描画する孤の位置を指定
			// 左：赤、上：黄、右：青、下：紺とする
			int startAngle = (int) Math.round(colorAngle * i);
			int sweepAngle = (int) Math.round(colorAngle);

			// 色と色の間には隙間を作る
			Path path = new Path();
			path.addArc(rect, Math.round(startAngle + HARF_SPACE),
					Math.round(sweepAngle - SPACE));

			// 彩度、明度は100%
			paint.setColor(getColorFromIndex(i));

			// 選択枠
			if (i == selectIndex) {
				Path selectPath = new Path();
				selectPath.addArc(rect, Math.round(startAngle),
						Math.round(sweepAngle));
				canvas.drawPath(selectPath, selectPaint);
			}

			// 描画
			canvas.drawPath(path, paint);
		}
	}

	private int getColorFromIndex(int index) {
		return Color.HSVToColor(new float[] { Math.round(colorAngle * index),
				1.0f, 1.0f });
	}

	public void setSelectColorFromXY(float x, float y) {
		// タッチされた位置の角度を求める
		float angle = (float) java.lang.Math.atan2(y, x);
		// need to turn angle [-PI ... PI] into unit [0....1]
		float unit = (float) (angle / (2.0 * Math.PI));
		if (unit < 0) {
			unit += 1;
		}

		// 選択したインデックス
		selectIndex = Math.round((hueNum - 1) * unit);
	}

	public void setSelectColorFromColor(int color) {
		float h = getHue(color);
		selectIndex = Math.round(hueNum * h / 360.0f);
	}

	public boolean contains(float r) {
		if (r <= radius) {
			if (r > radius - ARC_WIDTH) {
				return true;
			}
		}
		return false;
	}

	public float getHue() {
		return getHue(getColorFromIndex(selectIndex));
	}

	private float getHue(int color) {
		float hsv[] = new float[3];
		Color.colorToHSV(color, hsv);
		return hsv[0];
	}

	public float getArcWidth() {
		return ARC_WIDTH + SELECT_WIDTH;
	}

}
