/**
 * @file  MathUtility.h
 * @brief w֘A[eBeB.
 *
 * @author JIN
 *
 * Copyright (C) 2004- JIN All rights reserved.
 */
#ifndef MATHUTILITY_H_
#define MATHUTILITY_H_

#include <cmath>
#include <numeric>
#include <limits>
#include <functional>

#undef min
#undef max

namespace Math {

/////////////////////////////////////////////////////////////////////////////

//static const double PI = 3.14159265358979;
static const double PI = atan(1.0) * 4.0;
static const double PI_HALF = 0.5 * PI;
static const double PI_DOUBLE = PI + PI;
static const double PI_INVERSE = 1.0 / PI;

static const double DELTA = 1.0e-12;
static const double EPSILON = 1.0e-16;

/////////////////////////////////////////////////////////////////////////////

#ifndef _tstof
inline double tstof(LPCTSTR str)
{
	TCHAR* p;
	return _tcstod(str, &p);
}
#define _tstof Math::tstof
#endif	// #ifndef _tstof

#ifndef _tstoi
inline int tstoi(LPCTSTR str)
{
	TCHAR* p;
	return _tcstol(str, &p, 10);
}
#define _tstoi Math::tstoi
#endif	// #ifndef _tstof

/////////////////////////////////////////////////////////////////////////////

/**
 * double --> ϊ.
 */
template <typename T>
class DoubleTo : std::unary_function<double, T>
{
public:
	T operator()(double f)
	{
		ASSERT(std::numeric_limits<T>::min() <= f && f <= std::numeric_limits<T>::max());

		if (f > 0.0)
			return static_cast<T>(f + 0.5);
		else
			return static_cast<T>(f - 0.5);
	}
};

/**
 * double --> long double ϊ.
 */
template<>
class DoubleTo<long double> : std::unary_function<double, long double>
{
public:
	long double operator()(double f)
	{
		return f;
	}
};

/**
 * double --> double ϊ.
 */
template<>
class DoubleTo<double> : std::unary_function<double, double>
{
public:
	double operator()(double f)
	{
		return f;
	}
};

/**
 * double --> float ϊ.
 */
template<>
class DoubleTo<float> : std::unary_function<double, float>
{
public:
	float operator()(double f)
	{
		ASSERT(std::numeric_limits<float>::min() <= f && f <= std::numeric_limits<float>::max());

		return static_cast<float>(f);
	}
};

/////////////////////////////////////////////////////////////////////////////

/**
 * {l擾.
 */
template <typename T>
T Double(T data)
{
	return data + data;
}

/**
 * 悵l擾.
 */
template <typename T>
T square(T data)
{
	return data * data;
}

/**
 * sqrt(x^2 + y^2) 擾.
 */
inline double norm(double x, double y)
{
	return sqrt(square(x) + square(y));
}

/**
 * ől.
 */
template <typename T1, typename T2>
T1 Max(T1 lhs, T2 rhs)
{
	return lhs > (T1)rhs ? lhs : (T1)rhs;
}

/**
 * ŏl.
 */
template <typename T1, typename T2>
T1 Min(T1 lhs, T2 rhs)
{
	return lhs < (T1)rhs ? lhs : (T1)rhs;
}

/**
 * ŏlƍől̊Ԃɓ悤ɂ.
 */
template <typename T1, typename T2, typename T3>
T1 Round(T1 value, T2 valueMin, T3 valueMax)
{
	return Max(Min(value, valueMax), valueMin);
}

/**
 * ceil vZ邪A傤ǐl̂̂ +1 .
 */
inline double CeilPlus(double f)
{
	double fCeil = ceil(f);
	return fCeil == f ? fCeil + 1.0 : fCeil;
}

/**
 * x**y 擾.
 */
inline double Pow(double x, double y)
{
#if 0	// "pow"  "|Z + exp" ̕
	return pow(x, y);
#else
	return exp(y * log(x));
#endif
}

/**
 * x**y vZ.
 *  x ɑ΂ČJԂvZƂɎgp.
 */
class CPow
{
public:
	CPow(double x) : m_fLogX(log(x))
	{
	}

	/**
	 * x**y 擾.
	 */
	double operator()(double y) const
	{
		return exp(y * m_fLogX);
	}

private:
	/// log(x)
	double m_fLogX;
};

/**
 * log_x(y) vZ.
 *  x ɑ΂ČJԂvZƂɎgp.
 */
class CLog
{
public:
	CLog(double x) : m_fLogXInv(1.0 / log(x))
	{
	}

	/**
	 * log_x(y) 擾.
	 */
	double operator()(double y) const
	{
		return log(y) * m_fLogXInv;
	}

private:
	/// 1 / log(x)
	double m_fLogXInv;
};

/**
 * 10**x 擾.
 */
inline double pow10(double x)
{
#if 1	// pow10 ͑xł͂ȂxKv (10ỉĤ)
	return pow(10.0, x);
#else
	static const double ln10 = log(10.0);
	return exp(x * ln10);
#endif
}

/**
 * f ̏] (f, 2 * f, ...) ̂A
 * x ɍł߂擾.
 */
inline double Multiplize(double x, double f)
{
	ASSERT(f >= EPSILON);
	return floor(x / f + 0.5) * f;
}
/**
 * fBase ܂ f ̏] (fBase + f, fBase + 2 * f, ...) ̂A
 * x ɍł߂擾.
 */
inline double Multiplize(double x, double f, double fBase)
{
	ASSERT(f >= EPSILON);
	return floor((x - fBase) / f + 0.5) * f + fBase;
}

inline double MultiplizeUp(double x, double f)
{
	ASSERT(f >= EPSILON);
	return ceil(x / f) * f;
}
inline double MultiplizeUp(double x, double f, double fBase)
{
	ASSERT(f >= EPSILON);
	return ceil((x - fBase) / f) * f + fBase;
}

inline double MultiplizeDown(double x, double f)
{
	ASSERT(f >= EPSILON);
	return floor(x / f) * f;
}
inline double MultiplizeDown(double x, double f, double fBase)
{
	ASSERT(f >= EPSILON);
	return floor((x - fBase) / f) * f + fBase;
}

/**
 * z.
 * value1  value2 ̊Ԃ fRatio1 : fRatio2 ɕ.
 * (fRatio1 + fRatio2 == 1.0).
 */
inline double ProportionalDivisionSub(
	double value1, double value2, double fRatio1, double fRatio2)
{
	ASSERT(fabs(fRatio1 + fRatio2 - 1.0) < DELTA);
	return value1 * fRatio2 + value2 * fRatio1;
}

/**
 * value1  value2 ̊Ԃ fRatio : 1.0 - fRatio ɕ.
 * (fRatio: [0.0, 1.0]).
 */
inline double ProportionalDivision(double value1, double value2, double fRatio)
{
	return ProportionalDivisionSub(value1, value2, fRatio, 1.0 - fRatio);
}

/**
 * value1  value2 ̊Ԃ fRatio1 : fRatio2 ɕ.
 */
inline double ProportionalDivision(
	double value1, double value2, double fRatio1, double fRatio2)
{
	double fInverseSum = 1.0 / (fRatio1 + fRatio2);
	return ProportionalDivisionSub(value1, value2, fRatio1 * fInverseSum, fRatio2 * fInverseSum);
}

/**
 * num / den (؂Ȃ΁Anum / den + 1) vZ (^̂).
 */
template <typename T>
inline T RoundUpDivide(T num, T den)
{
	return (num + den - 1) / den;
}

/**
 * num / den (؂Ȃ΁Anum / den ̎ľܓ) vZ (^̂).
 */
template <typename T>
inline T RoundDivide(T num, T den)
{
	return (num + den / 2) / den;
}

/////////////////////////////////////////////////////////////////////////////

/**
 * őlƍŏl擾.
 */
template <typename T>
void GetMinMax(const T* data, ULONG_PTR nCount, T& dataMin, T& dataMax)
{
	dataMin = data[0];
	dataMax = data[0];
	for (ULONG_PTR i = 1; i < nCount; ++i) {
		if (dataMin > data[i])
			dataMin = data[i];
		else if (dataMax < data[i])
			dataMax = data[i];
	}
}

/**
 * ϒl擾.
 */
template <typename T>
double GetAverage(const T* data, ULONG_PTR nCount)
{
#if 0
	// I[o[t[邽߁AϒlvZ
	double avg = 0.0;
	for (ULONG_PTR i = 0; i < nCount; ++i) {
		avg = ProportionalDivision(avg, (double)data[i], 1, i);
	}
	return avg;
#else
	// I[o[t[CɂA܂avZB
	return std::accumulate(data, data + nCount, 0.0) / nCount;
#endif
}

/**
 * rhs ɂ̂ conv Kp plus.
 */
template <typename T, class CONVERTER>
class ConvertPlus : std::binary_function<T, T, T>
{
public:
	ConvertPlus(const CONVERTER& conv) : m_conv(conv)
	{
	}

	T operator()(T lhs, T rhs) const
	{
		return lhs + m_conv(rhs);
	}

private:
	const CONVERTER& m_conv;
};

/**
 * ϒl擾 (el conv Kp).
 */
template <typename T, class CONVERTER>
double GetAverage(const T* data, ULONG_PTR nCount, const CONVERTER& conv)
{
	return std::accumulate(
		data, data + nCount, 0.0, ConvertPlus<double, CONVERTER>(conv))
		/ nCount;
}

#if 0
/**
 * vf̕U擾.
 */
class VarianceConverter : std::unary_function<double, double>
{
public:
	VarianceConverter(double average) : m_average(average)
	{
	}

	double operator()(double value) const
	{
		return square(value - m_average);
	}

private:
	double m_average;
};

/**
 * U擾.
 */
template <typename T>
double GetVariance(const T* data, ULONG_PTR nCount, double average)
{
	typedef VarianceConverter VConv;
	typedef ConvertPlus<double, VConv> CPlus;
	return std::accumulate(
		data, data + nCount, 0.0, CPlus(VConv(average)))
		/ nCount;
}

/**
 * W΍擾.
 */
template <typename T>
double GetStandardDeviation(const T* data, ULONG_PTR nCount, T average)
{
	return sqrt(GetVariance(data, nCount, average));
}
#endif

/////////////////////////////////////////////////////////////////////////////

typedef std::unary_function<double, double> ConverterBase;

/**
 * _~[ϊ.
 */
class DirectConverter : public ConverterBase
{
public:
	double operator()(double fSrcData) const
	{
		return fSrcData;
	}
	double Next(double fDstData) const
	{
		return fDstData + 1.0;
	}
};

/**
 * [fSrcMin, fSrcMax] ̐`f[^ [fDstMin, fDstMax] ̐`f[^ɕϊ.
 */
class LinearToLinearConverter : public ConverterBase
{
public:
	LinearToLinearConverter() : m_fRatio(1.0), m_fOffset(0.0)
	{
	}
	LinearToLinearConverter(
		double fSrcMin, double fSrcMax, double fDstMin, double fDstMax)
	{
		Initialize(fSrcMin, fSrcMax, fDstMin, fDstMax);
	}
	/**
	 * conv1 KpĂ conv2 Kp.
	 */
	LinearToLinearConverter(
		const LinearToLinearConverter& conv1, const LinearToLinearConverter& conv2)
	{
		Initialize(conv1, conv2);
	}

	void Initialize(double fSrcMin, double fSrcMax, double fDstMin, double fDstMax)
	{
		m_fRatio = (fDstMax - fDstMin) / (fSrcMax - fSrcMin);
		m_fOffset = fDstMin - m_fRatio * fSrcMin;
	}
	void Initialize(const LinearToLinearConverter& conv1, const LinearToLinearConverter& conv2)
	{
		m_fRatio = conv1.Ratio() * conv2.Ratio();
		m_fOffset = conv1.Offset() * conv2.Ratio() + conv2.Offset();
	}

	double operator()(double fSrcData) const
	{
		return m_fRatio * fSrcData + m_fOffset;
	}
	/**
	 * fDstData ̌ƂȂ̂ fSrcData Ƃꍇ́AfSrcData + 1 ɑ΂l.
	 */
	double Next(double fDstData) const
	{
		return fDstData + m_fRatio;
	}

	double Ratio() const
	{
		return m_fRatio;
	}
	double Offset() const
	{
		return m_fOffset;
	}

protected:
	double m_fRatio;
	double m_fOffset;
};

/**
 * [fLinearMin, fLinearMax] ̐`f[^ [fLogarithmicMin, fLogarithmicMax] ̑ΐf[^ɕϊ.
 */
class LinearToLogarithmicConverter : public ConverterBase
{
public:
	LinearToLogarithmicConverter() : m_fBase(1.0), m_fLnBase(0.0), m_fRatio(1.0)
	{
	}
	/**
	 * `̈ fLinearData ́Aΐ̈
	 * fLogarithmicMin * ((fLogarithmicMax / fLogarithmicMin) ^ ((fLinearData - fLinearMin) / (fLinearMax - fLinearMin)))
	 * ɑΉ.
	 */
	LinearToLogarithmicConverter(
		double fLinearMin, double fLinearMax, double fLogarithmicMin, double fLogarithmicMax)
	{
		Initialize(fLinearMin, fLinearMax, fLogarithmicMin, fLogarithmicMax);
	}
	/**
	 * conv1 KpĂ conv2 Kp.
	 */
	LinearToLogarithmicConverter(
		const LinearToLinearConverter& conv1, const LinearToLogarithmicConverter& conv2)
	{
		Initialize(conv1, conv2);
	}

	void Initialize(double fLinearMin, double fLinearMax, double fLogarithmicMin, double fLogarithmicMax)
	{
#if 0
		m_fBase = Pow(fLogarithmicMax / fLogarithmicMin, 1.0 / (fLinearMax - fLinearMin));
		m_fLnBase = log(m_fBase);
		m_fRatio = fLogarithmicMin / Pow(m_fBase, fLinearMin);
#else
		m_fLnBase = 1.0 / (fLinearMax - fLinearMin) * log(fLogarithmicMax / fLogarithmicMin);
		m_fBase = exp(m_fLnBase);
		m_fRatio = fLogarithmicMin * exp(-fLinearMin * m_fLnBase);
#endif
	}
	void Initialize(const LinearToLinearConverter& conv1, const LinearToLogarithmicConverter& conv2)
	{
#if 0
		m_fBase = Pow(conv2.Base(), conv1.Ratio());
		m_fLnBase = log(m_fBase);
		m_fRatio = conv2.Ratio() * Pow(conv2.Base(), conv1.Offset());
#else
		m_fLnBase = conv1.Ratio() * conv2.LnBase();
		m_fBase = exp(m_fLnBase);
		m_fRatio = conv2.Ratio() * exp(conv1.Offset() * conv2.LnBase());
#endif
	}

	double operator()(double fLinearData) const
	{
#if 0	// "pow"  "|Z + exp" ̕
		return Pow(m_fBase, fLinearData) * m_fRatio;
#else
		return exp(fLinearData * m_fLnBase) * m_fRatio;
#endif
	}
	/**
	 * fLogarithmicData ̌ƂȂ̂ fLinearData Ƃꍇ́AfLinearData + 1 ɑ΂l.
	 */
	double Next(double fLogarithmicData) const
	{
		return fLogarithmicData * m_fBase;
	}

	double Base() const
	{
		return m_fBase;
	}
	double LnBase() const
	{
		return m_fLnBase;
	}
	double Ratio() const
	{
		return m_fRatio;
	}

protected:
	double m_fBase;
	double m_fLnBase;
	double m_fRatio;
};

/**
 * [fLogarithmicMin, fLogarithmicMax] ̑ΐf[^ [fLinearMin, fLinearMax] ̐`f[^ɕϊ.
 */
class LogarithmicToLinearConverter : public ConverterBase
{
public:
	LogarithmicToLinearConverter() : m_fRatio(1.0), m_fOffset(0.0)
	{
	}
	/**
	 * ΐ̈ fLogarithmicData ́A`̈
	 * fLinearMin + (fLinearMax - fLinearMin) * log(fLogarithmicData / fLogarithmicMin) / log(fLogarithmicMax / fLogarithmicMin)
	 * ɑΉ.
	 */
	LogarithmicToLinearConverter(
		double fLogarithmicMin, double fLogarithmicMax, double fLinearMin, double fLinearMax)
	{
		Initialize(fLogarithmicMin, fLogarithmicMax, fLinearMin, fLinearMax);
	}
	/**
	 * conv1 KpĂ conv2 Kp.
	 */
	LogarithmicToLinearConverter(
		const LogarithmicToLinearConverter& conv1, const LinearToLinearConverter& conv2)
	{
		Initialize(conv1, conv2);
	}

	void Initialize(double fLogarithmicMin, double fLogarithmicMax, double fLinearMin, double fLinearMax)
	{
		m_fRatio = (fLinearMax - fLinearMin) / log(fLogarithmicMax / fLogarithmicMin);
		m_fOffset = fLinearMin - m_fRatio * log(fabs(fLogarithmicMin));
	}
	void Initialize(const LogarithmicToLinearConverter& conv1, const LinearToLinearConverter& conv2)
	{
		m_fRatio = conv1.Ratio() * conv2.Ratio();
		m_fOffset = conv1.Offset() * conv2.Ratio() + conv2.Offset();
	}

	double operator()(double fLogarithmicData) const
	{
		return m_fRatio * log(Max(fabs(fLogarithmicData), EPSILON)) + m_fOffset;
	}

	double Ratio() const
	{
		return m_fRatio;
	}
	double Offset() const
	{
		return m_fOffset;
	}

protected:
	double m_fRatio;
	double m_fOffset;
};

/////////////////////////////////////////////////////////////////////////////

/**
 * w肵lȉ̒l̂Ał傫 2 ׂ̂.
 */
inline DWORD_PTR flp2(DWORD_PTR dw)
{
	dw |= dw >> 1;
	dw |= dw >> 2;
	dw |= dw >> 4;
	dw |= dw >> 8;
	dw |= dw >> 16;
#ifdef _WIN64
	dw |= dw >> 32;
#endif	// #ifdef _WIN64
	return dw - (dw >> 1);
}

/**
 * w肵lȏ̒l̂Ał 2 ׂ̂.
 */
inline DWORD_PTR clp2(DWORD_PTR dw)
{
	dw -= 1;
	dw |= dw >> 1;
	dw |= dw >> 2;
	dw |= dw >> 4;
	dw |= dw >> 8;
	dw |= dw >> 16;
#ifdef _WIN64
	dw |= dw >> 32;
#endif	// #ifdef _WIN64
	return dw + 1;
}

#ifdef _WIN64

#if 0
/**
 * oCg].
 */
inline DWORD_PTR reverseBytes(DWORD_PTR dw)
{
	return
		(dw << 56) | ((dw & 0xff00) << 40) | ((dw & 0xff0000) << 24) | ((dw & 0xff000000) << 8) |
		((dw >> 8) & 0xff000000) | ((dw >> 24) & 0xff0000) | ((dw >> 40) & 0xff00) | (dw >> 56);
}

/**
 * rbg].
 */
inline DWORD_PTR reverseBits(DWORD_PTR dw)
{
	dw = (dw & 0x5555555555555555) << 1 | (dw >> 1) & 0x5555555555555555;
	dw = (dw & 0x3333333333333333) << 2 | (dw >> 2) & 0x3333333333333333;
	dw = (dw & 0x0f0f0f0f0f0f0f0f) << 4 | (dw >> 4) & 0x0f0f0f0f0f0f0f0f;
	return reverseBytes(dw);
}
#endif

/**
 * 1 ̃rbg̐.
 */
inline DWORD_PTR population(DWORD_PTR dw)
{
	dw = dw - ((dw >> 1) & 0x5555555555555555);
	dw = (dw & 0x3333333333333333) + ((dw >> 2) & 0x3333333333333333);
	dw = (dw + (dw >> 4)) & 0x0f0f0f0f0f0f0f0f;
	dw = dw + (dw >> 8);
	dw = dw + (dw >> 16);
	dw = dw + (dw >> 32);
	return dw & 0x000000000000007f;
}

#else	// #ifdef _WIN64

#if 0
/**
 * oCg].
 */
inline DWORD_PTR reverseBytes(DWORD_PTR dw)
{
	return (dw << 24) | ((dw & 0xff00) << 8) | ((dw >> 8) & 0xff00) | (dw >> 24);
}

/**
 * rbg].
 */
inline DWORD_PTR reverseBits(DWORD_PTR dw)
{
	dw = (dw & 0x55555555) << 1 | (dw >> 1) & 0x55555555;
	dw = (dw & 0x33333333) << 2 | (dw >> 2) & 0x33333333;
	dw = (dw & 0x0f0f0f0f) << 4 | (dw >> 4) & 0x0f0f0f0f;
	return reverseBytes(dw);
}
#endif

/**
 * 1 ̃rbg̐.
 */
inline DWORD_PTR population(DWORD_PTR dw)
{
	dw = dw - ((dw >> 1) & 0x55555555);
	dw = (dw & 0x33333333) + ((dw >> 2) & 0x33333333);
	dw = (dw + (dw >> 4)) & 0x0f0f0f0f;
	dw = dw + (dw >> 8);
	dw = dw + (dw >> 16);
	return dw & 0x0000003f;
}

#endif	// #ifdef _WIN64

/**
 *  0 ȊÕrbg̐.
 * (number of non-trailing zero bits).
 */
inline DWORD_PTR nntz(DWORD_PTR dw)
{
	return population(dw | 0 - dw);
}

/**
 *  0 ̃rbg̐.
 * (number of trailing zero bits).
 */
inline DWORD_PTR ntz(DWORD_PTR dw)
{
	// HACK: 1(byte) == 8(bit) ߑłɂĂ
	return sizeof(DWORD_PTR) * 8 - nntz(dw);
}

/**
 * 擪 0 ̃rbg̐.
 * (number of leading zero bits).
 */
inline DWORD_PTR nlz(DWORD_PTR dw)
{
	dw = dw | (dw >> 1);
	dw = dw | (dw >> 2);
	dw = dw | (dw >> 4);
	dw = dw | (dw >> 8);
	dw = dw | (dw >> 16);
#ifdef _WIN64
	dw = dw | (dw >> 32);
#endif	// #ifdef _WIN64
	return population(~dw);
}

}	// namespace Math

#endif
