#!/bin/sh
#
# backend of MikuInstaller
#
# Copyright (C) 2008 MikuInstaller Project. All rights reserved.
# http://mikuinstaller.sourceforge.jp/
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
# THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
# PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
# BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
# BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
# WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
# OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
# IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#
# These environment variables must be set:
#   MIKUBUNDLE
#   WINEPREFIX
#   MIKUROOT             -- $HOME/Library/Application Support/MikuInstaller
#   MIKUINSTALLERAPP     -- absolute path to MikuInstaller.app
#   MIKUINSTALLERVERSION -- version of MikuInstaller
#

WINE="$MIKUBUNDLE/Contents/MacOS/wine"
LOGFILE="$MIKUROOT/InstallLog.txt"

loghead () {
  wineversion=`("$WINE" --miku=version) 2>/dev/null` \
  || wineversion='(unknown)'
  mkdir "$MIKUROOT" 2>/dev/null || :
  echo "-------- "`date '+%Y%m%dT%H%M%S'`" BEGIN --------" >> "$LOGFILE"
  echo "**" `uname -a` >> "$LOGFILE"
  echo "** Wine.bundle version $wineversion" >> "$LOGFILE"
  echo "** MIKUBUNDLE=$MIKUBUNDLE" >> "$LOGFILE"
  echo "** WINEPREFIX=$WINEPREFIX" >> "$LOGFILE"
  echo "** MIKUROOT=$MIKUROOT" >> "$LOGFILE"
  echo "** MIKUINSTALLERAPP=$MIKUINSTALLERAPP" >> "$LOGFILE"
  echo "** MIKUINSTALLERVERSION=$MIKUINSTALLERVERSION" >> "$LOGFILE"
}

logtail () {
  echo "-------- "`date '+%Y%m%dT%H%M%S'`" $status END --------" >> "$LOGFILE"
  if [ "$status" != "0" ]; then
    echo "********" >> "$LOGFILE"
    export >> "$LOGFILE"
    echo "********" >> "$LOGFILE"
  fi
}

if [ -n "$BASH_VERSION" ]; then

log () {
  loghead
  echo "** $@" >> "$LOGFILE"
  "$@" 2>&1 | tee -a "$LOGFILE"
  status=${PIPESTATUS[0]}
  logtail
  return $status
}

else

log () {
  loghead
  echo "** $@" >> "$LOGFILE"
  (
    tmp=`mktemp -d -t tmp` || exit $?
    trap "rm -f $tmp/pipe; rmdir $tmp" EXIT
    mkfifo $tmp/pipe
    "$@" > $tmp/pipe 2>&1 &
    pid=$!
    cat $tmp/pipe | tee -a "$DSTFILE"
    wait $pid
    status=$?
    logtail
    exit $status
  )
}

fi

cmd="$1"
shift
case "$cmd" in

  install)
    install_main () {
      WINEBUNDLE="$1"
      set -x
      test -n "$WINEBUNDLE" || return 1
      test -d "$WINEBUNDLE" || return 1
      mkdir "$MIKUBUNDLE"
      (cd "$WINEBUNDLE" && pax -rw -v -H -pap Contents "$MIKUBUNDLE") \
        || return $?
      mkdir "$MIKUROOT/prefix" || :
    }
    log install_main "$@"
    ;;

  open)
    log "$WINE" --miku=debug --miku=shellexec -c "$@"
    ;;

  winecfg)
    log "$WINE" --miku=debug --miku=winecmd winecfg
    ;;

  addapp)
    log "$WINE" --miku=debug --miku=addapp -d"$1"
    ;;

  prefixcreate)
    mkdir -p "$WINEPREFIX" || :
    log "$WINE" --miku=debug --miku=prefixcreate
    ;;

  prefixupdate)
    prefixupdate_main () {
      APPDIR="$1"
      set -x
      for i in "$MIKUROOT/prefix"/*; do
        [ -d "$i" ] || continue
        WINEPREFIX="$i" "$WINE" --miku=debug --miku=prefixcreate
	WINEPREFIX="$i" "$WINE" --miku=debug --miku=addapp -d"$APPDIR"
      done
    }
    log prefixupdate_main "$@"
    ;;

  prefixremove)
    log rm -r "$WINEPREFIX"
    ;;

  uninstall)
    uninstall_main () {
      set -x
      rm -r "$MIKUBUNDLE"
    }
    log uninstall_main
    ;;

  migrate)
    migrate_main () {
      WINEKITROOT="$1"
      set -x
      mkdir -p "$MIKUROOT/prefix" || :
      if [ -d "$WINEKITROOT/prefix" ]; then
        files=`cd "$WINEKITROOT/prefix" && echo *`
        if [ "$?" = "0" ] && [ "x$files" != "x*" ]; then
          mv -v "$WINEKITROOT/prefix"/* "$MIKUROOT/prefix"
        fi
      fi
      rm -r "$WINEKITROOT"
    }
    log migrate_main "$@"
    ;;

  *)
    exit 128
    ;;
esac
