/*
 * Controller.m
 *
 * Copyright (C) 2008 MikuInstaller Project. All rights reserved.
 * http://mikuinstaller.sourceforge.jp/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#import "Controller.h"

#define SCRIPT_LAUNCH  "launch"
#define SCRIPT_ACTIVE  "active"
#define SCRIPT_OPEN    "open"

#define MIKUBUNDLE     "MIKUBUNDLE"
#define APP_BUNDLE_ID  "jp.sourceforge.mikuinstaller.MikuInstaller"

#ifdef DEBUG
#define TRACE(args) NSLog args
#else
#define TRACE(args)
#endif

@implementation Controller

#ifdef MIKUBUNDLE
static NSString *
absolutePathForWineBundle()
{
	NSArray *appSupportPath;
	NSString *appPath, *winePath;
	NSBundle *bundle;
	NSFileManager *fm = [NSFileManager defaultManager];

	/*
	 * Search Wine.bundle in the following order:
	 * (1) ~/Library/Application Support/MikuInstaller
	 * (2) Resources of MikuInstaller.app
	 */
	appSupportPath = NSSearchPathForDirectoriesInDomains(
				NSApplicationSupportDirectory,
				NSUserDomainMask,
				YES);
	winePath = [appSupportPath objectAtIndex:0];
	winePath = [winePath stringByAppendingString:
				     @"/MikuInstaller/Wine.bundle"];
	if ([fm fileExistsAtPath:
			[winePath stringByAppendingString:
					  @"/Contents/Info.plist"]]) {
		TRACE((@"found : %@\n", winePath));
		return winePath;
	}

	appPath = [[NSWorkspace sharedWorkspace]
			  absolutePathForAppBundleWithIdentifier:
				  @APP_BUNDLE_ID];
	TRACE((@"looking for in %@\n", appPath));
	bundle = appPath
		? [NSBundle bundleWithPath:appPath]
		: nil;
	winePath = bundle
		? [bundle pathForResource:@"Wine" ofType:@"bundle"]
		: nil;

	TRACE((@"found : %@\n", winePath));
	return winePath;
}
#endif /* MIKUBUNDLE */

static id
getScript(NSBundle *bundle, NSString *name)
{
	NSAppleScript *script = nil;
	NSString *filename;
	NSURL *url;
	NSDictionary *error;

	filename = [bundle pathForResource:name ofType:@"scpt"];
	if (!filename)
		filename = [bundle pathForResource:name ofType:@"applescript"];
	if (filename) {
		url = [NSURL fileURLWithPath:filename];
		script = [[[NSAppleScript alloc] autorelease]
				 initWithContentsOfURL:url
				 error:&error];
		if (script && ![script isCompiled]) {
			if (![script compileAndReturnError:&error])
				script = nil;
		}
	}

	if (script)
		return script;
	else
		return [bundle pathForResource:name ofType:nil];
}

- (id)init
{
	static NSString * const actions[] = {
		@SCRIPT_LAUNCH,
		@SCRIPT_ACTIVE,
		@SCRIPT_OPEN
	};
	NSBundle *mainBundle;
	NSMutableDictionary *dic;
	NSString *filename;
	id script;
	int i;

	[super init];

	launched = NO;
	currentTasks = [[NSMutableArray array] retain];

	mainBundle = [NSBundle mainBundle];

	filename = [mainBundle pathForResource:@"env" ofType:@"plist"];
	environmentVariables =
		filename
		? [NSDictionary dictionaryWithContentsOfFile:filename]
		: [NSDictionary dictionary];
	[environmentVariables retain];

#ifdef MIKUBUNDLE
	if (![environmentVariables valueForKey:@MIKUBUNDLE]) {
		dic = [NSMutableDictionary dictionaryWithCapacity:10];
		[dic addEntriesFromDictionary:environmentVariables];
		filename = absolutePathForWineBundle();
		if (filename) {
			NSLog(@"Wine.bundle: %@", filename);
			[dic setObject:filename forKey:@MIKUBUNDLE];
		}
		[environmentVariables release];
		environmentVariables =
			[[NSDictionary dictionaryWithDictionary:dic] retain];
	}
#endif /* MIKUBUNDLE */

	dic = [NSMutableDictionary dictionaryWithCapacity:10];
	for (i = 0; i < sizeof(actions)/sizeof(actions[0]); i++) {
		script = getScript(mainBundle, actions[i]);
		if (script)
			[dic setObject:script forKey:actions[i]];
	}
	scriptForAction = [[NSDictionary dictionaryWithDictionary:dic] retain];

	return self;
}

- (void)terminateAll
{
	NSEnumerator *e = [currentTasks objectEnumerator];
	NSTask *task;

	while ((task = [e nextObject])) {
		TRACE((@"terminate task %p", task));
		[task terminate];
	}
}

- (void)dealloc
{
	[self terminateAll];
	[currentTasks release];
	[environmentVariables release];
	[scriptForAction release];
	[[NSNotificationCenter defaultCenter] removeObserver:self];

	[super dealloc];
}

- (void)startTask:(NSString *)command arguments:(NSArray *)args
{
	NSMutableDictionary *env;

	NSTask *task = [[[NSTask alloc] init] autorelease];
	[task setStandardInput:[NSFileHandle fileHandleWithNullDevice]];
	[task setLaunchPath:command];
	if (args)
		[task setArguments:args];

	env = [NSMutableDictionary dictionaryWithCapacity:10];
	[env setDictionary:[[NSProcessInfo processInfo] environment]];
	[env addEntriesFromDictionary:environmentVariables];
	[task setEnvironment:env];

	[currentTasks addObject:task];

	[[NSNotificationCenter defaultCenter]
		addObserver:self
		selector:@selector(taskTerminated:)
		name:NSTaskDidTerminateNotification
		object:task];

	TRACE((@"task %p: executing command %@", task, command));
	[task launch];
}

- (void)taskTerminated:(NSNotification *)notification
{
	NSTask *task = [notification object];

	NSLog(@"task %p terminated at status %d",
	      task, [task terminationStatus]);

	[[NSNotificationCenter defaultCenter]
		removeObserver:self
		name:NSTaskDidTerminateNotification
		object:task];

	[currentTasks removeObjectIdenticalTo:task];
	TRACE((@"%d tasks remain", [currentTasks count]));

	if ([currentTasks count] == 0)
		[NSApp terminate:self];
}

- (void)invoke:(NSString *)actionName arguments:(NSArray *)args
{
	id script;
	NSDictionary *error;

	TRACE((@"action: %@", actionName));

	script = [scriptForAction valueForKey:actionName];
	if (script && [script isKindOfClass:[NSAppleScript class]]) {
		TRACE((@"execute applescript %@", actionName));
		if (![(NSAppleScript*)script executeAndReturnError:&error])
			NSLog(@"applescript error");
		TRACE((@"execute applescript %@ finished", actionName));
	}
	else if (script) {
		[self startTask:script arguments:args];
	}
}

- (void)application:(NSApplication *)sender openFiles:(NSArray *)filenames
{
	NSEnumerator *e = [filenames objectEnumerator];
	NSString *filename;

	TRACE((@"openFiles: launched? %d\n", launched));

	while ((filename = [e nextObject])) {
		[self invoke:@SCRIPT_OPEN
		      arguments:[NSArray arrayWithObject:filename]];
	}
	launched = YES;
}

- (void)applicationDidFinishLaunching:(NSNotification *)notification
{
	TRACE((@"DidFinishLaunching: launched? %d\n", launched));

	if (!launched)
		[self invoke:@SCRIPT_LAUNCH arguments:nil];

	if ([currentTasks count] == 0)
		[NSApp terminate:self];
}

- (void)applicationDidBecomeActive:(NSNotification *)notification
{
	[self invoke:@SCRIPT_ACTIVE arguments:nil];
}

- (void)applicationWillTerminate:(NSNotification *)notification
{
	[self terminateAll];
}

@end
