/*******************************************************************************
 * booter/protected_mode/ata.c
 *                                                                   2012/10/06
 * Copyright (C) 2012 Mochi
 ******************************************************************************/
#include <ia-32/instructions.h>

/* レジスタ */
#define REGISTER_DATA		0x01F0		/* Data register */
#define REGISTER_SECCNT		0x01F2		/* Sector counter register */
#define REGISTER_SECNUM		0x01F3		/* Sector number register */
#define REGISTER_CYLLOW		0x01F4		/* Cylinder low register */
#define REGISTER_CYLHIGH	0x01F5		/* Cylinder high register */
#define REGISTER_DEVHEAD	0x01F6		/* Device/head register */
#define REGISTER_STATUS		0x01F7		/* Status register */
#define REGISTER_COMMAND	0x01F7		/* Command register */

/* ステータスレジスタ */
#define STATUS_BSY	0x80		/* Busy bit */
#define STATUS_DRQ	0x08		/* Data request bit */

/* コマンド */
#define COMMAND_READ	0x20	/* 読み込みコマンド */

/* 割込みフラグ */
volatile int ata_intrq;

/*
 * HDDからデータをメモリに読み込む
 * 戻り値：
 *     成功（0）
 *     失敗（-1）
 * 引数：
 *     lba			：読み込み元LBA
 *     buffer		：読込み先先頭アドレス
 *     sector_size	：読込みセクタ数
 */
int read_disk(unsigned int lba, void *buffer, unsigned int sector_size) {
	int i;
	int j;
	int k;
	unsigned int temp_lba;		/* LBA */
	unsigned int temp_size;		/* サイズ */
	unsigned char status;		/* ステータスレジスタ */
	
	/* 256セクタ毎 */
	for (i = 0; i < sector_size; i += 256) {
		/* アドレス計算 */
		temp_lba = lba + i;
		
		/* サイズ計算 */
		temp_size = sector_size - i;
		if (temp_size > 256) {
			temp_size = 256;
		}
		
		/* BSY と DRQ が 0 になるまで待ち */
		while ((in_byte(REGISTER_STATUS) & (STATUS_BSY | STATUS_DRQ)) != 0);
		
		/* デバイス設定及びLBA high設定 */
		out_byte(REGISTER_DEVHEAD, 0x40 | ((temp_lba >> 24) & 0x0F));
		
		/* BSY と DRQ が 0 になるまで待ち */
		while ((in_byte(REGISTER_STATUS) & (STATUS_BSY | STATUS_DRQ)) != 0);
		
		/* パラメータ設定 */
		out_byte(REGISTER_SECCNT, (temp_size == 256) ? 0 : (temp_size & 0xFF));
		out_byte(REGISTER_SECNUM, temp_lba & 0xFF);
		out_byte(REGISTER_CYLLOW, (temp_lba >> 8) & 0xFF);
		out_byte(REGISTER_CYLHIGH, (temp_lba >> 16) & 0xFF);
		
		/* 割込みフラグ初期化 */
		ata_intrq = 0;
		
		/* 読込みコマンド実行 */
		out_byte(REGISTER_COMMAND, COMMAND_READ);
		
		/* 1セクタ毎 */
		for (j = 0; j < temp_size; j++) {
			/* 割込み待ち */
			while (ata_intrq == 0) {
				hlt();
			}
			
			/* 割込みフラグ初期化 */
			ata_intrq = 0;
			
			/* BSY が 0 になるまで待ち */
			while (((status = in_byte(REGISTER_STATUS)) & STATUS_BSY) != 0);
			
			/* DRQ が 1 */
			if ((status & STATUS_DRQ) != 0) {
				/* 1セクタ読み込み */
				for (k = 0; k < 256; k++) {
					((unsigned short *)buffer)[i * 256 * 256 + j * 256 + k] =
						in_word(REGISTER_DATA);
				}
				
			/* DRQ が 0 */
			} else {
				/* エラー */
				return -1;
				
			}
		}
	}
	
	return 0;
}
