

// -------------------------------------------------------------------------------------------------------------
// Vertex shader implementations with per vertex fog
// -------------------------------------------------------------------------------------------------------------

struct AdvancedVS
{
    float4 posH     : POSITION0;
    float3 CamW     : TEXCOORD0;     
    half3  nrmW     : TEXCOORD1;
    half4  diffuse  : COLOR0;           // (Local Light) Diffuse color
    half4  spec     : COLOR1;           // (Local Light) Specular color
    half4  atten    : TEXCOORD2;        // (Atmospheric haze) Attennuate incoming fragment color
    half4  insca    : TEXCOORD3;        // (Atmospheric haze) "Inscatter" Add to incoming fragment color
    half2  tex0     : TEXCOORD4;
};

struct TileMeshVS
{
    float4 posH     : POSITION0;
    float3 CamW     : TEXCOORD0;
    half2  tex0     : TEXCOORD1;
    half3  nrmW     : TEXCOORD2;
    half4  atten    : COLOR0;           // (Atmospheric haze) Attennuate incoming fragment color
    half4  insca    : COLOR1;           // (Atmospheric haze) "Inscatter" Add to incoming fragment color
};

struct MeshVS
{
    float4 posH     : POSITION0;
    float3 CamW     : TEXCOORD0;
    half2  tex0     : TEXCOORD1;
    half3  nrmW     : TEXCOORD2;
};


#include "NormalMap.fx"


AdvancedVS MeshTechVS(MESH_VERTEX vrt)
{
    AdvancedVS outVS = (AdvancedVS)0;

    float3 posX = mul(float4(vrt.posL, 1.0f), gGrpT).xyz;       // Apply meshgroup specific transformation
    float3 posW = mul(float4(posX, 1.0f), gW).xyz;              // Apply world transformation matrix
    float3 nrmX = mul(float4(vrt.nrmL.xyz, 0.0f), gGrpT).xyz;
    float3 nrmW = mul(float4(nrmX, 0.0f), gW).xyz;

	// A vector from the vertex to the camera
	outVS.CamW  = -posW;
	outVS.tex0  = vrt.tex0;
    outVS.nrmW  = nrmW;
	outVS.posH  = mul(float4(posW, 1.0f), gVP);

    half4 locW;
    LocalVertexLight(outVS.diffuse, outVS.spec, locW, nrmW, posW);
   
    // Atmospheric haze --------------------------------------------------------

    AtmosphericHaze(outVS.atten, outVS.insca, outVS.posH.z, posW);

    outVS.insca *= (gSun.diffuse+gSun.ambient);

    return outVS;
}



float4 MeshTechPS(AdvancedVS frg) : COLOR
{
	// Normalize input
	float3 CamW = normalize(frg.CamW);
    float3 nrmW = normalize(frg.nrmW);
    half4 cTex = 1;

    if (gTextured) {
        if (gClamp) cTex = tex2D(ClampS, frg.tex0); 
        else        cTex = tex2D(WrapS, frg.tex0);
    }

    if (gModAlpha || !gTextured) cTex.a *= gMat.diffuse.a;	
    if (gFullyLit) return float4(cTex.rgb*gMat.diffuse.rgb, cTex.a);

    float3 r = reflect(gSun.direction, nrmW);
    float  d = max(0,dot(-gSun.direction, nrmW));
	float  s = pow(max(dot(r, CamW), 0.0f), gMat.specPower); 

    if (gMat.specPower<1.0 || d<=0) s = 0;

    half3 diff = gMat.diffuse.rgb  * (frg.diffuse.rgb + d * gSun.diffuse.rgb) + (gMat.ambient.rgb*gSun.ambient.rgb) + (gMat.emissive.rgb);
    half3 spec = gMat.specular.rgb * (frg.spec.rgb    + s * gSun.specular.rgb);

   
    // -------------------------------------------------------------------------
	half3 color  = cTex.rgb * saturate(diff) + saturate(spec);
    //float3 color  = dayTex.rgb * diff + spec;
    //float3 color = 1.0f - exp(-1.0f*(dayTex.rgb*diff+spec));
    // -------------------------------------------------------------------------

    return float4(color.rgb*frg.atten.rgb+frg.insca.rgb, cTex.a);
}


float4 MeshTechGlassPS(AdvancedVS frg) : COLOR
{
	// Normalize input
	float3 nrmW = normalize(frg.nrmW);
	float3 CamW = normalize(frg.CamW);
   
    if (gFullyLit) return gMat.diffuse;

    float3 r = reflect(gSun.direction, nrmW);
    float  d = max(0,dot(-gSun.direction, nrmW));
	float  s = pow(max(dot(r, CamW), 0.0f), gMat.specPower); 

    if (gMat.specPower<1.0 || d<=0) s = 0;

    half3 diff = gMat.diffuse.rgb  * (frg.diffuse.rgb + d * gSun.diffuse.rgb) + (gMat.ambient.rgb*gSun.ambient.rgb) + (gMat.emissive.rgb);
    half3 spec = gMat.specular.rgb * (frg.spec.rgb    + s * gSun.specular.rgb);

    // -------------------------------------------------------------------------
	half3 color  = saturate(diff) + saturate(spec);
    //float3 color  = dayTex.rgb * diff + spec;
    //float3 color = 1.0f - exp(-1.0f*(dayTex.rgb*diff+spec));
    // -------------------------------------------------------------------------

    return float4(color.rgb*frg.atten.rgb+frg.insca.rgb, gMat.diffuse.a);
}


MeshVS SimpleMeshTechVS(MESH_VERTEX vrt)
{
    // Zero output.
	MeshVS outVS = (MeshVS)0;
	
	float3 posX = mul(float4(vrt.posL, 1.0f), gGrpT).xyz;       // Apply meshgroup specific transformation
    float3 posW = mul(float4(posX, 1.0f), gW).xyz;              // Apply world transformation matrix
    outVS.posH  = mul(float4(posW, 1.0f), gVP);
   
    float3 nrmX = mul(float4(vrt.nrmL.xyz, 0.0f), gGrpT).xyz;       // Apply meshgroup specific transformation
    float3 nrmW = mul(float4(posX, 0.0f), gW).xyz;              // Apply world transformation matrix
	
    outVS.nrmW  = nrmW;
    outVS.CamW  = -posW;
    outVS.tex0  = vrt.tex0;

    return outVS;
}


MeshVS TinyMeshTechVS(MESH_VERTEX vrt)
{
    // Zero output.
	MeshVS outVS = (MeshVS)0;
	
    float3 posW = mul(float4(vrt.posL, 1.0f), gW).xyz;              // Apply world transformation matrix
    outVS.posH  = mul(float4(posW, 1.0f), gVP);
    float3 nrmW = mul(float4(vrt.nrmL.xyz, 0.0f), gW).xyz;          // Apply world transformation matrix
    outVS.nrmW  = nrmW;
    outVS.CamW  = -posW;
    outVS.tex0  = vrt.tex0;

    return outVS;
}



float4 VCTechPS(MeshVS frg) : COLOR
{
    // Normalize input
	float3 nrmW = normalize(frg.nrmW);
	float3 CamW = normalize(frg.CamW);
    half4 cTex = 1;

    if (gTextured) {
        if (gClamp) cTex = tex2D(ClampS, frg.tex0); 
        else        cTex = tex2D(WrapS, frg.tex0);
    }

    if (gModAlpha || !gTextured) cTex.a *= gMat.diffuse.a;	
    if (gFullyLit) return float4(cTex.rgb*gMat.diffuse.rgb, cTex.a);

    float3 r = reflect(gSun.direction, nrmW);
    float  d = max(0,dot(-gSun.direction, nrmW));
	float  s = pow(max(dot(r, CamW), 0.0f), gMat.specPower); 

    if (gMat.specPower<1.0 || d<=0) s = 0;

    half3 diff = gMat.diffuse.rgb  * (d * gSun.diffuse.rgb) + (gMat.ambient.rgb*gSun.ambient.rgb) + (gMat.emissive.rgb);
    half3 spec = gMat.specular.rgb * (s * gSun.specular.rgb);

    half3 colr = cTex.rgb * saturate(diff) + saturate(spec);
   
    return float4(colr.rgb, cTex.a);
}


float4 HUDTechPS(MeshVS frg) : COLOR
{
    return tex2D(SimpleS, frg.tex0);
}


float4 MFDTechPS(MeshVS frg) : COLOR
{
	// Normalize input
    
	float3 nrmW = normalize(frg.nrmW);
	float3 CamW = normalize(frg.CamW);
	
	float3 r = reflect(gSun.direction, nrmW);
	float  s = 0.0f;

    if (gMat.specPower>0) s = pow(max(dot(r, CamW), 0.0f), gMat.specPower);

    float4 spec  = s * (gMat.specular*gSun.specular); 
    float4 color = tex2D(MFDSamp, frg.tex0) + min(spec,1.0f);
	
    return float4(color.rgb, 1.0f);
}



// =============================================================================
// Planet Rings Technique
// =============================================================================

float4 RingTechPS(MeshVS frg) : COLOR
{
    float4 color = tex2D(RingS, frg.tex0);
    if (dot(normalize(frg.nrmW),normalize(frg.CamW))>0) return float4(color.rgb*0.35f, color.a);
    return float4(color.rgb, color.a);
}


// =============================================================================
// Base Tile Rendering Technique
// =============================================================================

TileMeshVS BaseTileVS(NTVERTEX vrt)
{
    // Null the output
	TileMeshVS outVS = (TileMeshVS)0;

	float3 posW  = mul(float4(vrt.posL, 1.0f), gW).xyz;
	outVS.posH   = mul(float4(posW, 1.0f), gVP);
    outVS.nrmW   = mul(float4(vrt.nrmL, 0.0f), gW).xyz;
	outVS.tex0   = vrt.tex0;
    outVS.CamW   = -posW;

    // Atmospheric haze --------------------------------------------------------

    AtmosphericHaze(outVS.atten, outVS.insca, outVS.posH.z, posW);

    half4 diffuse;
    float ambi, nigh;

    LegacySunColor(diffuse, ambi, nigh, outVS.nrmW);

    outVS.insca *= (diffuse+ambi);

    return outVS;
}


float4 BaseTilePS(TileMeshVS frg) : COLOR
{
    // Normalize input
	float3 nrmW = normalize(frg.nrmW);
    float3 CamW = normalize(frg.CamW);
	
	half4 cTex = tex2D(ClampS, frg.tex0);
	
	float3 r = reflect(gSun.direction, nrmW);
	float  s = pow(max(dot(r, CamW), 0.0f), 20.0f) * (1.0f-cTex.a);
	float  d = saturate(dot(-gSun.direction, nrmW)*1.5);
  
    if (d<=0) s = 0;
       
    half3 clr = cTex.rgb * (max(d,0) * gSun.diffuse.rgb + s * gSun.specular.rgb + gSun.ambient.rgb);
    
    return float4(clr.rgb*frg.atten.rgb+frg.insca.rgb, cTex.a);
}




// =============================================================================
// Mesh Shadow Technique
// =============================================================================

HazeVS ShadowMeshTechVS(MESH_VERTEX vrt)
{
    // Zero output.
	HazeVS outVS = (HazeVS)0;

	float3 posX = mul(float4(vrt.posL, 1.0f), gGrpT).xyz;       // Apply meshgroup specific transformation
    float3 posW = mul(float4(posX, 1.0f), gW).xyz;              // Apply world transformation matrix
    outVS.posH  = mul(float4(posW, 1.0f), gVP);

    return outVS;
}


float4 ShadowTechPS(HazeVS frg) : COLOR
{
    return float4(0.0f, 0.0f, 0.0f, gMix);
}


technique ShadowTech
{
    pass P0
    {
        vertexShader = compile VS_MOD ShadowMeshTechVS();
        pixelShader  = compile PS_MOD ShadowTechPS();

        AlphaBlendEnable = true;
        BlendOp = Add;
        ZEnable = false; 
        SrcBlend = SrcAlpha;
        DestBlend = InvSrcAlpha;    
        ZWriteEnable = false;  
        
		StencilEnable = true;
		StencilRef    = 1;
		StencilMask   = 1;
		StencilFunc   = NotEqual;
		StencilPass   = Replace;
    }
}





// This is the default mesh rendering technique --------------------------------
//

technique VesselTech
{
    pass P0
    {
        vertexShader = compile VS_MOD MeshTechNMVS();
        pixelShader  = compile PS_MOD MeshTechNMPS();

        AlphaBlendEnable = true;
        BlendOp = Add;
        ZEnable = true; 
        SrcBlend = SrcAlpha;
        DestBlend = InvSrcAlpha;    
        ZWriteEnable = true;
    }

    pass P1
    {
        vertexShader = compile VS_MOD MeshTechVS();
        pixelShader  = compile PS_MOD MeshTechPS();

        AlphaBlendEnable = true;
        BlendOp = Add;
        ZEnable = true; 
        SrcBlend = SrcAlpha;
        DestBlend = InvSrcAlpha;    
        ZWriteEnable = true;
    }

    pass P2
    {
        vertexShader = compile VS_MOD MeshTechVS();
        pixelShader  = compile PS_MOD MeshTechGlassPS();

        AlphaBlendEnable = true;
        BlendOp = Add;
        ZEnable = true; 
        SrcBlend = SrcAlpha;
        DestBlend = InvSrcAlpha;    
        ZWriteEnable = true;
    }
}



technique BaseTileTech
{
    pass P0
    {
        vertexShader = compile VS_MOD BaseTileNMVS();
        pixelShader  = compile PS_MOD BaseTileNMPS();
        
        AlphaBlendEnable = true;
        BlendOp = Add;
        SrcBlend = SrcAlpha;
        DestBlend = InvSrcAlpha;
        ZEnable = false;
        ZWriteEnable = false;
    }   

    pass P1
    {
        vertexShader = compile VS_MOD BaseTileVS();
        pixelShader  = compile PS_MOD BaseTilePS();
        
        AlphaBlendEnable = true;
        BlendOp = Add;
        SrcBlend = SrcAlpha;
        DestBlend = InvSrcAlpha;
        ZEnable = false;
        ZWriteEnable = false;
    }   
}


technique VCTech
{
    pass P0
    {
        vertexShader = compile VS_MOD SimpleMeshTechVS();
        pixelShader  = compile PS_MOD VCTechPS();

        AlphaBlendEnable = true;
        BlendOp = Add;
        SrcBlend = SrcAlpha;
        DestBlend = InvSrcAlpha;
        ZEnable = true;
        ZWriteEnable = true;
    }
}


technique VCMFDTech
{
    pass P0
    {
        vertexShader = compile VS_MOD SimpleMeshTechVS();
        pixelShader  = compile PS_MOD MFDTechPS();

        AlphaBlendEnable = false;
        ZEnable = true;
        ZWriteEnable = true;
    }
}


technique VCHudTech
{
    pass P0
    {
        vertexShader = compile VS_MOD SimpleMeshTechVS();
        pixelShader  = compile PS_MOD HUDTechPS();

        AlphaBlendEnable = true;
        BlendOp = Add;
        SrcBlend = One;
        DestBlend = One;
        ZEnable = false;
    }
}


// This is used for rendering beacons ------------------------------------------
//
technique RingTech
{
    pass P0
    {
        vertexShader = compile VS_MOD TinyMeshTechVS();
        pixelShader  = compile PS_MOD RingTechPS();
        
        AlphaBlendEnable = true;
        BlendOp = Add;
        SrcBlend = SrcAlpha;
        DestBlend = InvSrcAlpha;
        ZWriteEnable = true;
        ZEnable = false;
        CullMode = NONE;
    }   
}