struct MESH_VERTEX {
	float3 PosL		: POSITION;
	float3 NormL	: NORMAL;
	float3 TangL	: TANGENT;
	float2 TCrd		: TEXCOORD;
};

struct VSOut {
	float4 PosH		: SV_POSITION;
	float3 NormW	: NORMAL;
	float3 CamW		: TEXCOORD0;
	float2 TCrd		: TEXCOORD1;
};

struct Sun_spec {
	float4 SunDir;
	float4 ambient;
	float4 diffuse;
	float4 specular;	//64
};

cbuffer cb_VS_per_frame		: register( b0 )
{
	row_major matrix VP;
};

cbuffer cb_VS_VC_per_group	: register( b1 )
{
	row_major matrix W;
};

VSOut VS_VC( MESH_VERTEX In ) {
	VSOut Out;
	float3 PosW = mul( float4( In.PosL, 1.0f ), W ).xyz;
	Out.PosH = mul( float4( PosW, 1.0f ), VP );

	Out.NormW = mul( float4( In.NormL, 0.0f ), W ).xyz;

	Out.CamW = -PosW;

	Out.TCrd = In.TCrd;
	return Out;
}

Texture2D MainTex;
SamplerState MainSampler;

cbuffer cb_PS_per_object	: register( b0 )
{
	Sun_spec Sun;			//64
};

cbuffer cb_PS_Mesh			: register( b1 )
{
	float3 ambient;
	float opacity;		//16

	float3 diffuse;
	uint flags;		//32

	float3 specular;
	float spec_power;	//48

	float3 emissive;
	uint fully_lit;		//64
};

//======================================
//		VC meshes
//======================================

float4 PS_VC( VSOut In )	: SV_Target
{
	In.NormW = normalize( In.NormW );
	In.CamW = normalize( In.CamW );

	float4 Tex = float4( 1.0f, 1.0f, 1.0f, opacity );
	[branch]
	if( flags & 1 )
		Tex = MainTex.Sample( MainSampler, In.TCrd );

	[branch]
	if( flags & 2 )
		Tex.a *= opacity;

	[branch]
	if( flags & 64 )
		return float4( Tex.rgb*saturate( diffuse + emissive ), opacity );

	float d = max( 0.0f, dot( -Sun.SunDir.xyz, In.NormW ) );
	float s = 0.0f;

	[branch]
	if( spec_power > 2.0f && d > 0.005f )
		s = pow( max( dot( reflect( Sun.SunDir.xyz, In.NormW ), In.CamW ), 0.0f ), spec_power );

	half3 diff = diffuse*( d*Sun.diffuse.rgb ) + ambient*Sun.ambient.rgb + emissive;
	half3 spec = specular*( s*Sun.specular.rgb );
	half3 col = Tex.rgb*saturate( diff ) + saturate( spec );

	return float4( col, Tex.a );
}

//======================================
//		VC MFDs
//======================================

float4 PS_MFD( VSOut In )	: SV_Target
{
	In.NormW = normalize( In.NormW );
	In.CamW = normalize( In.CamW );

	float s = 0.0f;
	[branch]
	if( spec_power > 2.0f )
		s = pow( max( dot( reflect( Sun.SunDir.xyz, In.NormW ), In.CamW ), 0.0f ), 40.0f );

	float4 col = float4( 0.0f, 0.0f, 0.0f, 1.0f );
	[branch]
	if( flags & 1 )
		col = MainTex.Sample( MainSampler, In.TCrd );
	float4 color = col + saturate( s*float4( 0.5f, 0.5f, 0.5f, 0.0f ) );

	return float4( col.rgb, 1.0f );
}

//======================================
//		VC HUD
//======================================

float4 PS_HUD( VSOut In )	: SV_Target
{
	[branch]
	if( flags & 1 )	clip( -1 );
	float4 color = MainTex.Sample( MainSampler, In.TCrd );
	float sat = max( color.r, max( color.g, color.b ) );
	clip( sat - 0.3f );
	color.rgb = saturate( color.rgb*5.0f );
	color.a = sat*5.0f;
	return color;
}