
/* --------------------------------------------- */
/*  H8-3069F User interface                      */
/*                                               */
/*  CPU    : Renesus H8/3069F 25MHz              */
/*  Memory : ROM 512KB, RAM 16KB E-RAM 2MB       */
/*                (c) KAZ.Imamura                */
/* --------------------------------------------- */

#include "ui.h"
#include "key.h"
#include "sl811.h"
#include "usb_ms.h"
#include "fat.h"
#include "serial.h"
#include "srcver.h"

extern unsigned char eram_start;
extern unsigned char eram_end;

#define MEMCHK_BLOCK_SIZE 0x1000

// -------------------------------------------
//  Proto type definitions
// -------------------------------------------
void ui_1ms_handler(void);
int ui_initialize(void);
int ui_process(void);
int ui_cmdset(UI_COMMAND cmd);

// Locals
static int ui_function_nop(UI_COMMAND uicmd);
static int ui_function_memchk(UI_COMMAND uicmd);
static int ui_function_memview(UI_COMMAND uicmd);
static int ui_function_enduser_mode(UI_COMMAND uicmd);

// -------------------------------------------
//  Variables
// -------------------------------------------

const UI_APP_TABLE ui_app_table[] = {
	{	 0,		ui_function_enduser_mode,	"  END USER MODE ", 200		},
	{	 1,		ui_function_nop,			"  RX START      ", 0		},
	{	 2,		ui_function_nop,			"  TX START      ", 1000	},
	{	 3,		ui_function_memchk,			"  MEMORY CHECK  ", 20		},
	{	 4,		ui_function_nop,			"  CONFIG        ", 0		},
	{	 5,		ui_function_usb,			"  USB INFO.     ", 200		},
	{	 6,		ui_function_usbms_debug,	"  MASS-STORAGE  ", 500		},
	{	 7,		ui_function_fat,			"  FAT(Debug)    ", 300		},
	{	 8,		ui_function_serial,			"  Serial(Debug) ", 300		},
	{	 9,		ui_function_memview,		"  MEMVIEW(Debug)", 300		},
	{	-1,		ui_function_nop,			"0123456789ABCDEF", 0		},
};

unsigned int *memchk_start;
unsigned int  memchk_ptn;
unsigned int  memchk_rslt;

// Locals
unsigned short ui_timer;
unsigned short ui_wait_timer;
unsigned int   ui_event_timer;
unsigned short ui_proc;

// ui function related
unsigned int current_index;
unsigned int cursol_position;

// Command queue releated
UI_COMMAND ui_cmd_queue[MAX_CMD_QUEUE+1];
unsigned int ui_queue_rp;
unsigned int ui_queue_wp;

// proc related
enum key_process_mode {
	UI_00_INIT,
	UI_01_STARTUP,
	UI_02_TOP_LEVEL,
	UI_03_EXECUTING,
};

// -------------------------------------------
//  Interrupt handlers
// -------------------------------------------
void ui_1ms_handler(void) {
	if( ui_wait_timer )    ui_wait_timer--;
	
	// UI interval events
	// TODO: need to consider about conflict of current_index.
	if( ui_event_timer ) {
		if( !(--ui_event_timer) ) {
			// set event
			UI_COMMAND ui_cmd;
			ui_cmd.cmd = UI_CMD_INTEVAL;
			ui_cmdset( ui_cmd );					// TODO: error handling
			
			// timer reset
			ui_event_timer = ui_app_table[current_index].interval_time;
		}
	}
	
	// 4ms interval
	if( !( ++ui_timer  & 0x03 ) ) {
		
	}
}


// -------------------------------------------
//  Initialize
// -------------------------------------------
int ui_initialize(void) {
	ui_wait_timer = 0;
	ui_proc = UI_00_INIT;
	ui_event_timer = 0;
	
	ui_queue_rp = ui_queue_wp = 0;
	current_index = 0;
	cursol_position = 0;
}


// -------------------------------------------
//  Main process
// -------------------------------------------
int ui_process(void) {
	if(ui_wait_timer) return 0;
	
	switch( ui_proc ) {
	case UI_00_INIT:
		sc1602_set_buffer(0, " PANDA-P        ");
		sc1602_set_buffer_version(1, SRCVER_MAJOR, SRCVER_MINOR, SRCVER_BUILD);
		
		ui_queue_rp = ui_queue_wp = 0;
		current_index = 0;
		cursol_position = 0;
		ui_wait_timer = 1000;
		ui_proc++;
		break;
		
	case UI_01_STARTUP:
#ifdef START_WITH_ENDUSERMODE
		{
			UI_COMMAND ui_cmd;
			ui_cmd.cmd = UI_CMD_STARTED;
			ui_cmdset( ui_cmd );											// TODO: error handling
			ui_event_timer = ui_app_table[current_index].interval_time;		// timer reset
			ui_proc = UI_03_EXECUTING;
		}
#else
		if( ui_queue_rp || ui_queue_wp ) {
			ui_queue_rp = ui_queue_wp = 0;
			ui_proc++;
		}
#endif
		break;
		
	case UI_02_TOP_LEVEL:
		// Event check
		if( ui_queue_rp != ui_queue_wp ) {
			UI_COMMAND ui_cmd;
			
			switch( ui_cmd_queue[ui_queue_rp & MAX_CMD_QUEUE].cmd ) {
			case UI_CMD_NOP:
			case UI_CMD_INTEVAL:
			case UI_CMD_STARTED:
			case UI_CMD_KEY_PRESS_BACK:
				break;
				
			case UI_CMD_KEY_PRESS_UP:
				if( ui_cmd_queue[ui_queue_rp & MAX_CMD_QUEUE].param == OFF_EDGE ) break; // Ignore off edge
				if( cursol_position == 1 ) {
					cursol_position = 0;
					current_index--;
				} else {
					if( current_index != 0 ) current_index--;
				}
				break;
			case UI_CMD_KEY_PRESS_DOWN:
				if( ui_cmd_queue[ui_queue_rp & MAX_CMD_QUEUE].param == OFF_EDGE ) break; // Ignore off edge
				if( cursol_position == 0 ) {
					cursol_position = 1;
					current_index++;
				} else {
					if( ui_app_table[current_index+1].num != -1 ) current_index++;
				}
				break;
			case UI_CMD_KEY_PRESS_OK:
				if( ui_cmd_queue[ui_queue_rp & MAX_CMD_QUEUE].param == OFF_EDGE ) break; // Ignore off edge
				// app execute
				// set event
				ui_cmd.cmd = UI_CMD_STARTED;
				ui_cmdset( ui_cmd );											// TODO: error handling
				ui_event_timer = ui_app_table[current_index].interval_time;		// timer reset
				ui_proc = UI_03_EXECUTING;
				break;
			}
			ui_queue_rp++;
		}
		
		// Menu view
		if( cursol_position == 0) { // cursol in first line
			sc1602_set_buffer_cursol( 0, ui_app_table[current_index  ].display );
			sc1602_set_buffer       ( 1, ui_app_table[current_index+1].display );
		} else {
			sc1602_set_buffer       ( 0, ui_app_table[current_index-1].display );
			sc1602_set_buffer_cursol( 1, ui_app_table[current_index  ].display );
		}
		break;
	
	case UI_03_EXECUTING:
		if( ui_queue_rp != ui_queue_wp ) {
			int ret_val;
			ret_val =  ui_app_table[current_index].pExecFunc(ui_cmd_queue[ui_queue_rp & MAX_CMD_QUEUE]);
			if( ret_val != UI_RET_BUSY ) ui_queue_rp++;
			
			switch(ret_val) {
			case UI_RET_QUIT:
				ui_proc = UI_02_TOP_LEVEL;
				ui_event_timer = 0; // Timer stop
				break;
			default:
				break;
			}
		}
		break;
	default:
		//ui_proc = UI_00_INIT;
		break;
	}
	return 0;
}


// -------------------------------------------
//  Command set to Q
// -------------------------------------------
int ui_cmdset(UI_COMMAND uicmd) {
	// Overflow check
	if( (ui_queue_rp&MAX_CMD_QUEUE) == ((ui_queue_wp+1)&MAX_CMD_QUEUE ) ) {
		return -1;										// TODO: error handling
	}
	
	if( ui_wait_timer ) return -1;
	
	// Accept
	ui_cmd_queue[ui_queue_wp&MAX_CMD_QUEUE] = uicmd;
	ui_queue_wp++;
	
	return 0;
}


// -------------------------------------------
//  Local functions
// -------------------------------------------

// -------------------------------------------
//  UI Function - nop
// -------------------------------------------
static int ui_function_nop(UI_COMMAND uicmd) {
	static unsigned char counter;
	int ret_val;
	
	ret_val = UI_RET_READY;
	switch( uicmd.cmd ) {
	case UI_CMD_NOP:
		break;
		
	case UI_CMD_INTEVAL:
		counter++;
		sc1602_set_buffer_uint8(1, "1ms Counter:    ", counter);
		break;
		
	case UI_CMD_STARTED:
		counter = 0;
		sc1602_set_buffer(0, "NOP FUNCTION    ");
		sc1602_set_buffer(1, "                ");
		break;
		
	case UI_CMD_KEY_PRESS_UP:
		if( ui_cmd_queue[ui_queue_rp & MAX_CMD_QUEUE].param == OFF_EDGE ) break; // Ignore off edge
		sc1602_set_buffer(0, "NOP FUNCTION 1  ");
		break;
	case UI_CMD_KEY_PRESS_DOWN:
		if( ui_cmd_queue[ui_queue_rp & MAX_CMD_QUEUE].param == OFF_EDGE ) break; // Ignore off edge
		sc1602_set_buffer(0, "NOP FUNCTION 2  ");
		break;
	case UI_CMD_KEY_PRESS_BACK:
		if( ui_cmd_queue[ui_queue_rp & MAX_CMD_QUEUE].param == OFF_EDGE ) break; // Ignore off edge
		ret_val = UI_RET_QUIT;
		break;
	case UI_CMD_KEY_PRESS_OK:
		if( ui_cmd_queue[ui_queue_rp & MAX_CMD_QUEUE].param == OFF_EDGE ) break; // Ignore off edge
		sc1602_set_buffer(0, "NOP FUNCTION 4  ");
		break;
	}
	return ret_val;
}

// -------------------------------------------
//  UI Function - mem view
// -------------------------------------------
static int ui_function_memview(UI_COMMAND uicmd) {
	static unsigned char *addr;
	static short key_status; // 0=OFF, 1~100: up, -1~-100: down
	int ret_val;
	
	ret_val = UI_RET_READY;
	switch( uicmd.cmd ) {
	case UI_CMD_NOP:
		break;
		
	case UI_CMD_INTEVAL:
		if( key_status == 0 ) {
			sc1602_set_buffer_dump(1, addr);
		} else {
			// Status update
			if( key_status > 0 && key_status< 100 ) key_status++;
			if( key_status < 0 && key_status>-100 ) key_status--;
			
			// Address update
			if( key_status >    2 && key_status <   10 ) addr -= 0x8;
			if( key_status >   11 && key_status <   50 ) addr -= 0x10;
			if( key_status >   51 && key_status <   70 ) addr -= 0x100;
			if( key_status >   90                      ) addr -= 0x1000;
			if( key_status <   -2 && key_status >  -10 ) addr += 0x8;
			if( key_status <  -11 && key_status >  -50 ) addr += 0x10;
			if( key_status <  -51 && key_status >  -70 ) addr += 0x100;
			if( key_status <  -90                      ) addr += 0x1000;
			sc1602_set_buffer_ulong(0, "ADDR(h):        ", (unsigned long) addr);
			sc1602_set_buffer(1, "----------------");
		}
		break;
		
	case UI_CMD_STARTED:
		addr = (unsigned char *) 0x400000;
		key_status = 0;
		sc1602_set_buffer_ulong(0, "ADDR(h):        ", (unsigned long) addr);
		sc1602_set_buffer(1, "----------------");
		break;
		
	case UI_CMD_KEY_PRESS_UP:
		if( ui_cmd_queue[ui_queue_rp & MAX_CMD_QUEUE].param == OFF_EDGE ) {
			key_status = 0;
		} else {
			if( addr > (unsigned char *) 0x400000 ) addr -= 0x08;
			sc1602_set_buffer_ulong(0, "ADDR(h):        ", (unsigned long) addr);
			sc1602_set_buffer(1, "----------------");
			key_status = 1;
		}
		break;
	case UI_CMD_KEY_PRESS_DOWN:
		if( ui_cmd_queue[ui_queue_rp & MAX_CMD_QUEUE].param == OFF_EDGE ) {
			key_status = 0;
		} else {
			if( addr <= (unsigned char *) 0x500000 ) addr += 0x08;
			sc1602_set_buffer_ulong(0, "ADDR(h):        ", (unsigned long) addr);
			sc1602_set_buffer(1, "----------------");
			key_status = -1;
		}
		break;
	case UI_CMD_KEY_PRESS_BACK:
		if( ui_cmd_queue[ui_queue_rp & MAX_CMD_QUEUE].param == OFF_EDGE ) break; // Ignore off edge
		ret_val = UI_RET_QUIT;
		break;
	case UI_CMD_KEY_PRESS_OK:
		if( ui_cmd_queue[ui_queue_rp & MAX_CMD_QUEUE].param == OFF_EDGE ) {
			key_status = 0;
		} else {
			addr += 0x1000;
			if( addr >= (unsigned char *) 0x500000 ) addr = (unsigned char *)0x400000;
			sc1602_set_buffer_ulong(0, "ADDR(h):        ", (unsigned long) addr);
			sc1602_set_buffer(1, "----------------");
			key_status = -1;
		}
		break;
	}
	return ret_val;
}
extern void memfill_word( unsigned int val );

// -------------------------------------------
//  UI Function - mem chk
// -------------------------------------------
static int ui_function_memchk(UI_COMMAND uicmd) {
	static unsigned int *addr;
	static short check_proc;
	static unsigned char wait_count;		// Process wait for LCD drawing
	int ret_val, err;
	
	ret_val = UI_RET_READY;
	switch( uicmd.cmd ) {
	case UI_CMD_NOP:
		break;
		
	case UI_CMD_INTEVAL:
		switch( check_proc ) {
			case 0:		// start check phase
				check_proc++;
#ifdef UI_DEBUG_ON
			printf("[U/I] MemChk:\r\n");
#endif
				//break;
				
			case 1:		// Data write phase
				memfill_word(0xAAAA);
				addr = (unsigned int *) &eram_start;
				
				sc1602_set_buffer(0, "Mem chk(Even) ..");
				sc1602_set_buffer(1, "Mem chk(Odd)  ..");
				wait_count = 20;
				check_proc++;
				break;
				
			case 2:		// Data verify phase
				if( wait_count ) {
					wait_count--;
					return;
				}
				memchk_start = addr;
				memchk_ptn = 0xAAAA;
				memchk_rslt= 0;
				verify_mem256K();
				
#ifdef UI_DEBUG_ON
				printf("[U/I] Addr:%08X\r\n", addr);
#endif
				if(memchk_rslt) {
					sc1602_set_buffer(0, "Mem chk(Even) NG");
					sc1602_set_buffer(1, "Mem chk(Odd)  ..");
					check_proc = 100;
				} else {
					addr += 0x20000;
					if( addr >= (unsigned int *)&eram_end ) {
#ifdef UI_DEBUG_ON
						printf("[U/I] MemChk(1ST): OK.\r\n");
#endif
						sc1602_set_buffer(0, "Mem chk(Even) OK");
						sc1602_set_buffer(1, "Mem chk(Odd)  ..");
						wait_count = 20;
						check_proc = 3;
					} else {
						check_proc = 2;
					}
					
//					sc1602_set_buffer_ulong(1, "ADDR(h):        ", (unsigned long) addr);
				}
				break;
				
			case 3:		// Data write phase
				if( wait_count ) {
					wait_count--;
					return;
				}
				memfill_word(0x5555);
				addr = (unsigned int *) &eram_start;
				check_proc++;
				break;
				
			case 4:		// Data verify phase
				memchk_start = addr;
				memchk_ptn = 0x5555;
				memchk_rslt= 0;
				verify_mem256K();
				
#ifdef UI_DEBUG_ON
				printf("[U/I] Addr:%08X\r\n", addr);
#endif
				if(memchk_rslt) {
					sc1602_set_buffer(0, "Mem chk(Even) OK");
					sc1602_set_buffer(1, "Mem chk(Odd)  NG");
					check_proc = 100;
				} else {
					addr += 0x20000;
					if( addr >= (unsigned int *)&eram_end ) {
#ifdef UI_DEBUG_ON
						printf("[U/I] MemChk(2ND): OK.\r\n");
#endif
						check_proc = 101;
					} else {
						check_proc = 4;
					}
					
//					sc1602_set_buffer_ulong(1, "ADDR(h):        ", (unsigned long) addr);
				}
				break;
				
			case 100:		// NG.
				break;
			
			case 101:		// Good.
				sc1602_set_buffer(0, "Mem chk(Even) OK");
				sc1602_set_buffer(1, "Mem chk(Odd)  OK");
				break;
				
			default:
				break;
		}
		break;
		
	case UI_CMD_STARTED:
		check_proc = 0;
		
//		addr = &eram_start;
//		sc1602_set_buffer_ulong(1, "ADDR(h):        ", (unsigned long) addr);
		sc1602_set_buffer(0, "Checking memory ");
		break;
		
	case UI_CMD_KEY_PRESS_BACK:
		if( ui_cmd_queue[ui_queue_rp & MAX_CMD_QUEUE].param == OFF_EDGE ) break; // Ignore off edge
		ret_val = UI_RET_QUIT;
		break;

	case UI_CMD_KEY_PRESS_UP:
	case UI_CMD_KEY_PRESS_DOWN:
	case UI_CMD_KEY_PRESS_OK:
		break;
	}
	return ret_val;
}


enum enduser_mode_step {
	STEP_WAIT_FOR_READY,
	STEP_DIR_VIEW,
	STEP_FILE_READ,
	STEP_WAIT_FOR_START,
	STEP_TANSFER,
	STEP_COMPLETE,
};

// -------------------------------------------
//  UI Function - End user mode
// -------------------------------------------
static int ui_function_enduser_mode(UI_COMMAND uicmd) {
	int ret_val;
	static unsigned char current_index, cursol_position, mode_step;
	unsigned char *p_buffer_addr;

	p_buffer_addr = &eram_start;
	ret_val = UI_RET_READY;
	
	switch( uicmd.cmd ) {
		case UI_CMD_NOP:
			break;
			
		case UI_CMD_STARTED:
				current_index = 0;
				cursol_position = 0;
				mode_step = STEP_WAIT_FOR_READY;
				break;
			break;
			
		case UI_CMD_KEY_PRESS_UP:
			if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
			switch( mode_step ) {
				case STEP_DIR_VIEW:
					if( cursol_position == 1 ) {
						cursol_position = 0;
						current_index--;
					} else {
						if( current_index != 0 ) current_index--;
					}
					break;
					
				default:
					break;
			}
			break;
		case UI_CMD_KEY_PRESS_DOWN:
			if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
			switch( mode_step ) {
				case STEP_DIR_VIEW:
					if( cursol_position == 0 ) {
						if( TotalFileNum != 0) {
							cursol_position = 1;
							current_index++;
						}
					} else {
						if( current_index < TotalFileNum ) current_index++;
					}
					break;
					
				default:
					break;
			}
			break;
		case UI_CMD_KEY_PRESS_BACK:
			if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
			switch( mode_step ) {
				case STEP_FILE_READ:
					mode_step = STEP_DIR_VIEW;
					break;
					
				case STEP_DIR_VIEW:
					ret_val = UI_RET_QUIT;
					break;
					
				case STEP_TANSFER:
					serial_status(CLASS_REQ_RESET);
					mode_step = STEP_DIR_VIEW;
					break;
					
				default:
					break;
			}
			break;
		case UI_CMD_KEY_PRESS_OK:
			if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
			switch( mode_step ) {
				case STEP_WAIT_FOR_START:
					if( usbms_status( CLASS_REQ_NONE ) == CLASS_STS_READY ) {
						TransferStartRequest(p_buffer_addr, CurrentDirInfo[current_index].FileSize);
						mode_step = STEP_TANSFER;
					}
					break;
					
				case STEP_DIR_VIEW:
					if( CurrentDirInfo[current_index].Attributes & DIR_ATTR_DIRECTORY ) {
						// Change dir
						fat_change_directory(CurrentDirInfo[current_index].FirstCluster);
						current_index = 0;
					} else {
						// Open File
						fat_read_file(CurrentDirInfo[current_index].FirstCluster, CurrentDirInfo[current_index].FileSize );
						mode_step = STEP_FILE_READ;
					}
					break;
					
				default:
					break;
			}
			break;
		case UI_CMD_INTEVAL:
			switch( mode_step ) {
				case STEP_WAIT_FOR_READY:
					if( fat_status(CLASS_REQ_NONE) == CLASS_STS_READY ) 
						mode_step = STEP_DIR_VIEW;
					else {
						sc1602_set_buffer(0,"SYSTEM STARTING ");
						sc1602_set_buffer(1,"Wait a momnet...");
					}
					break;
					
				case STEP_DIR_VIEW:
					// Directory
					if( fat_status(CLASS_REQ_NONE) != CLASS_STS_READY ) {
						sc1602_set_buffer(0," Reading...     ");
						sc1602_set_buffer(1,"                ");
					} else if(TotalFileNum == 0) {
						sc1602_set_buffer(0,"--- NO FILE --- ");
						sc1602_set_buffer(1,"                ");
					} else {
						if( cursol_position == 0) { // cursol in first line
							sc1602_set_buffer_filename_cursol( 0, 
								(CurrentDirInfo[current_index].Attributes & DIR_ATTR_DIRECTORY), 
								CurrentDirInfo[current_index].Name );
							if( current_index < TotalFileNum) {
								sc1602_set_buffer_filename ( 1, 
									(CurrentDirInfo[current_index+1].Attributes & DIR_ATTR_DIRECTORY), 
									CurrentDirInfo[current_index+1].Name );
							} else {
								sc1602_set_buffer(1,"                ");
							}
						} else {
							sc1602_set_buffer_filename       ( 0, 
								(CurrentDirInfo[current_index-1].Attributes & DIR_ATTR_DIRECTORY), 
								CurrentDirInfo[current_index-1].Name );
							sc1602_set_buffer_filename_cursol( 1, 
								(CurrentDirInfo[current_index].Attributes & DIR_ATTR_DIRECTORY), 
								CurrentDirInfo[current_index].Name );
						}
					}
					break;
					
				case STEP_FILE_READ:
					if( fat_status(CLASS_REQ_NONE) == CLASS_STS_READY ) {
						mode_step = STEP_WAIT_FOR_START;
					} else {
						sc1602_set_buffer_filename       ( 0, 0, CurrentDirInfo[current_index].Name );
						sc1602_set_buffer_progress_kb(1, ReadFileStatus.ReadSize, ReadFileStatus.FileSize);
					}
					break;
					
				case STEP_WAIT_FOR_START:
					sc1602_set_buffer_filename       ( 0, 0, CurrentDirInfo[current_index].Name );
					sc1602_set_buffer (1," Ready to start.");
					break;
					
				case STEP_TANSFER:
					ui_serial_progress();
					break;
					
				default:
					break;
			}
			break;
			
	}
	return ret_val;
}


