
/* --------------------------------------------- */
/*  H8-3069F SC-1602 control function            */
/*                                               */
/*  CPU    : Renesus H8/3069F 25MHz              */
/*  Memory : ROM 512KB, RAM 16KB E-RAM 2MB       */
/*                (c) KAZ.Imamura                */
/* --------------------------------------------- */


#include "sc1602.h"

#define _DB4	P4DR.BIT.B0
#define _DB5	P4DR.BIT.B1
#define _DB6	P4DR.BIT.B2
#define _DB7	P4DR.BIT.B3
#define _RS		P4DR.BIT.B4
#define _E		P4DR.BIT.B5


// -------------------------------------------
//  Proto type definitions
// -------------------------------------------
void sc1602_1ms_handler(void);
int sc1602_initialize(void);
int sc1602_process(void);
int sc1602_set_buffer(unsigned char buf, char *data);
int sc1602_set_buffer_cursol(unsigned char buf, char *data);
int sc1602_set_buffer_uint8(unsigned char buf, char *data, unsigned char uint8);
int sc1602_set_buffer_ulong(unsigned char buf, char *data, unsigned long val);
int sc1602_set_buffer_dump (unsigned char buf, unsigned char *data);
int sc1602_set_buffer_variable1 (unsigned char buf, unsigned char *data);
int sc1602_set_buffer_variable2 (unsigned char buf, unsigned short *data);
int sc1602_set_buffer_variable3 (unsigned char buf, unsigned long *data);
int sc1602_set_buffer_progress_kb (unsigned char buf, unsigned long val1, unsigned long val2);

int sc1602_set_buffer_version (unsigned char buf, unsigned char major, unsigned char minor, unsigned char build);

// Locals
void sc1602_command_output4(char cmd);
void sc1602_command_output8(char cmd);
void sc1602_character_put(char data);
void sc1602_clear_screen( void );
void sc1602_locate_screen( unsigned int x, unsigned int y );


// -------------------------------------------
//  Variables
// -------------------------------------------
char sc1602_buffer[MAX_LINE][MAX_COLUMN];
int  sc1602_buffer_updates[MAX_LINE];

// Locals
volatile int lcd_wait_timer;
volatile int lcd_proc;
volatile int lcd_disable_timer;
const char hex2ascii[] = "0123456789ABCDEF";

enum lcd_process_mode {
	LCD_00_INIT,
	LCD_01_INIT_START1,
	LCD_02_INIT_WAIT1,
	LCD_03_INIT_START2,
	LCD_04_INIT_WAIT2,
	LCD_05_INIT_START3,
	LCD_06_INIT_WAIT3,
	LCD_07_4BIT_SET,
	LCD_08_DUTY_SET,
	LCD_09_DISPLAY_SET,
	LCD_10_MODE_SET,
	LCD_11_CLEAR_DISPLAY,
	LCD_12_CHECK1,
	LCD_13_OUTPUT1,
	LCD_14_CHECK2,
	LCD_15_OUTPUT2,
	
};

// -------------------------------------------
//  Interrupt handlers
// -------------------------------------------
void sc1602_1ms_handler(void) {
	if( lcd_wait_timer )    lcd_wait_timer--;
	if( lcd_disable_timer ) lcd_disable_timer--;
}


// -------------------------------------------
//  Initialize
// -------------------------------------------
int sc1602_initialize(void) {
	lcd_wait_timer = 0;
	lcd_proc = LCD_00_INIT;
	lcd_disable_timer = 500;

#if 0
	sc1602_set_buffer(0, " NCTX-BOX Rev.0 ");
	sc1602_set_buffer(1, "    KAZ.Imamura ");
#endif
}

// -------------------------------------------
//  Main process
// -------------------------------------------
int sc1602_process(void) {
	static int data_pos;
	if( lcd_disable_timer ) return -1;
	
	switch( lcd_proc ) {
	case LCD_00_INIT:
		_RS = 0;
		lcd_wait_timer = 20;			// 20ms wait timer set
		lcd_proc++;
		break;
		
	case LCD_01_INIT_START1:
	case LCD_03_INIT_START2:
	case LCD_05_INIT_START3:
		if( !lcd_wait_timer ) {
			sc1602_command_output4(0x03);
			lcd_proc++;
		}
		break;
		
	case LCD_02_INIT_WAIT1:
	case LCD_04_INIT_WAIT2:
	case LCD_06_INIT_WAIT3:
		lcd_proc++;
		lcd_wait_timer = 5;					// 5ms wait timer set
		_E = 0;
		break;
	
	case LCD_07_4BIT_SET:
		if( !lcd_wait_timer ) {
			sc1602_command_output4(0x02);
			lcd_proc++;
			lcd_wait_timer = 5;				// 5ms wait timer set
		}
		break;
		
	case LCD_08_DUTY_SET:
		if( !lcd_wait_timer ) {
			sc1602_command_output8(0x28);
			lcd_proc++;
			lcd_wait_timer = 5;				// 5ms wait timer set
		}
		break;
		
	case LCD_09_DISPLAY_SET:
		if( !lcd_wait_timer ) {
			sc1602_command_output8(0x0C); // display on,cursor off,blink off 
			lcd_proc++;
			lcd_wait_timer = 5;				// 5ms wait timer set
		}
		break;
		
	case LCD_10_MODE_SET:
		if( !lcd_wait_timer ) {
			sc1602_command_output8(0x06); // address:auto increment,cursor shift:right
			lcd_proc++;
			lcd_wait_timer = 5;				// 5ms wait timer set
		}
		break;
		
	case LCD_11_CLEAR_DISPLAY:
		if( !lcd_wait_timer ) {
			sc1602_command_output8(0x01); // clear display
			lcd_proc++;
			lcd_wait_timer = 5;				// 5ms wait timer set
		}
		break;
	case LCD_12_CHECK1:
		if( sc1602_buffer_updates[0] ) {
			sc1602_buffer_updates[0] = 0;
			sc1602_locate_screen(0,0);
			data_pos=0;
			lcd_proc++;
		} else {
			lcd_proc = LCD_14_CHECK2;
		}
		break;
	
	case LCD_13_OUTPUT1:
		sc1602_character_put( sc1602_buffer[0][data_pos++] );
		if( data_pos >= 16 ) {
			lcd_proc++;
		}
		break;
			
	case LCD_14_CHECK2:
		if( sc1602_buffer_updates[1] ) {
			sc1602_buffer_updates[1] = 0;
			sc1602_locate_screen(0,1);
			data_pos=0;
			lcd_proc++;
		} else {
			lcd_proc = LCD_12_CHECK1;
		}
		break;
	
	case LCD_15_OUTPUT2:
		sc1602_character_put( sc1602_buffer[1][data_pos++] );
		if( data_pos >= 16 ) {
			lcd_proc = LCD_12_CHECK1;
		}
		break;
			
	default:
		lcd_proc = LCD_00_INIT;
		break;
	}
	return 0;
}

// -------------------------------------------
//  Buffer set service function
// -------------------------------------------
int sc1602_set_buffer(unsigned char buf, char *data) {
	unsigned char rp;
	
	if( buf >= MAX_LINE ) return -1;
	
	for(rp=0; rp<MAX_COLUMN; rp++) {
		sc1602_buffer[buf][rp] = data[rp];
	}
	sc1602_buffer_updates[buf] = 1;
	
	return 0;
}

// -------------------------------------------
//  Buffer set service function W/ cursol
// -------------------------------------------
int sc1602_set_buffer_cursol(unsigned char buf, char *data) {
	sc1602_set_buffer(buf,data);
	sc1602_buffer[buf][0] = '>';
}

// -------------------------------------------
//  Buffer set service function W/ uint8 3digit
// -------------------------------------------
int sc1602_set_buffer_uint8(unsigned char buf, char *data, unsigned char uint8) {
	sc1602_set_buffer(buf,data);
	sc1602_buffer[buf][13] = uint8/100+0x30;
	uint8 %= 100;
	sc1602_buffer[buf][14] = uint8/10+0x30;
	sc1602_buffer[buf][15] = uint8%10+0x30;
}

// -------------------------------------------
//  Buffer set service function W/ ulong 8 digit
// -------------------------------------------
int sc1602_set_buffer_ulong(unsigned char buf, char *data, unsigned long val) {
	sc1602_set_buffer(buf,data);
	
	sc1602_buffer[buf][ 8] =  hex2ascii[(val >>28) & 0x0F ];
	sc1602_buffer[buf][ 9] =  hex2ascii[(val >>24) & 0x0F ];
	sc1602_buffer[buf][10] =  hex2ascii[(val >>20) & 0x0F ];
	sc1602_buffer[buf][11] =  hex2ascii[(val >>16) & 0x0F ];
	sc1602_buffer[buf][12] =  hex2ascii[(val >>12) & 0x0F ];
	sc1602_buffer[buf][13] =  hex2ascii[(val >> 8) & 0x0F ];
	sc1602_buffer[buf][14] =  hex2ascii[(val >> 4) & 0x0F ];
	sc1602_buffer[buf][15] =  hex2ascii[(val >> 0) & 0x0F ];
}

// -------------------------------------------
//  Buffer set service function (mem dump 8byte)
// -------------------------------------------
int sc1602_set_buffer_dump (unsigned char buf, unsigned char *data) {
	sc1602_set_buffer(buf," ");
	
	sc1602_buffer[buf][ 0] =  hex2ascii[(*(data + 0) >> 4) & 0x0F ];
	sc1602_buffer[buf][ 1] =  hex2ascii[(*(data + 0)     ) & 0x0F ];
	sc1602_buffer[buf][ 2] =  hex2ascii[(*(data + 1) >> 4) & 0x0F ];
	sc1602_buffer[buf][ 3] =  hex2ascii[(*(data + 1)     ) & 0x0F ];
	sc1602_buffer[buf][ 4] =  hex2ascii[(*(data + 2) >> 4) & 0x0F ];
	sc1602_buffer[buf][ 5] =  hex2ascii[(*(data + 2)     ) & 0x0F ];
	sc1602_buffer[buf][ 6] =  hex2ascii[(*(data + 3) >> 4) & 0x0F ];
	sc1602_buffer[buf][ 7] =  hex2ascii[(*(data + 3)     ) & 0x0F ];
	sc1602_buffer[buf][ 8] =  hex2ascii[(*(data + 4) >> 4) & 0x0F ];
	sc1602_buffer[buf][ 9] =  hex2ascii[(*(data + 4)     ) & 0x0F ];
	sc1602_buffer[buf][10] =  hex2ascii[(*(data + 5) >> 4) & 0x0F ];
	sc1602_buffer[buf][11] =  hex2ascii[(*(data + 5)     ) & 0x0F ];
	sc1602_buffer[buf][12] =  hex2ascii[(*(data + 6) >> 4) & 0x0F ];
	sc1602_buffer[buf][13] =  hex2ascii[(*(data + 6)     ) & 0x0F ];
	sc1602_buffer[buf][14] =  hex2ascii[(*(data + 7) >> 4) & 0x0F ];
	sc1602_buffer[buf][15] =  hex2ascii[(*(data + 7)     ) & 0x0F ];
}

// -------------------------------------------
//  Buffer set service function (variable view 1byte)
// -------------------------------------------
int sc1602_set_buffer_variable1 (unsigned char buf, unsigned char *data) {
	//                     0123456789ABCDEF
	sc1602_set_buffer(buf,"     0x   (    )");
	
	// Hex
	sc1602_buffer[buf][ 7] =  hex2ascii[(*data >> 4) & 0x0F ];
	sc1602_buffer[buf][ 8] =  hex2ascii[(*data     ) & 0x0F ];
	// Decimal
	sc1602_buffer[buf][12] =  hex2ascii[(*data/100)       ];
	sc1602_buffer[buf][13] =  hex2ascii[(*data/10)   % 10 ];
	sc1602_buffer[buf][14] =  hex2ascii[(*data)      % 10 ];
}

// -------------------------------------------
//  Buffer set service function (variable view 2byte)
// -------------------------------------------
int sc1602_set_buffer_variable2 (unsigned char buf, unsigned short *data) {
	//                     0123456789ABCDEF
	sc1602_set_buffer(buf," 0x     (      )");
	
	// Hex
	sc1602_buffer[buf][ 3] =  hex2ascii[(*data >> 12) & 0x0F ];
	sc1602_buffer[buf][ 4] =  hex2ascii[(*data >>  8) & 0x0F ];
	sc1602_buffer[buf][ 5] =  hex2ascii[(*data >>  4) & 0x0F ];
	sc1602_buffer[buf][ 6] =  hex2ascii[(*data      ) & 0x0F ];
	// Decimal
	sc1602_buffer[buf][10] =  hex2ascii[(*data/10000)     ];
	sc1602_buffer[buf][11] =  hex2ascii[(*data/1000) % 10 ];
	sc1602_buffer[buf][12] =  hex2ascii[(*data/100)  % 10 ];
	sc1602_buffer[buf][13] =  hex2ascii[(*data/10)   % 10 ];
	sc1602_buffer[buf][14] =  hex2ascii[(*data)      % 10 ];
}

// -------------------------------------------
//  Buffer set service function (variable view 4byte)
// -------------------------------------------
int sc1602_set_buffer_variable3 (unsigned char buf, unsigned long *data) {
	//                     0123456789ABCDEF
	sc1602_set_buffer(buf,"      0x        ");
	
	// Hex
	sc1602_buffer[buf][ 8] =  hex2ascii[(*data >> 28) & 0x0F ];
	sc1602_buffer[buf][ 9] =  hex2ascii[(*data >> 24) & 0x0F ];
	sc1602_buffer[buf][10] =  hex2ascii[(*data >> 20) & 0x0F ];
	sc1602_buffer[buf][11] =  hex2ascii[(*data >> 16) & 0x0F ];
	sc1602_buffer[buf][12] =  hex2ascii[(*data >> 12) & 0x0F ];
	sc1602_buffer[buf][13] =  hex2ascii[(*data >>  8) & 0x0F ];
	sc1602_buffer[buf][14] =  hex2ascii[(*data >>  4) & 0x0F ];
	sc1602_buffer[buf][15] =  hex2ascii[(*data      ) & 0x0F ];
}

// -------------------------------------------
//  Buffer set service function (Progress with KB))
// -------------------------------------------
int sc1602_set_buffer_progress_kb (unsigned char buf, unsigned long val1, unsigned long val2) {
	unsigned int val3;
	//                     0123456789ABCDEF
	sc1602_set_buffer(buf,"    K/    K    %");
	
	val1 /= 1024;
	sc1602_buffer[buf][ 0] =  hex2ascii[(val1/1000)      ];
	sc1602_buffer[buf][ 1] =  hex2ascii[(val1/100)  % 10 ];
	sc1602_buffer[buf][ 2] =  hex2ascii[(val1/10)   % 10 ];
	sc1602_buffer[buf][ 3] =  hex2ascii[(val1)      % 10 ];
	val2 /= 1024;
	sc1602_buffer[buf][ 6] =  hex2ascii[(val2/1000)      ];
	sc1602_buffer[buf][ 7] =  hex2ascii[(val2/100)  % 10 ];
	sc1602_buffer[buf][ 8] =  hex2ascii[(val2/10)   % 10 ];
	sc1602_buffer[buf][ 9] =  hex2ascii[(val2)      % 10 ];

	val3 = (unsigned int) (val1*100/val2);
	sc1602_buffer[buf][12] =  hex2ascii[(val3/100)  % 10 ];
	sc1602_buffer[buf][13] =  hex2ascii[(val3/10)   % 10 ];
	sc1602_buffer[buf][14] =  hex2ascii[(val3)      % 10 ];
}

// -------------------------------------------
//  Buffer set service function (version)
// -------------------------------------------
int sc1602_set_buffer_version (unsigned char buf, unsigned char major, unsigned char minor, unsigned char build) {
	//                     0123456789ABCDEF
	sc1602_set_buffer(buf," Ver:   .   .   ");
	
	// Major
	sc1602_buffer[buf][ 5] =  hex2ascii[(major/100)  % 10 ];
	sc1602_buffer[buf][ 6] =  hex2ascii[(major/10)   % 10 ];
	sc1602_buffer[buf][ 7] =  hex2ascii[(major)      % 10 ];
	// Minor
	sc1602_buffer[buf][ 9] =  hex2ascii[(minor/100)  % 10 ];
	sc1602_buffer[buf][10] =  hex2ascii[(minor/10)   % 10 ];
	sc1602_buffer[buf][11] =  hex2ascii[(minor)      % 10 ];
	// Build
	sc1602_buffer[buf][13] =  hex2ascii[(build/100)  % 10 ];
	sc1602_buffer[buf][14] =  hex2ascii[(build/10)   % 10 ];
	sc1602_buffer[buf][15] =  hex2ascii[(build)      % 10 ];
}

// -------------------------------------------
//  Local functions
// -------------------------------------------
void sc1602_command_output4(char cmd) {
	_DB4 = (cmd & 0x01) ? 1 : 0;
	_DB5 = (cmd & 0x02) ? 1 : 0;
	_DB6 = (cmd & 0x04) ? 1 : 0;
	_DB7 = (cmd & 0x08) ? 1 : 0;
	_E = 1;
	wait_100us();
	_E = 0;
	wait_100us();
}

void sc1602_command_output8(char cmd) {
	sc1602_command_output4((cmd>>4) & 0x0F);
	sc1602_command_output4((cmd   ) & 0x0F);
}

void sc1602_character_put(char data) {
	_RS = 1;
	sc1602_command_output4((data>>4) & 0x0F);
	sc1602_command_output4((data   ) & 0x0F);
	_RS = 0;
}

void sc1602_clear_screen( void ) {
	sc1602_command_output8(0x01);
	lcd_disable_timer = 1640;
}

void sc1602_locate_screen( unsigned int x, unsigned int y ) {
	sc1602_command_output8(0x80|x|((y&0x01)<<6));
//	sc1602_command_output8(0x80+x+0x40*y);
}
