﻿// Copyright(C) 2003, 2005 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2008 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: PrefixedExpressionNode.cs 430 2011-10-19 06:31:33Z darai $

using System;

namespace Protra.Lib.Lang
{
	/// <summary>
	/// 前置式のノードを表すクラス。
	/// </summary>
	public class PrefixedExpressionNode : Node
	{
		/// <summary>
		/// 演算の種類を表すキャラクタ
		/// </summary>
		char op;
		/// <summary>
		/// @作用素の値に対応するノード
		/// </summary>
		Node atNode;
		/// <summary>
		/// キャストのタイプ
		/// </summary>
		Value.Type type;
		/// <summary>
		/// 式のノード
		/// </summary>
		Node expNode;

		/// <summary>
		/// 構文解析する。
		/// </summary>
		/// <exception cref="Protra.Lib.Lang.ParseException">
		/// 構文解析中にエラーが発生したときにthrowされる。
		/// </exception>
        /// <return>
        /// 解析した結果のノード。
        /// </return>
		public override Node Parse()
		{
            Token = Scanner.Token;
            if (Token.Type != TokenType.Operator)
                return new SuffixedExpressionNode().Parse();
            string token = Token.Value;
			switch (token)
			{
				case "{": // @作用素
                    op = token[0];
                    Scanner.Scan();
					atNode = new ExpressionNode().Parse();
					if (Scanner.Token.Value != "}")
						throw new ParseException("'}' expected", Token);
                    Scanner.Scan();
					break;
				case "+":
				case "-":
				case "!":
					op = token[0];
					Scanner.Scan();
					break;
				case "(": // キャストの可能性あり
                    Token peek = Scanner.Peek(); // 先読み
                    if (peek.Type != TokenType.Type)
                        return new SuffixedExpressionNode().Parse();
                    op = token[0];
                    Scanner.Scan();
                    type = (Value.Type)Enum.Parse
                        (typeof(Value.Type), Scanner.Token.Value, true);
                    Scanner.Scan();
                    if (Scanner.Token.Value != ")")
                        throw new ParseException("')' expected", Scanner.Token);
                    Scanner.Scan();
					break;
				default:
					return new SuffixedExpressionNode().Parse();
			}
            expNode = new PrefixedExpressionNode().Parse();
            return this;
		}

		/// <summary>
		/// プログラムを実行する。
		/// </summary>
		/// <exception cref="Protra.Lib.Lang.RuntimeException">
		/// プログラム実行中にエラーが発生した場合にthrowされる。
		/// </exception>
		/// <param name="resource">リソース</param>
		/// <param name="at">int型@作用素の値</param>
		/// <param name="ats">string型@作用素の値</param>
		/// <returns>実行結果</returns>
		public override Value Execute(Resource resource, int at, string ats)
		{
            try
            {
                switch (op)
                {
                    case '{': // @作用素
                        Value val = atNode.Execute(resource, at, ats);
                        if (val.ValueType == Value.Type.Int)
                            return expNode.Execute(resource, at + (int)val.InnerValue, ats);
                        else if (val.ValueType == Value.Type.String)
                            return expNode.Execute(resource, at, (string)val.InnerValue);
                        else
                            throw new RuntimeException("non-(int or string) value for at operator", Token);
                    case '+':
                        return +expNode.Execute(resource, at, ats);
                    case '-':
                        return -expNode.Execute(resource, at, ats);
                    case '!':
                        return !expNode.Execute(resource, at, ats);
                    case '(': // キャスト
                        val = expNode.Execute(resource, at, ats);
                        if (val == null)
                            throw new RuntimeException("null casted", Token);
                        return val.Cast(type);
                }
            }
            catch (RuntimeException e)
            {
                e.Token = Token;
                throw e;
            }
			return expNode.Execute(resource, at, ats);
		}
	}
}
