﻿// Copyright (C) 2003, 2005 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2008, 2013 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: AssignmentNode.cs 472 2013-06-26 15:26:49Z panacoran $

namespace Protra.Lib.Lang
{
    /// <summary>
    /// 代入式のノードを表すクラス。
    /// </summary>
    public class AssignmentNode : ExpressionNode
    {
        /// <summary>
        /// 代入される式のノード。
        /// </summary>
        private ExpressionNode _lvalueNode;

        /// <summary>
        /// 代入する式のノード。
        /// </summary>
        private ExpressionNode _rvalueNode;

        /// <summary>
        /// 構文解析する。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.ParseException">
        /// 構文解析中にエラーが発生したときにthrowされる。
        /// </exception>
        /// <return>
        /// 解析した結果のノード。
        /// </return>
        public override ExpressionNode Parse(Scanner scanner)
        {
            Token = scanner.Token;
            var node = new LogicalExpressionNode().Parse(scanner);
            if (scanner.Token.Value != "=")
                return node;
            if (!(node is VariableNode || node is SuffixedExpressionNode))
                throw new RuntimeException("invalid assignment.", Token);
            _lvalueNode = node;
            scanner.Scan();
            _rvalueNode = new ExpressionNode().Parse(scanner);
            return this;
        }

        /// <summary>
        /// 式を評価する。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// 評価中にエラーが発生した場合にthrowされる。
        /// </exception>
        /// <param name="resource">リソース</param>
        /// <param name="at">int型@作用素の値</param>
        /// <param name="ats">string型@作用素の値</param>
        /// <returns>値</returns>
        public override Value Evaluate(Resource resource, int at, string ats)
        {
            var val = _rvalueNode.Evaluate(resource, at, ats);
            if (_lvalueNode is VariableNode)
                return ((VariableNode)_lvalueNode).Assign(resource, val);
            if (_lvalueNode is SuffixedExpressionNode)
                return ((SuffixedExpressionNode)_lvalueNode).Assign(resource, at, ats, val);
            throw new RuntimeException("invalid assignment.", Token);
        }
    }
}
