﻿// Copyright (C) 2003 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2004, 2005 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: BrandTreeView.cs 321 2010-03-29 11:37:51Z panacoran $

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Windows.Forms;
using System.Text.RegularExpressions;
using Protra.Lib;
using Protra.Lib.Data;

namespace Protra.Controls
{
	/// <summary>
	/// 銘柄リストを表示するTreeViewです。
	/// </summary>
	public class BrandTreeView : System.Windows.Forms.TreeView
	{
		/// <summary>
		/// 業種の分類
		/// </summary>
		private string[] categories = {
			"水産/鉱業/建設", "食品", "繊維/紙/パルプ", "化学/医薬品",
			"石油/ゴム/窯業", "鉄鋼/非鉄金属", "機械",
			"電機", "輸送機/精密", "商業/百貨店", "銀行/証券/損保",
			"不動産/輸送", "電力/ガス/サービス" };
		/// <summary>
		/// コードのインデックス
		/// </summary>
		private int[] codeIndex = {
			1300, 2000, 3000, 4000, 5000, 5400, 6000, 6500,
			7000, 8000, 8300, 8800, 9400, 10000 };
        
		/// <summary>
		/// 必要なデザイナ変数です。
		/// </summary>
		private System.ComponentModel.Container components = null;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		public BrandTreeView()
		{
			// この呼び出しは、Windows.Forms フォーム デザイナで必要です。
			InitializeComponent();

			this.BeforeExpand += new TreeViewCancelEventHandler(OnBeforeExpand);
			//Initialize();
		}

		/// <summary>
		/// 使用されているリソースに後処理を実行します。
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Component Designer generated code
		/// <summary>
		/// デザイナ サポートに必要なメソッドです。このメソッドの内容を
		/// コード エディタで変更しないでください。
		/// </summary>
		private void InitializeComponent()
		{
			components = new System.ComponentModel.Container();
		}
		#endregion

		/// <summary>
		/// 初期化します。
		/// </summary>
		public void Initialize()
		{
			Nodes.Clear();
			// 市場の登録
			foreach (var market in BrandData.MarketNames)
			{

				TreeNode tn = new TreeNode(market);
				tn.Nodes.Add("dummy");
				Nodes.Add(tn);
			}
			// 1100未満の銘柄の登録
            for (int i = 1000; i < 1100; i++)
            {
                Brand b = GlobalEnv.BrandData[i.ToString()];
                if (b.Unit == 0)
                    continue;
                Nodes.Add(b.Code + " " + b.Name);
            }
			// 日経平均を選択
			Select();
            SelectedNode = Nodes[BrandData.Markets.Length];
		}

		private void OnBeforeExpand(object sender, System.Windows.Forms.TreeViewCancelEventArgs e)
		{
			// ダミーノードのクリア
			e.Node.Nodes.Clear();

			if(e.Node.Parent == null) // 市場から業種への展開
			{
				foreach(string category in categories)
				{
					TreeNode tn = new TreeNode(category);
					tn.Nodes.Add("dummy");
					e.Node.Nodes.Add(tn);
				}
				return;
			}

			// 業種から銘柄への展開
            for (int i = codeIndex[e.Node.Index]; i < codeIndex[e.Node.Index + 1]; i++)
            {
                Brand b = GlobalEnv.BrandData[i.ToString()];
                if (b.Unit == 0)
                    continue;
                if (b.Market != BrandData.Markets[e.Node.Parent.Index])
                    continue;
                e.Node.Nodes.Add(b.Code + " " + b.Name);
            }
		}

		/// <summary>
		/// 選択されている銘柄を取得します。
		/// </summary>
		public string SelectedCode
		{
			get
			{
				TreeNode node = this.SelectedNode;
				if(node == null)
					return null;
				Regex regex = new Regex("^([0-9]+) .+$");
				Match match = regex.Match(node.Text);
				if(match.Success)
                    return match.Groups[1].Value;
				else
					return null;
			}
		}
	}
}
