﻿// Copyright (C) 2003, 2005 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2008 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: ComparisonNode.cs 430 2011-10-19 06:31:33Z darai $

using System;
using System.Collections.Generic;

namespace Protra.Lib.Lang
{
	/// <summary>
	/// 大小比較のノードを表すクラス。
	/// </summary>
	public class ComparisonNode : Node
	{
		/// <summary>
		/// ノードのリスト
		/// </summary>
        private List<Node> nodeList;
		/// <summary>
		/// 演算の種類を表す文字列のリスト
		/// </summary>
		private List<string> opStrList;

		/// <summary>
		/// 構文解析する。
		/// </summary>
		/// <exception cref="Protra.Lib.Lang.ParseException">
		/// 構文解析中にエラーが発生したときにthrowされる。
		/// </exception>
        /// <return>
        /// 解析した結果のノード。
        /// </return>
		public override Node Parse()
		{
			Token = Scanner.Token;
			Node node = new ArithmeticExpressionNode().Parse();
            string token = Scanner.Token.Value;
            if (!(token == "<" || token == ">" || token == "<=" || token == ">="))
                return node;
            nodeList = new List<Node>();
            opStrList = new List<string>();
			nodeList.Add(node);
			do
			{
				opStrList.Add(token);
				Scanner.Scan();
				nodeList.Add(new ArithmeticExpressionNode().Parse());
                token = Scanner.Token.Value;
			}
            while (token == "<" || token == ">" || token == "<=" || token == ">=");
            return this;
		}

		/// <summary>
		/// プログラムを実行する。
		/// </summary>
		/// <exception cref="Protra.Lib.Lang.RuntimeException">
		/// プログラム実行中にエラーが発生した場合にthrowされる。
		/// </exception>
		/// <param name="resource">リソース</param>
		/// <param name="at">int型@作用素の値</param>
		/// <param name="ats">string型@作用素の値</param>
		/// <returns>実行結果</returns>
		public override Value Execute(Resource resource, int at, string ats)
		{
			try
			{
                Value val1 = nodeList[0].Execute(resource, at, ats);
				for(int i = 1; i < nodeList.Count; i++)
				{
                    Value val2 = nodeList[i].Execute(resource, at, ats);
					switch (opStrList[i - 1])
					{
						case "<":
							val1 = new Value(val1 < val2);
							break;
						case ">":
							val1 = new Value(val1 > val2);
							break;
						case "<=":
							val1 = new Value(val1 <= val2);
							break;
						default:
							val1 = new Value(val1 >= val2);
							break;
					}
				}
				return val1;
			}
			catch(RuntimeException e)
			{
				if (e.Token == null)
					e.Token = Token;
				throw e;
			}		
		}
	}
}
