﻿// Copyright (C) 2003, 2005 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2008 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: ReturnStatementNode.cs 430 2011-10-19 06:31:33Z darai $

using System;

namespace Protra.Lib.Lang
{
	/// <summary>
	/// return文のノードを表すクラス。
	/// </summary>
	public class ReturnStatementNode : Node
	{
		/// <summary>
		/// 式のノード
		/// </summary>
		private Node expNode;

		/// <summary>
		/// 構文解析する。
		/// </summary>
		/// <exception cref="Protra.Lib.Lang.ParseException">
		/// 構文解析中にエラーが発生したときにthrowされる。
		/// </exception>
        /// <return>
        /// 解析した結果のノード。
        /// </return>
		public override Node Parse()
		{
			Token = Scanner.Token;
			if (!Scanner.Scan())
				throw new ParseException("';' expected", Scanner.Token);
			if (Scanner.Token.Value == ";")
                return this;
			expNode = new ExpressionNode().Parse();
			if (Scanner.Token.Value != ";")
				throw new ParseException("';' expected", Scanner.Token);
            return this;
		}

		/// <summary>
		/// プログラムを実行する。
		/// </summary>
		/// <exception cref="Protra.Lib.Lang.RuntimeException">
		/// プログラム実行中にエラーが発生した場合にthrowされる。
		/// </exception>
		/// <exception cref="Protra.Lib.Lang.ReturnStatementExecutedException">
		/// 正常に戻り値が計算された場合にthrowされる。
		/// </exception>
		/// <param name="resource">リソース</param>
		/// <param name="at">int型@作用素の値</param>
		/// <param name="ats">string型@作用素の値</param>
		/// <returns>実行結果</returns>
		public override Value Execute(Resource resource, int at, string ats)
		{
			Value val = null;
			if(expNode != null)
				val = expNode.Execute(resource, at, ats);
			throw new ReturnStatementExecutedException(val);
		}
	}

	/// <summary>
	/// ReturnStatementNode.Executeが呼ばれたときにthrowされる例外。
	/// </summary>
	public class ReturnStatementExecutedException : Exception
	{
		/// <summary>
		/// 戻り値を示す値
		/// </summary>
		private Value val;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="val">戻り値を示す値</param>
		public ReturnStatementExecutedException(Value val)
		{
			this.val = val;
		}

		/// <summary>
		/// 戻り値を示す値を取得する。
		/// </summary>
		public Value Value
		{
			get { return val; }
		}
	}
}
