// Copyright (C) 2005-2008 panacorn <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: PriceDataUpdator.cs,v 1.38 2008-04-18 19:29:00 panacoran Exp $

using System;
using System.Text;
using System.Collections;
using System.ComponentModel;
using System.IO;
using System.Net;
using Protra.Lib.Db;
using Protra.Lib.Archiver;

namespace Protra.Lib.Update
{
	/// <summary>
	/// f[^\[X񋓌^łB
	/// </summary>
	public enum PriceDataSource
	{
		/// <summary>
		/// 
		/// </summary>
		KabukaJoho,
#if MUJINZOU
        /// <summary>
        /// s
        /// </summary>
        Mujinzou,
#endif
        /// <summary>
        /// f[^_E[hTCg
        /// </summary>
        KdbCom,
        /// <summary>
        /// Yahoo!t@CiX
        /// </summary>
        YahooFinance
	}

    /// <summary>
    /// f[^킷NXB
    /// </summary>
    public class PriceData
    {
        private int code;
        private Db.MarketId marketId;
        private string name;
        private DateTime date;
        private int open;
        private int high;
        private int low;
        private int close;
        private double volume;
        private double split = 1;

        /// <summary>
        /// R[h
        /// </summary>
        public int Code
        {
            get { return code; }
            set { code = value; }
        }

        /// <summary>
        /// sID
        /// </summary>
        public Db.MarketId MarketId
        {
            get { return marketId; }
            set { marketId = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        public string Name
        {
            get { return name; }
            set { name = value; }
        }

        /// <summary>
        /// t
        /// </summary>
        public DateTime Date
        {
            get { return date; }
            set { date = value; }
        }

        /// <summary>
        /// nl
        /// </summary>
        public int Open
        {
            get { return open; }
            set { open = value; }
        }

        /// <summary>
        /// l
        /// </summary>
        public int High
        {
            get { return high; }
            set { high = value; }
        }

        /// <summary>
        /// l
        /// </summary>
        public int Low
        {
            get { return low; }
            set { low = value; }
        }

        /// <summary>
        /// Il
        /// </summary>
        public int Close
        {
            get { return close; }
            set { close = value; }
        }

        /// <summary>
        /// o
        /// </summary>
        public double Volume
        {
            get { return volume; }
            set { volume = value; }
        }

        /// <summary>
        /// l
        /// </summary>
        public double Split
        {
            get { return split; }
            set { split = value; }
        }
    }

    /// <summary>
    /// HTTPɂt@C̃_E[hƈkꂽt@C̓WJsB
    /// </summary>
    public class DownloadUtil
    {
        string url;
        string referer;
        WebProxy proxy;
        CookieContainer cookies;
        DateTime ifModifiedSince;
        Stream responseStream;

        /// <summary>
        /// RXgN^B
        /// </summary>
        public DownloadUtil()
        {
            cookies = new CookieContainer();
        }

        /// <summary>
        /// URLݒ肷B
        /// </summary>
        public string Url
        {
            set { url = value; }
        }

        /// <summary>
        /// refererݒ肷B
        /// </summary>
        public string Referer
        {
            set { referer = value; }
        }

        /// <summary>
        /// vLVݒ肷B
        /// </summary>
        /// <param name="address"></param>
        /// <param name="port"></param>
        public void SetProxy(string address, int port)
        {
            proxy = new WebProxy(address, port);
        }

        /// <summary>
        /// IfModifiedSinceݒ肷B
        /// </summary>
        public DateTime IfModifiedSince
        {
            set { ifModifiedSince = value; }
        }

        /// <summary>
        /// ݒɊÂHTTPNGXg𔭍sAX|Xǂނ߂StreamԂB
        /// </summary>
        /// <returns>X|Xǂނ߂Stream</returns>
        public Stream GetResponse()
        {
            if (url == null)
                return null;
            HttpWebRequest request = (HttpWebRequest)WebRequest.Create(url);
            if (referer != null)
                request.Referer = referer;
            if (proxy != null)
                request.Proxy = proxy;
            request.CookieContainer = cookies;
            if (ifModifiedSince != DateTime.MinValue)
                request.IfModifiedSince = ifModifiedSince;
            HttpWebResponse response;
            try
            {
                response = (HttpWebResponse)request.GetResponse();
            }
            catch (WebException e)
            {
                if (e.Status == WebExceptionStatus.ProtocolError)
                    return null;
                throw;
            }
            if (response.StatusCode != HttpStatusCode.OK)
            {
                response.Close();
                return null;
            }
            responseStream = response.GetResponseStream();
            return responseStream;
        }

        /// <summary>
        /// ݒɊÂLHAňkꂽt@CHTTPŃ_E[hēWJB
        /// </summary>
        /// <returns>WJt@Cǂނ߂Stream</returns>
        public Stream DownloadAndExtract()
        {
            if (GetResponse() == null)
                return null;
            CleanUpTmpDir();
            string name = url.Substring(url.LastIndexOf('/') + 1);
            string archive = Path.Combine(Global.DirTmp, name);
            using (FileStream fs = File.Create(archive))
            {
                int n;
                byte[] buf = new byte[4096];
                while ((n = responseStream.Read(buf, 0, buf.Length)) > 0)
                    fs.Write(buf, 0, n);
                fs.Close();
            }
            IExtracter e = new LhaExtracter();
            e.Extract(archive, Global.DirTmp);
            // i[Ăt@C̖Oꍇ̂ŁA
            // A[JCut@CĂglob patternŒނグB
            File.Delete(archive);
            string[] files = Directory.GetFiles(Global.DirTmp, "*");
            if (files == null || files.Length != 1)
                throw new ApplicationException("f[^t@C̓WJɎs܂B");
            return File.Open(files[0], FileMode.Open);
        }

        private void CleanUpTmpDir()
        {
            if (Directory.Exists(Global.DirTmp))
                Directory.Delete(Global.DirTmp, true);
            Directory.CreateDirectory(Global.DirTmp);
        }
    }

	/// <summary>
	/// ̍XVsۃNXłB
	/// </summary>
	abstract public class PriceDataUpdator
	{
        private DownloadUtil downloadUtil;
        private DateTime date; // ̓t
        private DateTime endDate; // ׂŌ̓tB
        
        private int numRecords; // ̃R[h
        private int totalRecords; // ׂ̃R[h
        private int doneRecords; // R[h
        private int numDays; // 
        private int totalDays; // ׂ
        private long startTime; // ̊Jn
        private long timeToDownload; // t@C̃_E[hɂ
        private long downloadStart; // _E[h̊Jn

        /// <summary>
        /// XVɎgpDownloadUtilNX̃CX^X擾B
        /// </summary>
        protected DownloadUtil DownloadUtil
        {
            get { return downloadUtil; }
        }

        /// <summary>
        /// ̃R[h擾B
        /// </summary>
        protected int NumRecords
        {
            get { return numRecords; }
        }

        /// <summary>
        /// ׂ̃R[h擾܂͐ݒ肷B
        /// </summary>
        protected int TotalRecords
        {
            get { return totalRecords; }
            set { totalRecords = value; }
        }

        /// <summary>
        /// R[hCNgB
        /// </summary>
        protected void IncrementRecords()
        {
            numRecords++;
            doneRecords++;
        }

        /// <summary>
        /// CNgB
        /// </summary>
        protected void IncrementDays()
        {
            numDays++;
            numRecords = 0;
        }

        /// <summary>
        /// ׂfNgB
        /// </summary>
        protected void DecrementTotalDays()
        {
            totalDays--;
        }

        /// <summary>
        /// ̓t擾܂͐ݒ肷B
        /// </summary>
        protected DateTime Date
        {
            get { return date; }
            set { date = value; }
        }
        
        /// <summary>
        ///  ̊JnɕKvȐݒB
        /// <param name="begin">Jnt</param>
        /// <param name="end">It</param>
        /// </summary>
        protected void Start(DateTime begin, DateTime end)
        {
            for (DateTime d = begin; d <= end; d = d.AddDays(1))
                if (Utils.IsMarketOpen(d))
                {
                    if (date == DateTime.MinValue)
                        date = d; // ŏ̎sꂪJĂɐݒ肷B
                    totalDays++;
                }
            if (date == DateTime.MinValue) // begin > end̏ꍇB
                date = begin;
            endDate = end;
            startTime = DateTime.Now.Ticks;
        }

        /// <summary>
        /// tIɒBĂ邩H
        /// </summary>
        /// <returns></returns>
        protected bool ShouldContinue()
        {
            return date <= endDate;
        }

        /// <summary>
        /// t̎sꂪJĂɐi߂B
        /// </summary>
        protected void NextDate()
        {
            do
            {
                date = date.AddDays(1);
            }
            while (!Utils.IsMarketOpen(date) && date < endDate);
        }

        /// <summary>
        /// t@C̃_E[h̊JnL^B
        /// </summary>
        protected void StartDownload()
        {
            downloadStart = DateTime.Now.Ticks;
        }

        /// <summary>
        /// t@C̃_E[hɗvԂL^B
        /// </summary>
        protected void EndDownload()
        {
            timeToDownload += DateTime.Now.Ticks - downloadStart;
        }

        /// <summary>
        /// w肳ꂽf[^\[XɑΉۃNX̃CX^XԂB
        /// </summary>
        /// <param name="option">IvVw肷B</param>
        /// <returns></returns>
        static public PriceDataUpdator Create(Option option)
        {
            PriceDataUpdator r;
            switch (option.PriceDataSource)
            {
                case PriceDataSource.KabukaJoho:
                    r = new KabukaJohoUpdator();
                    break;
#if MUJINZOU
                case PriceDataSource.Mujinzou:
                    r = new MujinzouUpdator();
                    break;
#endif
                case PriceDataSource.KdbCom:
                    r = new KdbComUpdator();
                    break;
                case PriceDataSource.YahooFinance:
                    r = new YahooFinanceUpdator();
                    break;
                default:
                    return null;
            }
            r.downloadUtil = new DownloadUtil();
            if (option.UseProxy)
                r.downloadUtil.SetProxy(option.ProxyAddress, option.ProxyPort);
            return r;
        }

        /// <summary>
        /// f[^\[X̖Öꗗ擾B
        /// </summary>
        static public string[] DataSourceNames
        {
            get
            {
                return new string[] {
                    "",
#if MUJINZOU
                    "s",
#endif
                    "f[^_E[hTCg",
                    "Yahoo!t@CiX"
                };
            }
        }

        /// <summary>
        /// f[^\[X̐擾B
        /// </summary>
        /// <param name="dataSource">f[^\[Xw肷</param>
        /// <returns>f[^\[X̐</returns>
        static public string GetDescription(PriceDataSource dataSource)
        {
            switch (dataSource)
            {
                case PriceDataSource.KabukaJoho:
                    return "2000N2005N̓؂Jasdaq̃f[^擾ł܂A" +
                        "}U[Y̖JasdaqƂēo^܂B" +
                        "2006N͓؁A؂Jasdaq̃f[^擾ł܂B";
#if MUJINZOU
                case PriceDataSource.Mujinzou:
                    return "1996N̓؁A؁A؂Jasdaq̃f[^擾ł܂B" +
                        "2008N1ōXVIĂ܂B";
#endif
                case PriceDataSource.KdbCom:
                    return "NO̓؁A؁A؂Jasdaq̃f[^擾ł܂B";
                case PriceDataSource.YahooFinance:
                    return "1991N̓؁A؁A؂Jasdaq̃f[^擾ł܂B" +
                        "dꂵĂꍇɂ͗Ds̃f[^擾ł܂B" +
                        "ɎԂ܂B";
            }
            return "";
        }

        /// <summary>
        /// f[^݂ŏ̓t擾B
        /// </summary>
        /// <param name="dataSource">f[^\[Xw肷</param>
        /// <returns>t</returns>
        static public DateTime GetDataSince(PriceDataSource dataSource)
        {
            switch (dataSource)
            {
                case PriceDataSource.KabukaJoho:
                    return KabukaJohoUpdator.DataSince;
#if MUJINZOU
                case PriceDataSource.Mujinzou:
                    return MujinzouUpdator.DataSince;
#endif
                case PriceDataSource.KdbCom:
                    return KdbComUpdator.DataSince;
                case PriceDataSource.YahooFinance:
                    return YahooFinanceUpdator.DataSince;
            }
            return DateTime.MinValue;
        }

        /// <summary>
		/// f[^XVB
		/// </summary>
		/// <param name="worker">BackgroundWorker</param>
        /// <param name="e">DoWorkCxg̈</param>
		abstract public void Update(BackgroundWorker worker, DoWorkEventArgs e);

        /// <summary>
        /// f[^XVB
        /// </summary>
        /// <param name="worker">BackgroundWroker</param>
        /// <param name="e">DoWrokCxg̈</param>
        protected void UpdateSplitData(BackgroundWorker worker, DoWorkEventArgs e)
        {
            DateTime maxDate = SplitTable.MaxDate();
            if (maxDate > DateTime.MinValue &&
                    SplitTable.GetRecords(2814).Length == 0)
            {
                // f[^ԈႦĂ̂ō蒼B
                SplitTable.Delete();
                maxDate = DateTime.MinValue;
            }
            ArrayList updated = ReadOmegaChartSplitData(worker, e, maxDate);
            if (updated == null)
                return;
            for (int i = 0; i < updated.Count; i++)
            {
                if (worker.CancellationPending)
                {
                    e.Cancel = true;
                    return;
                }
                if (updated.Count > 100 && (i % 100 == 0 || i == updated.Count))
                    // ŏ͎Ԃ̂Ōo߂\B
                    worker.ReportProgress(i * 100 / updated.Count, "f[^");
                PriceData tmp = (PriceData)updated[i];
                SplitTable.Add(tmp.Code, tmp.Date, tmp.Split);
            }
        }

        /// <summary>
        /// OmegaCharť`index.txtǂݍōXVꂽf[^ԂB
        /// </summary>
        /// <param name="worker">BackgroundWorker</param>
        /// <param name="e">DoWorkCxg̈</param>
        /// <param name="maxDate">̃f[^̍ŐV̓t</param>
        private ArrayList ReadOmegaChartSplitData(BackgroundWorker worker, DoWorkEventArgs e, DateTime maxDate)
		{
			downloadUtil.IfModifiedSince = maxDate;
            downloadUtil.Url = "http://protra.sourceforge.jp/data/index.txt.lzh";
            Stream stream = downloadUtil.DownloadAndExtract();
            if (stream == null)
                return null;
            if (worker.CancellationPending)
            {
                e.Cancel = true;
                stream.Close();
                return null;
            }
            StreamReader reader = new StreamReader(stream, Encoding.GetEncoding("shift_jis"));
            ArrayList updated = new ArrayList();
            string line;
            while ((line = reader.ReadLine()) != null)
            {
                int code = 0;
                foreach (string entry in line.Split(','))
                {
                    if (code == 0)
                    {
                        try
                        {
                            code = int.Parse(entry);
                        }
                        catch
                        {
                            break;
                        }
                        continue;
                    }
                    if (!entry.StartsWith("S:"))
                        continue;
                    int y = int.Parse(entry.Substring(2, 4));
                    int m = int.Parse(entry.Substring(6, 2));
                    int d = int.Parse(entry.Substring(8, 2));
                    DateTime date = new DateTime(y, m, d);
                    if (date <= maxDate)
                        continue;
                    PriceData r = new PriceData();
                    r.Code = code;
                    r.Date = date;
                    r.Split = double.Parse(entry.Substring(11));
                    updated.Add(r);
                }
            }
            reader.Close();
            return updated;
		}

        int lastProgress = -1; // ŏ͓t̏񂾂\B

        /// <summary>
        /// c̏vԂƐivZĕ\B
        /// </summary>
        /// <param name="worker">BackgroundWorker</param>
        /// <param name="e">DoWorkCxg̈</param>
		protected void UpdateProgress(BackgroundWorker worker, DoWorkEventArgs e)
		{
            int progress = Progress();
            if (lastProgress == progress)
                return;
            lastProgress = progress;
            string message = String.Format("{0:d} ({1}/{2}) ", date, numDays + 1, totalDays) + LeftTime();
            worker.ReportProgress(progress, message);
		}

        /// <summary>
        /// ivZĕԂB
        /// </summary>
        /// <returns>%ł킵i</returns>
        public int Progress()
        {
            return (totalRecords == 0) ? 0 : numRecords * 100 / totalRecords;
        }

        /// <summary>
        /// c莞Ԃ𕶎\ŕԂB
        /// </summary>
        /// <returns>ŕ\c莞</returns>
        private string LeftTime()
        {
            int leftTime = CalcLeftTime();
            if (leftTime == 0) // n܂O0ɂȂB
                return "";
            string s = "c";
            if (leftTime <= 50)
                s += " " + leftTime + " b";
            else
            {
                if (leftTime >= 3600)
                {
                    s += " " + (leftTime / 3600) + " ";
                    leftTime %= 3600;
                }
                if (leftTime > 0)
                    s += " " + ((leftTime + 59) / 60) + " "; // ؂グ
            }
            return s;
        }

        /// <summary>
        /// c莞ԂvZB
        /// </summary>
        /// <returns>bł킵c莞</returns>
        private int CalcLeftTime()
        {
            long leftTime = 0;
            long timeToRecords = (DateTime.Now.Ticks - startTime) - timeToDownload;
            int downloaded = (numRecords > 0) ? numDays + 1 : numDays;
            if (downloaded > 0)
                leftTime = timeToDownload / downloaded * (totalDays - downloaded);
            if (doneRecords > 0)
            {
                int leftRecords = totalRecords - numRecords + (totalDays - numDays - 1) * totalRecords;
                leftTime += timeToRecords / doneRecords * leftRecords;
            }
            return (int)(leftTime / 10000000);
        }
	}
}
