﻿// Copyright (C) 2003, 2005 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2008, 2013 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: VariableNode.cs 455 2013-06-08 11:52:33Z panacoran $

using System.Collections.Generic;

namespace Protra.Lib.Lang
{
    /// <summary>
    /// 変数のノードを表すクラス。
    /// </summary>
    public class VariableNode : Node
    {
        /// <summary>
        /// スタックフレーム中のインデックス。
        /// </summary>
        private int _index;

        /// <summary>
        /// 構文解析する。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.ParseException">
        /// 構文解析中にエラーが発生したときにthrowされる。
        /// </exception>
        /// <return>
        /// 解析した結果のノード。
        /// </return>
        public override Node Parse()
        {
            Token = Scanner.Token;
            Scanner.Scan();
            if (Token.Type == TokenType.LocalVariable)
            {
                var lvt = LvtStack.Peek();
                if (lvt.ContainsKey(Token.Value))
                    _index = lvt[Token.Value];
                else
                    throw new ParseException("undefined local variable '" + Token.Value + "'", Token);
            }
            return this;
        }

        /// <summary>
        /// プログラムを実行する。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされる。
        /// </exception>
        /// <param name="resource">リソース</param>
        /// <param name="at">int型@作用素の値</param>
        /// <param name="ats">string型@作用素の値</param>
        /// <returns>実行結果</returns>
        public override Value Execute(Resource resource, int at, string ats)
        {
            if (Token.Type != TokenType.GlobalVariable)
                return resource.Stack.Peek()[_index];
            try
            {
                return resource.GlobalVariableTable[Token.Value];
            }
            catch (KeyNotFoundException)
            {
                return null;
            }
        }

        /// <summary>
        /// 代入を実行する。
        /// </summary>
        /// <param name="resource">リソース</param>
        /// <param name="val">代入する値</param>
        /// <returns>実行結果</returns>
        public Value Assign(Resource resource, Value val)
        {
            if (Token.Type == TokenType.GlobalVariable)
                resource.GlobalVariableTable[Token.Value] = val;
            else
                resource.Stack.Peek()[_index] = val;
            return val;
        }
    }
}