#pragma once

#include <C2/lm/vector3.h>
#include <C2/lm/quat4.h>

#include "ManipulatorBase.h"

#include "GlInclude.h"


namespace lib_gl
{


class CameraManipulator : public Manipulator
{
public:
	virtual void SetGLModelview(void);

	void MultGLLookAt(void);

	void LookAt( const lm::vec3f& i_eye , const lm::vec3f& i_view , const lm::vec3f& i_up );

	//! xNg̐ݒ. ̂̕w肷, ƒ悤ɕ␳.
	void SetUpAngle( const lm::vec3f& i_up );

	//! _璍_ւ̋擾
	float GetDistanceToLookPos(void) const;

	//! __ƂȂxNg擾
	lm::vec3f GetViewToEye(void) const;

	//! __ƂȂxNg擾
	lm::vec3f GetEyeToView(void) const;

	lm::vec3f GetFront(void) const;
	lm::vec3f GetUp(void) const;
	lm::vec3f GetBinormal(void) const;

	//! ^u. _𒆐SɎ_].
	//! @param alpha - ʊp̉]px(EnȂ琳ŏ猩Ĕv)
	//! @param beta  - Vp̉]px(ŃJE猩Ĕvj
	void Tumble( float alpha , float beta );

	//! _𒆐SɎ_].
	//! @param alpha - ʊp̉]px(EnȂ琳ŏ猩Ĕv)
	//! @param beta  - Vp̉]px(ŃJE猩Ĕvj
	void Rotate( float alpha , float beta );

	//! gbN. Ԃɑ΂ĕsɈړ.
	//! @param right - Eւ̈ړ
	//! @param up    - ւ̈ړ
	void Track( float right , float up );
	void Track( float right , float up , float front );

	//! h[. _ɋ߂Â.
	//! @param dz           - _ɋ߂Â.
	//! @param min_distance - _ɍł߂ÂƂ̋.
	void Dolly( float dz , float min_distance = 0.0f );

	//! ݂̎_, _ԋ̔䗦Ńh[
	//! @param ratio - n_, _Ԃ̋ړOratio{ɂȂʒuɃZbg.
	void DollyByRatio( float ratio );

	//! _, _𕽍sړ
	void Translate( const lm::vec3f& v );
	void Translate( float x , float y , float z );

	//! _wʒuɂȂ悤ɕsړ
	void TranslateViewPosTo( const lm::vec3f& pos );

	//! 2̎_ʐ`Ԃ_߂
	//! @param ma    - _1
	//! @param mb    - _2
	//! @param ratio - ԌW. 0maƈv, 1mbƈv.
	void SetSlerp(const CameraManipulator& ma, const CameraManipulator& mb, float ratio);

	//! ̋ɍWl擾
	//! @param alpha - ʊp̉]px(EnȂ琳ŏ猩Ĕv)
	//! @param beta  - Vp̉]px(ŃJE猩Ĕvj
	void GetViewAngles(float& alpha, float& beta) const;


public:
	lm::vec3f  m_EyePos;   //!< _(ϑ҂̈ʒu)
	lm::vec3f  m_ViewPos;  //!< _(ϑ҂߂Ă_̈ʒu)
	lm::vec3f  m_Up;       //!< VxNg
};



inline
void CameraManipulator::SetGLModelview(void)
{
	glPushAttrib(GL_TRANSFORM_BIT);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	MultGLLookAt();

	glPopAttrib();
}

//! JgsLookAts
inline
void CameraManipulator::MultGLLookAt(void)
{
	// _
	double px = static_cast<double>( m_EyePos.x );
	double py = static_cast<double>( m_EyePos.y );
	double pz = static_cast<double>( m_EyePos.z );
	// _
	double vx = static_cast<double>( m_ViewPos.x );
	double vy = static_cast<double>( m_ViewPos.y );
	double vz = static_cast<double>( m_ViewPos.z );
	// 
	double ox = static_cast<double>( m_Up.x );
	double oy = static_cast<double>( m_Up.y );
	double oz = static_cast<double>( m_Up.z );

	gluLookAt( px , py , pz , vx , vy , vz , ox , oy , oz );
}


inline
void CameraManipulator::SetUpAngle( const lm::vec3f& i_up )
{
	lm::vec3f local_eye = GetViewToEye();
	lm::vec3f binormal = cross( local_eye , i_up );
	m_Up = cross( binormal , local_eye ).get_normalize();
}

inline
void CameraManipulator::LookAt( const lm::vec3f& i_eye , const lm::vec3f& i_view , const lm::vec3f& i_up )
{
	m_EyePos = i_eye;
	m_ViewPos = i_view;
	SetUpAngle( i_up );
}

//! _璍_ւ̋擾
inline
float CameraManipulator::GetDistanceToLookPos(void) const
{
	return ( m_EyePos - m_ViewPos ).length();
}

//! __ƂȂxNg擾
inline
lm::vec3f CameraManipulator::GetViewToEye(void) const
{
	return ( m_EyePos - m_ViewPos );
}

//! __ƂȂxNg擾
inline
lm::vec3f CameraManipulator::GetEyeToView(void) const
{
	return ( m_ViewPos - m_EyePos );
}

inline
lm::vec3f CameraManipulator::GetFront(void) const
{
	return GetEyeToView().get_normalize();
}

inline
lm::vec3f CameraManipulator::GetUp(void) const
{
	return m_Up.get_normalize();
}

inline
lm::vec3f CameraManipulator::GetBinormal(void) const
{
	return cross( GetEyeToView() , m_Up ).get_normalize();
}


//! ^u. _𒆐SɎ_].
//! @param alpha - ʊp̉]px(EnȂ琳ŏ猩Ĕv)
//! @param beta  - Vp̉]px(ŃJE猩Ĕvj
inline
void CameraManipulator::Tumble( float alpha , float beta )
{
	lm::vec3f local_eye = GetViewToEye();
	local_eye.rotate_y( alpha );
	m_Up.rotate_y( alpha );

	lm::vec3f binormal = cross( m_Up , local_eye ).get_normalize();
	local_eye.rotate( beta , binormal );
	m_Up.rotate( beta , binormal );

	m_EyePos = m_ViewPos + local_eye;
}

//! _𒆐SɎ].
//! @param alpha - ʊp̉]px(EnȂ琳ŏ猩Ĕv)
//! @param beta  - Vp̉]px(ŃJE猩Ĕvj
inline
void CameraManipulator::Rotate( float alpha , float beta )
{
	lm::vec3f local_eye = GetViewToEye();
	local_eye.rotate_y( alpha );
	m_Up.rotate_y( alpha );

	lm::vec3f binormal = cross( m_Up , local_eye ).get_normalize();
	local_eye.rotate( beta , binormal );
	m_Up.rotate( beta , binormal );

	m_ViewPos = m_EyePos - local_eye;
}

//! gbN. Ԃɑ΂ĕsɈړ.
//! @param right - Eւ̈ړ
//! @param up    - ւ̈ړ
inline
void CameraManipulator::Track( float right , float up )
{
	lm::vec3f n = GetUp();
	lm::vec3f b = GetBinormal();

	lm::vec3f move = b * right + n * up;
	m_ViewPos += move;
	m_EyePos += move;
}

inline
void CameraManipulator::Track( float right , float up , float front )
{
	lm::vec3f n = GetUp();
	lm::vec3f b = GetBinormal();
	lm::vec3f f = GetFront();

	lm::vec3f move = b * right + n * up + f * front;
	m_ViewPos += move;
	m_EyePos += move;
}

//! h[. _ɋ߂Â.
//! @param dz           - _ɋ߂Â.
//! @param min_distance - _ɍł߂ÂƂ̋.
inline
void CameraManipulator::Dolly( float dz , float min_distance )
{
	lm::vec3f local_eye = GetViewToEye();
	float distance = local_eye.length();
	distance -= dz;
	distance = (std::max)( min_distance , distance );
	local_eye.normalize();
	local_eye *= distance;

	m_EyePos = m_ViewPos + local_eye;
}

//! ݂̎_, _ԋ̔䗦Ńh[
//! @param ratio - n_, _Ԃ̋ړOratio{ɂȂʒuɃZbg.
inline
void CameraManipulator::DollyByRatio( float ratio )
{
	m_EyePos = m_ViewPos + GetViewToEye() * ratio;
}

//! _, _𕽍sړ
inline
void CameraManipulator::Translate( const lm::vec3f& v )
{
	Translate(v.x, v.y, v.z);
}

inline
void CameraManipulator::Translate( float x , float y , float z )
{
	lm::vec3f vmove(x, y, z);
	m_EyePos  += vmove;
	m_ViewPos += vmove;
}

inline
void CameraManipulator::TranslateViewPosTo( const lm::vec3f& pos )
{
	Translate(pos - m_ViewPos);
}


inline
void CameraManipulator::SetSlerp(const CameraManipulator& ma, const CameraManipulator& mb, float ratio)
{
	lm::vec3f ax = ma.GetFront();
	lm::vec3f ay = ma.GetUp();
	lm::vec3f az = ma.GetBinormal();
	lm::vec3f bx = mb.GetFront();
	lm::vec3f by = mb.GetUp();
	lm::vec3f bz = mb.GetBinormal();

	lm::quat4f qs;
	qs.set_rotate_from_coord(ax, ay, az);

	lm::quat4f qe;
	qe.set_rotate_from_coord(bx, by, bz);

	lm::quat4f::minimize_rotation(qs, qe);

	lm::quat4f qi = lm::quat4f::interpolate(qs, qe, ratio);

	float r0 = (1.0f - ratio);
	float r1 = ratio;
	m_EyePos = ma.m_EyePos * r0 + mb.m_EyePos * r1;

	float len = ma.GetDistanceToLookPos() * r0 + mb.GetDistanceToLookPos() * r1;
	m_ViewPos = m_EyePos + qi.get_ex() * len;

	SetUpAngle( qi.get_ey() );
}

inline
void CameraManipulator::GetViewAngles(float& alpha, float& beta) const
{
	alpha = 0.0f;
	beta = 0.0f;

	lm::vec3f d = GetViewToEye();
	float len = d.length();
	if (len == 0.0f)
		return;

	d /= len;

	beta = acos(d.y);
	alpha = atan2(d.z, d.x);

	//bool is_reverse = (m_Up.y < 0.0f);
	//if (is_reverse)
	//{
	//	if (alpha > 0.0f)
	//		alpha -= M_PI;
	//	else
	//		alpha += M_PI;

	//	beta = -beta;
	//}
}


}
