#pragma once

#include "tuple3.h"


namespace lm
{

template<typename T>
class vector3 : public tuple3<T>
{
public:
	vector3(void)                                    : tuple3<T>( T(0) , T(0) , T(0) ) {}
	vector3( const T& _x, const T& _y, const T& _z ) : tuple3<T>( _x   , _y   , _z   ) {}
	vector3( const vector3<T>& _v )                  : tuple3<T>( _v.x , _v.y , _v.z ) {}
	vector3( const T* _v )                           : tuple3<T>( _v ) {}


	template<typename U>
	operator vector3<U>(void) const
	{
		return vector3<U>( static_cast<U>(x) , static_cast<U>(y) , static_cast<U>(z) );
	}


	void set_zero(void);
	static vector3<T> get_zero(void);

	void add( const vector3<T>& _v );
	void add( const T* _v );
	void add( const T& _x, const T& _y, const T& _z );
	vector3<T>& operator+=( const vector3<T>& _v );

	void sub( const vector3<T>& _v );
	void sub( const T* _v );
	void sub( const T& _x, const T& _y, const T& _z );
	vector3<T>& operator-=( const vector3<T>& _v );

	vector3<T>& operator*=( const T& val );
	vector3<T>& operator/=( const T& val );

	vector3<T> operator-(void) const;


	bool is_zero(void) const;

	void normalize(void);
	vector3<T> get_normalize(void) const;

	T length(void) const;
	T square_length(void) const;


	void rotate_x( const T& angle );
	void rotate_y( const T& angle );
	void rotate_z( const T& angle );
	void rotate( const T& angle, const vector3<T>& axis );

	vector3<T> get_rotate_x( const T& angle ) const;
	vector3<T> get_rotate_y( const T& angle ) const;
	vector3<T> get_rotate_z( const T& angle ) const;
	vector3<T> get_rotate( const T& angle, const vector3<T>& axis ) const;
};


typedef vector3<double> vector3d;
typedef vector3<float>  vector3f;
typedef vector3<double> vec3d;
typedef vector3<float>  vec3f;


// global method 

template<typename T> inline
T dot(const vector3<T>& _a, const vector3<T>& _b)
{
	return _a.x * _b.x + _a.y * _b.y + _a.z * _b.z ;
}

template<typename T> inline
vector3<T> cross(const vector3<T>& _a, const vector3<T>& _b)
{
	vector3<T> ret;
	ret.x = _a.y * _b.z - _a.z * _b.y ;
	ret.y = _a.z * _b.x - _a.x * _b.z ;
	ret.z = _a.x * _b.y - _a.y * _b.x ;
	return ret;
}

//! XJ3d ( = { ( a x b )Ec } = { aE( b x c ) } )
template<typename T> inline
float scholar_triple(const vector3<T>& _a, const vector3<T>& _b, const vector3<T>& _c)
{
	return dot( cross( _a , _b ) , c );
}


// implements

template<typename T> inline
void vector3<T>::set_zero(void)
{
	this->set( T(0) , T(0) , T(0) );
}

template<typename T> inline
vector3<T> vector3<T>::get_zero(void)
{
	return vector3<T>( T(0) , T(0) , T(0) );
}


template<typename T> inline
void vector3<T>::add(const vector3<T>& _v)
{
	x += _v.x ;  y += _v.y ;  z += _v.z ;
}

template<typename T> inline
void vector3<T>::add(const T* _v)
{
	x += _v[0] ;  y += _v[1] ;  z += _v[2] ;
}

template<typename T> inline
void vector3<T>::add(const T& _x, const T& _y, const T& _z)
{
	x += _x ;  y += _y ;  z += _z ;
}

template<typename T> inline
vector3<T>& vector3<T>::operator+=(const vector3<T>& _v)
{
	x += _v.x ;  y += _v.y ;  z += _v.z ;
	return *this;
}


template<typename T> inline
void vector3<T>::sub(const vector3<T>& _v)
{
	x -= _v.x ;  y -= _v.y ;  z -= _v.z ;
}

template<typename T> inline
void vector3<T>::sub(const T* _v)
{
	x -= _v[0] ;  y -= _v[1] ;  z -= _v[2] ;
}

template<typename T> inline
void vector3<T>::sub(const T& _x, const T& _y, const T& _z)
{
	x -= _x ;  y -= _y ;  z -= _z ;
}

template<typename T> inline
vector3<T>& vector3<T>::operator-=(const vector3<T>& _v)
{
	x -= _v.x ;  y -= _v.y ;  z -= _v.z ;
	return *this;
}


template<typename T> inline
vector3<T>& vector3<T>::operator*=(const T& val)
{
	x *= val ;  y *= val ;  z *= val ;
	return *this;
}


template<typename T> inline
vector3<T>& vector3<T>::operator/=(const T& val)
{
	x /= val ;  y /= val ;  z /= val ;
	return *this;
}


template<typename T> inline
vector3<T> vector3<T>::operator-(void) const
{
	return vector3<T>( -x , -y , -z );
}


template<typename T> inline
bool vector3<T>::is_zero(void) const
{
	T zero = static_cast<T>(0);
	if( x == zero && y == zero && z == zero ) return true;
	else                                      return false;
}

template<typename T> inline
void vector3<T>::normalize(void)
{
	if( is_zero() )return;
	(*this) /= length();
}

template<typename T> inline
vector3<T> vector3<T>::get_normalize(void) const
{
	vector3<T> tmp = *this;
	tmp.normalize();
	return tmp;
}

template<typename T> inline
T vector3<T>::length(void) const
{
	return sqrt( square_length() );
}

template<typename T> inline
T vector3<T>::square_length(void) const
{
	return x * x + y * y + z * z ;
}


template<typename T> inline
void vector3<T>::rotate_x(const T& angle)
{
	T c = cos(angle)  ,  s = sin(angle);
	T ty = y * c - z * s ;
	T tz = y * s + z * c ;
	y = ty ;   z = tz ;
}

template<typename T> inline
void vector3<T>::rotate_y(const T& angle)
{
	T c = cos(angle)  ,  s = sin(angle);
	T tz = z * c - x * s ;
	T tx = z * s + x * c ;
	z = tz ;   x = tx ;
}

template<typename T> inline
void vector3<T>::rotate_z(const T& angle)
{
	T c = cos(angle)  ,  s = sin(angle);
	T tx = x * c - y * s ;
	T ty = x * s + y * c ;
	x = tx ;   y = ty ;
}

template<typename T> inline
void vector3<T>::rotate(const T& angle, const vector3<T>& axis)
{
	T c = cos(angle);  T s = sin(angle);
	T mc = T(1) - c ;

	const T& ax = axis.x ;  const T& ay = axis.y ;  const T& az = axis.z ;
	T tx = x ;  T ty = y ;  T tz = z ; 

	x = ( ax * ax * mc + c      ) * tx + ( ax * ay * mc - az * s ) * ty + ( ax * az * mc + ay * s ) * tz ;
	y = ( ay * ax * mc + az * s ) * tx + ( ay * ay * mc + c      ) * ty + ( ay * az * mc - ax * s ) * tz ;
	z = ( az * ax * mc - ay * s ) * tx + ( az * ay * mc + ax * s ) * ty + ( az * az * mc + c      ) * tz ;
}

template<typename T> inline
vector3<T> vector3<T>::get_rotate_x( const T& angle ) const
{
	vector3<T> t = *this;
	t.rotate_x( angle );
	return t;
}

template<typename T> inline
vector3<T> vector3<T>::get_rotate_y( const T& angle ) const
{
	vector3<T> t = *this;
	t.rotate_y( angle );
	return t;
}

template<typename T> inline
vector3<T> vector3<T>::get_rotate_z( const T& angle ) const
{
	vector3<T> t = *this;
	t.rotate_z( angle );
	return t;
}

template<typename T> inline
vector3<T> vector3<T>::get_rotate( const T& angle, const vector3<T>& axis ) const
{
	vector3<T> t = *this;
	t.rotate( angle , axis );
	return t;
}


// friend operations

template<typename T> inline
vector3<T> operator+(const vector3<T>& _a, const vector3<T>& _b)
{
	vector3<T> t = _a ;  t += _b ;  return t ;
}

template<typename T> inline
vector3<T> operator-(const vector3<T>& _a, const vector3<T>& _b)
{
	vector3<T> t = _a ;  t -= _b ;  return t ;
}

template<typename T> inline
vector3<T> operator*(const vector3<T>& _v, const T& _t)
{
	vector3<T> t = _v ;  t *= _t ;  return t ;
}

template<typename T> inline
vector3<T> operator*(const T& _t, const vector3<T>& _v)
{
	vector3<T> t = _v ;  t *= _t ;  return t ;
}

template<typename T> inline
vector3<T> operator/(const vector3<T>& _v, const T& _t)
{
	vector3<T> t = _v ;  t /= _t ;  return t ;
}


}
