
module Event_Compiler
  
  # イベントエレメント用ビジター
  class Event_Element_Visitor
    
    include Element_Visitor
    include Event_Compiler_Util
    
    # 初期化
    def initialize(map_data,event,page)
      @map_data = map_data
      @event = event
      @page = page
      @event_list = [] # イベントリスト(RPG::EventCommand)
      @event_command = nil
    end
    
    # イベントコマンド作成
    def create_event_command(code = 0, indent = 0, parameters = [])
      return RPG::EventCommand.new(code, indent, parameters)
    end
    # イベントコマンド追加
    def add_event_command(code = 0, indent = 0, parameters = [])
      @event_list.push create_event_command(code, indent, parameters)
    end
    # イベントコマンド取得
    def get_event_command(code = 0, indent = 0)
      @event_list.reverse_each do |event|
        if event.code == code and event.indent == indent
          return event
        end
      end
      return nil
    end
    
    
    # キャラクターＩＤの取得
    def get_character_id(value)
      if value == "プレイヤー"
        return -1
      elsif value == "このイベント"
        return 0
      else
        id,e = @map_data.events.find {|_id,_e| (not _e.nil?) and _e.name == value}
        if e.nil?
          Log.error("イベント[#{value}]が見つかりません。")
          return 0
        end
        return id
      end
      return 0
    end
    
    # トループＩＤの取得
    def get_troop_id(value)
      data_troops = Data_Loader.data_troops
      troop = data_troops.find {|t| (not t.nil?) and t.name == value}
      unless troop.nil?
        return troop.id
      end
      if value =~ /~[0-9]+$/
        return value.to_i
      end
      Log.error("トループ[#{value}]が見つかりません。")
      return 1
    end
    
    # Autioデータ作成
    def create_audio_parameter(value)
      values = value.split(/,|、/)
      audio= RPG::AudioFile.new()
      audio.name = values[0].to_s if values.size >= 1
      audio.volume = values[1].to_i if values.size >= 2
      audio.pitch = values[2].to_i if values.size >= 3
      return audio
    end
    
    # 子エレメント
    def child_accespt(element,i)
      return @event_list if element.child_elements.nil?
      return @event_list if element.child_elements.size - 1 < i
      return element.child_elements[i].accept(self)
    end
    
    # 次エレメント
    def next_accept(element)
      return @event_list if element.next_element.nil?
      return element.next_element.accept(self)
    end
    
    # デフォルト
    def visit_default(element)
      if element.name == ''
        add_event_command(0,element.indent,[])
      end
      return next_accept(element)
    end
    
    # 101 文章
    add_visitor_method("文章", :visit_message)
    def visit_message(element)
      # 文章作成１行目
      add_event_command(101,element.indent,[element.value])
      element.related_elements.each do |related_element|
        # 文章作成２行目以降
        add_event_command(401,related_element.indent,[related_element.value])
      end
      return next_accept(element)
    end
    
    # 102 選択肢の表示
    add_visitor_method("選択肢の表示", :visit_choices)
    def visit_choices(element)
      choices = element.value.split(/,|、/).collect {|c| c.strip}
      cancel = 0
      parameters = [choices]
      add_event_command(102,element.indent,parameters)
      element.related_elements.zip(element.child_elements) do |related_element,child_element|
        if related_element.nil? or child_element.nil?
          add_event_command(404,element.indent,[])
          break
        end
        if related_element.name =~ /キャンセルの場合/
          cancel = 5
          add_event_command(403,related_element.indent,[])
          child_element.accept(self)
        elsif related_element.name =~ /\[?(.+)\]?の場合(\*?)/
          select_value = $1.strip
          select_index = choices.index(select_value)
          if $2 == "*"
            cancel = select_index + 1
          end
          add_event_command(402,related_element.indent,[select_index,select_value])
          child_element.accept(self)
        end
      end
      parameters.push cancel
      return next_accept(element)
    end
    
    # 103 数値入力の処理
    add_visitor_method("数値入力の処理", :visit_input_number)
    def visit_input_number(element)
      variable_id = 1
      digits_max = 1
      if element.value =~ /\[(.+?)\]/
        variable_id = Variables.index($1.strip)
      end
      if element.value =~ /([0-9])桁/
        digits_max = [[$1.to_i,1].max,8].min
      end
      add_event_command(103,element.indent,[variable_id,digits_max])
      return next_accept(element)
    end
    
    # 104 文章オプション変更
    add_visitor_method("文章オプション変更", :visit_message_option)
    def visit_message_option(element)
      parameters = element.parameters
      position = ["上","中","下"].index(parameters[0])
      frame = ["表示","非表示"].index(parameters[1])
      position = 0 if position.nil?
      frame = 0 if frame.nil?
      add_event_command(104,element.indent,[position,frame])
      return next_accept(element)
    end
    
    # 105 ボタン入力の処理
    add_visitor_method("ボタン入力の処理", :visit_input_button)
    def visit_input_button(element)
      parameters = element.parameters
      if parameters.empty?
        # TODO エラー
        Log.error("ボタン入力の処理:パラメーターなし")
        return next_accept(element)
      end
      variable_name = parameters[0]
      variable_id = Variables.index(variable_name)
      if variable_id.nil?
        # TODO エラー
        Log.error("ボタン入力の処理:変数「#{variable_name}」がありません")
        return next_accept(element)
      end
      add_event_command(105,element.indent,[variable_id])
      return next_accept(element)
    end
    
    # 106 ウェイト
    add_visitor_method("ウェイト", :visit_wait)
    def visit_wait(element)
      parameters = element.parameters
      if parameters.empty?
        # TODO エラー
        Log.error("ウェイト:パラメーターなし")
        return next_accept(element)
      end
      wait = parameters[0].to_i
      add_event_command(106,element.indent,[wait])
      return next_accept(element)
    end
    
    OPERATIONS = ['==','>=','<=','>','<','!=']
    CONDITION_TYPES = [
    [0,'スイッチ',proc do |element,parameters|
      id = Event_Compiler_Util.parse_switch_id(element.value)
      val = Event_Compiler_Util.parse_switch_value(element.value[/==.*/])
      parameters.push id
      parameters.push val
    end
    ],
    [1,'変数', proc do |element,parameters|
      values = element.value.split(/(==|>=|<=|>|<|!=)/).collect {|v| v.strip}
      id = Event_Compiler_Util.parse_variable_id(values[0])
      ope = OPERATIONS.index(values[1])
      val = 0
      type = 0
      if values[2] =~ /^[0-9]+$/
        val = values[2].to_i
        type = 0
      elsif values[2] =~ /変数/
        val = Event_Compiler_Util.parse_variable_id(values[2])
        type = 1
      end
      parameters.push id
      parameters.push type
      parameters.push val
      parameters.push ope
    end
    ],
    [2,'セルフスイッチ', proc do |element,parameters|
      values = element.value.split(/==/).collect {|v| v.strip}
      if values.size == 2
        if values[0] =~ /(A|B|C|D)/
          parameters.push $1
        end
        if values[1] == "ON"
          parameters.push 0
        elsif values[1] == "OFF"
          parameters.push 1
        end
      end
    end
    ],
    [3,'タイマー', proc do |element,parameters|
      # タイマー[0:00]以上
      values = element.value.split(/[\[\]:]/).collect {|v| v.strip}
      if values.size == 4
        min = values[1]
        sec = values[2]
        if min =~ /^[0-9]+$/ and sec =~ /^[0-9]+$/
          parameters.push(min.to_i * 60 + sec.to_i)
        else
          parameters.push 0
        end
        if values[3] =~ /以上/
          parameters.push 0
        else
          parameters.push 1
        end
      end
    end
    ],
    [4,'アクター', proc do |element,parameters|
      values = element.value.split(/\s*が\s*/,2)
      if values.size == 2
        actor_id = Event_Compiler_Util.parse_actor_id(values[0])
        case values[1]
        when /パーティにいる/
          type = 0
          value = 1
        when /名前\[(.+?)\]になっている/
          type = 1
          value = $1
        when /\[(.+?)\]を習得済み/
          type = 2
          value = Event_Compiler_Util.parse_skill_id($1)
        when /武器\[(.+?)\]を装備中/
          name = $1
          type = 3
          value = Event_Compiler_Util.parse_weapon_id(name)
        when /防具\[(.+?)\]を装備中/
          name = $1
          type = 4
          value = Event_Compiler_Util.parse_armor_id(name)
        when /\[(.+?)\]になっている/
          name = $1
          type = 5
          value = Event_Compiler_Util.parse_state_id(name)
        when /\[(.+?)\]になっていない/
          name = $1
          state_id = Event_Compiler_Util.parse_state_id(name)
          # スクリプト条件に変更
          parameters.clear
          parameters.push 12 # スクリプト
          parameters.push "not $game_actors[#{actor_id}].states.include?(#{state_id})"
          return
        else
          Log.error("条件分岐：アクター条件(#{element.value})")
          return
        end
        parameters.push actor_id
        parameters.push type
        parameters.push value
      end
    end
    ],
    [5,'エネミー', proc do |element,parameters|
    end
    ],
    [6,'キャラクター', proc do |element,parameters|
    end
    ],
    [7,'ゴールド', proc do |element,parameters|
      if element.value =~ /ゴールド\s*([0-9]+?)\s*(以上|以下)/
        parameters.push($1.to_i)
        parameters.push($2 == "以上" ? 0 : 1)
      end
    end
    ],
    [8,'アイテム', proc do |element,parameters|
      if element.value =~ /アイテム\[(.+?)\]を持っている/
        parameters.push Event_Compiler_Util.parse_item_id($1)
      end
    end
    ],
    [9,'武器', proc do |element,parameters|
      if element.value =~ /武器\[(.+?)\]を持っている/
        parameters.push Event_Compiler_Util.parse_weapon_id($1)
      end
    end
    ],
    [10,'防具', proc do |element,parameters|
      if element.value =~ /防具\[(.+?)\]を持っている/
        parameters.push Event_Compiler_Util.parse_armor_id($1)
      end
    end
    ],
    [11,'ボタン', proc do |element,parameters|
      if element.value =~ /ボタン\s*(.+?)\s*が押されている/
        parameters.push Event_Compiler_Util.parse_button_id($1)
      end
    end
    ],
    [12,'スクリプト', proc do |element,parameters|
      values = element.value.split(/：/).collect {|s|s.strip}
      if values.size == 2
        parameters.push values[1]
      end
    end
    ],
    ]
    
    # 111 条件分岐
    add_visitor_method("条件分岐", :visit_if)
    def visit_if(element)
      condition_type = -1
      condition_proc = nil
      CONDITION_TYPES.each do |ctype,cprefix,cproc|
        if element.value =~ /^#{cprefix}/
          condition_type = ctype
          condition_proc = cproc
          break
        end
      end
      if condition_proc.nil?
        Log.error("分岐条件：タイプエラー#{element.data}")
        return next_accept(element)
      end
      parameters = []
      parameters.push condition_type
      condition_proc.call(element,parameters)
      add_event_command(111,element.indent,parameters)
      element.related_elements.zip(element.child_elements) do |related_element,child_element|
        if related_element.nil? or child_element.nil?
          break
        end
        child_element.accept(self)
        related_element.accept(self)
      end
      return next_accept(element)
    end
    
    # 411 それ以外の場合
    add_visitor_method("それ以外の場合", :visit_else)
    add_visitor_method("それ以外", :visit_else)
    def visit_else(element)
      add_event_command(411,element.indent,[])
      return next_accept(element)
    end
    
    # 412 分岐終了
    add_visitor_method("分岐終了", :visit_if_end)
    def visit_if_end(element)
      add_event_command(412,element.indent,[])
      return next_accept(element)
    end
    
    # 112 ループ
    add_visitor_method("ループ", :visit_loop)
    def visit_loop(element)
      add_event_command(112,element.indent,[])
      element.child_elements.each do |child_element|
        child_element.accept(self)
      end
      element.related_elements.each do |related_element|
        related_element.accept(self)
      end
      return next_accept(element)
    end
    
    # 413 以上繰り返し
    add_visitor_method("以上繰り返し", :visit_loop_end)
    def visit_loop_end(element)
      add_event_command(413,element.indent,[])
      return next_accept(element)
    end
    
    # 113 ループの中断
    add_visitor_method("ループの中断", :visit_loop_break)
    def visit_loop_break(element)
      add_event_command(113,element.indent,[])
      return next_accept(element)
    end
    
    # 115 イベント処理の中断
    add_visitor_method("イベント処理の中断", :visit_event_break)
    def visit_event_break(element)
      add_event_command(115,element.indent,[])
      return next_accept(element)
    end
    
    # 116 イベントの一時消去
    add_visitor_method("イベントの一時消去", :visit_event_erase)
    def visit_event_erase(element)
      add_event_command(116,element.indent)
      return next_accept(element)
    end
    
    # 117 コモンイベント
    add_visitor_method("コモンイベント", :visit_common_event)
    def visit_common_event(element)
      id = parse_common_event_id(element.value)
      add_event_command(117,element.indent,[id])
      return next_accept(element)
    end
    
    # 118 ラベル
    add_visitor_method("ラベル", :visit_label)
    def visit_label(element)
      label = element.value.strip
      add_event_command(118,element.indent,[label])
      return next_accept(element)
    end
    
    # 119 ラベルジャンプ
    add_visitor_method("ラベルジャンプ", :visit_label_jump)
    def visit_label_jump(element)
      label = element.value.strip
      add_event_command(119,element.indent,[label])
      return next_accept(element)
    end
    
    # 121 スイッチの操作
    add_visitor_method("スイッチの操作", :visit_switches)
    def visit_switches(element)
      values = element.value.gsub(/\[|\]/,'').split(/=/).collect {|v| v.strip}
      if values.size == 2
        switchs = values[0].split(/\.\./).collect {|v| v.strip}
        eid = sid = Event_Compiler_Util.parse_switch_id(switchs[0])
        eid = Event_Compiler_Util.parse_switch_id(switchs[1]) if switchs.size == 2
        flag = Event_Compiler_Util.parse_switch_value(values[1])
        add_event_command(121,element.indent,[sid,eid,flag])
      end
      return next_accept(element)
    end
    
    VARIABLE_OPERATIONS = ['=','+=','-=','*=','/=','%=']
    VARIABLE_OPERATION = /(\=|\+\=|\-\=|\*\=|\/\=|\%\=)/
    VARIABLE_ITEMS = ['定数','変数','乱数','アイテム','アクター','エネミー','キャラクター','その他']
    VARIABLE_ITEM = /^(#{VARIABLE_ITEMS.join('|')})\s*\[(.+?)\]/
    VARIABLE_ACTOR_OPERATION = ['レベル','EXP','HP','SP','MaxHP','MaxSP','腕力','器用さ','素早さ','魔力','攻撃力','物理防御','魔法防御','回避修正']
    VARIABLE_ENEMY_OPERATION = ['HP','SP','MaxHP','MaxSP','腕力','器用さ','素早さ','魔力','攻撃力','物理防御','魔法防御','回避修正']
    VARIABLE_CHARACTER_OPERATION = ['X座標','Y座標','向き','画面X座標','画面Y座標','地形タグ']
    VARIABLE_ETC_OPERATION = ['マップID','パーティ人数','ゴールド','歩数','プレイ時間','タイマー','セーブ回数']
    
    # 122 変数の操作
    add_visitor_method("変数の操作", :visit_variables)
    def visit_variables(element)
      if element.value =~ VARIABLE_OPERATION
        operation = $1
        operation_id= VARIABLE_OPERATIONS.index(operation)
        values = element.value.split(/#{Regexp.escape(operation)}/).collect {|v| v.strip}
        if values.size == 2
          variables = values[0].gsub(/\[|\]/,'').split(/\.\./).collect {|v| v.strip}
          eid = sid = Event_Compiler_Util.parse_variable_id(variables[0])
          eid = Event_Compiler_Util.parse_variable_id(variables[1]) if variables.size == 2
          if values[1] =~ /^[0-9]+$/
            # 定数
            variable_item_id = 0
            variable_value_data1 = values[1].to_i
          elsif values[1] =~ VARIABLE_ITEM
            variable_item = $1
            variable_value = $2
            variable_item_id = VARIABLE_ITEMS.index(variable_item)
            variable_value_data1 = 0
            variable_value_data2 = 0
            case variable_item_id
            when 0 # 定数
              variable_value_data1 = variable_value.to_i
            when 1 # 変数
              variable_value_data1 = Event_Compiler_Util.parse_variable_id(variable_value)
            when 2 # 乱数
              tmp = variable_value.gsub(/\[|\]/,'').split(/\.\./).collect {|v| v.strip.to_i}
              if tmp.size == 2
                variable_value_data1 = tmp[0]
                variable_value_data2 = tmp[1]
              end
            when 3 # アイテム
              variable_value_data1 = Event_Compiler_Util.parse_item_id(variable_value)
            when 4 # アクター
              variable_value_data1 = Event_Compiler_Util.parse_actor_id(variable_value)
              VARIABLE_ACTOR_OPERATION.each_with_index do |name,index|
                if values[1] =~ /の\s*#{Regexp.escape(name)}$/
                  variable_value_data2 = index
                  break
                end
              end
            when 5 # エネミー
              # 戦闘用だ…
              #              variable_value_data1 = Event_Compiler_Util.parse_enemy_id(variable_value)
              #              VARIABLE_ENEMY_OPERATION.each_with_index do |name,index|
              #                if values[1] =~ /の\s*#{Regexp.escape(name)}$/
              #                  variable_value_data2 = index
              #                  break
              #                end
              #              end
            when 6 # キャラクター
              variable_value_data1 = get_character_id(variable_value)
              tmp = values[1].gsub(/ /,'')
              VARIABLE_CHARACTER_OPERATION.each_with_index do |name,index|
                if tmp =~ /の\s*#{Regexp.escape(name)}$/
                  variable_value_data2 = index
                  break
                end
              end
            when 7 # その他
              tmp = variable_value.gsub(/ /,'')
              VARIABLE_ETC_OPERATION.each_with_index do |name,index|
                if tmp =~ /#{Regexp.escape(name)}/
                  variable_value_data1 = index
                  break
                end
              end
            end
          else
            tmp = values[1].gsub(/ /,'')
            VARIABLE_ETC_OPERATION.each_with_index do |name,index|
              if tmp =~ /#{Regexp.escape(name)}/
                variable_value_data1 = index
                variable_item_id = 7
                break
              end
            end
          end
          add_event_command(122,element.indent,[sid,eid,operation_id,variable_item_id,variable_value_data1,variable_value_data2])
        end
      end
      
      return next_accept(element)
    end
    
    # 123 セルフスイッチの操作
    add_visitor_method("セルフスイッチの操作", :visit_self_switches)
    def visit_self_switches(element)
      values = element.value. split(/=/).collect {|v| v.strip}
      if values.size == 2
        switch = values[0]
        flag = Event_Compiler_Util.parse_switch_value(values[1])
      end
      add_event_command(123,element.indent,[switch,flag])
      return next_accept(element)
    end
    
    # 124 タイマーの操作
    add_visitor_method("タイマーの操作", :visit_timer)
    def visit_timer(element)
      if element.value =~ /始動/
        if element.value =~ /([0-9]+):([0-9]+)/
          sec = $1.to_i * 60 + $2.to_i
          add_event_command(124,element.indent,[0,sec])
        end
      elsif element.value =~ /停止/
        add_event_command(124,element.indent,[1])
      end
      return next_accept(element)
    end
    
    # 125 ゴールドの増減
    add_visitor_method("ゴールドの増減", :visit_gold)
    def visit_gold(element)
      if element.value =~ /\-/
        operation = 1 # 減らす
      else
        operation = 0 # 増やす
      end
      # operand_type : オペランドタイプ (0:定数 1:変数)
      if element.value =~ /変数/
        operand_type = 1
        operand = Event_Compiler_Util.parse_variable_id(element.value)
      else
        operand_type = 0
        if element.value =~ /([0-9]+)/
          operand = $1.to_i
        else
          operand = 0
        end
      end
      add_event_command(125,element.indent,[operation, operand_type, operand])
      return next_accept(element)
    end
    
    # 126 アイテムの増減
    add_visitor_method("アイテムの増減", :visit_item)
    def visit_item(element)
      if element.value =~ /\s*\[(.+?)\]\s*(\+|\-)\s*([0-9]+?)\s*/
        name = $1
        op = $2 == "+" ? 0 : 1
        num = $3
        id = Event_Compiler_Util.parse_item_id(name)
        add_event_command(126,element.indent,[id,op,0,num])
      elsif element.value =~ /\s*\[(.+?)\]\s*(\+|\-)\s*変数\s*\[(.+?)\]\s*/
        name = $1
        op = $2 == "+" ? 0 : 1
        num = $3
        vid = Event_Compiler_Util.parse_variable_id(num)
        id = Event_Compiler_Util.parse_item_id(name)
        add_event_command(126,element.indent,[id,op,1,vid])
      end
      return next_accept(element)
    end
    
    # 127 武器の増減
    add_visitor_method("武器の増減", :visit_weapon)
    def visit_weapon(element)
      return next_accept(element)
    end
    
    # 128 防具の増減
    add_visitor_method("防具の増減", :visit_armor)
    def visit_armor(element)
      return next_accept(element)
    end
    
    # 129 アクターの入れ替え
    add_visitor_method("アクターの入れ替え", :visit_actor_change)
    def visit_actor_change(element)
      parameters = element.parameters
      unless parameters.empty?
        actor_id = parse_actor_id(parameters[0])
        if parameters[0] =~ /加える/
          operation_type = 0
        else
          operation_type = 1
        end
        if parameters.size == 2 and parameters[1] =~ /初期化/
          initialize_actor = 1
        else
          initialize_actor = 0
        end
        add_event_command(129,element.indent,[actor_id,operation_type,initialize_actor])
      end
      return next_accept(element)
    end
    
    # 131 ウィンドウスキンの変更
    add_visitor_method("ウィンドウスキンの変更", :visit_windowskin_change)
    def visit_windowskin_change(element)
      return next_accept(element)
    end
    
    # 132 バトル BGM の変更
    add_visitor_method("バトル BGM の変更", :visit_battle_bgm_change)
    add_visitor_method("バトルBGMの変更", :visit_battle_bgm_change)
    def visit_battle_bgm_change(element)
      audio = create_audio_parameter(element.value)
      add_event_command(132,element.indent,[audio])
      return next_accept(element)
    end
    
    # 133 バトル終了 ME の変更
    add_visitor_method("バトル終了 ME の変更", :visit_battle_me_change)
    def visit_battle_me_change(element)
      return next_accept(element)
    end
    
    # 134 セーブ禁止の変更
    add_visitor_method("セーブ禁止の変更", :visit_save_disable)
    def visit_save_disable(element)
      if element.value =~ /許可/
        add_event_command(134,element.indent,[1])
      else
        add_event_command(134,element.indent,[0])
      end
      return next_accept(element)
    end
    
    # 135 メニュー禁止の変更
    add_visitor_method("メニュー禁止の変更", :visit_menu_disable)
    def visit_menu_disable(element)
      if element.value =~ /許可/
        add_event_command(135,element.indent,[1])
      else
        add_event_command(135,element.indent,[0])
      end
      return next_accept(element)
    end
    
    # 136 エンカウント禁止の変更
    add_visitor_method("エンカウント禁止の変更", :visit_encount_disable)
    def visit_encount_disable(element)
      return next_accept(element)
    end
    
    # 201 場所移動
    # ◆場所移動：テスト,(1,1)
    # ◆場所移動：[テスト],(1,1)
    # ◆場所移動：[テスト],(1,1),下
    # ◆場所移動：[テスト],(1,1),下,フェードなし
    # ◆場所移動：変数[マップＩＤ][Ｘ座標][Ｙ座標],下,フェードなし
    add_visitor_method("場所移動", :visit_move_to)
    def visit_move_to(element)
      type = 0
      map_id = 1
      new_x = 0
      new_y = 0
      new_direction = 0
      transition = 0
      if element.value =~ /^変数\[/
        # 変数指定
        type = 1
        args = element.value.split(/,|、/).collect {|v|v.strip}
        if args[0] =~ /\[(.+?)\]\[(.+?)\]\[(.+?)\]/
          # XXX
          map_id = Event_Compiler_Util.parse_variable_id($1)
          new_x = Event_Compiler_Util.parse_variable_id($2)
          new_y = Event_Compiler_Util.parse_variable_id($3)
        end
        if args.size >= 2
          new_direction = Event_Compiler_Util.parse_direction(args[1])
        end
        if args.size >= 3
          transition = 1
        end
      else
        # 直接指定
        type = 0
        args = element.value.split(/,|、/).collect {|v|v.strip}
        map_id = Event_Compiler_Util.parse_map_id(args[0])
        if args.size >= 3
          new_x = args[1].gsub(/\(|\)/,'').to_i
          new_y = args[2].gsub(/\(|\)/,'').to_i
        end
        if args.size >= 4
          new_direction = Event_Compiler_Util.parse_direction(args[1])
        end
        if args.size >= 5
          transition = 1
        end
      end
      add_event_command(201,element.indent,[type,map_id,new_x,new_y,new_direction,transition])
      return next_accept(element)
    end
    
    # 202 イベントの位置設定
    add_visitor_method("イベントの位置設定", :visit_event_position)
    def visit_event_position(element)
      parameters = []
      values = element.value.split(/,|、/).collect {|v|v.strip}
      parameters[0] = get_character_id(values[0])
      if values[1] =~ /^\(/
        # 直接指定
        parameters[1] = 0
        # TODO チェック
        parameters[2] = values[1][/[0-9]+/].to_i
        parameters[3] = values[2][/[0-9]+/].to_i
        parameters[4] = parse_direction(values[3])
      elsif values[1] =~ /^変数\[(.+?)\]\[(.+?)\]/
        # 変数指定
        parameters[1] = 1
        parameters[2] = parse_variable_id($1.strip)
        parameters[3] = parse_variable_id($2.strip)
        parameters[4] = parse_direction(values[2])
      elsif values[1] =~ /^\[(.+?)\]と交換/
        # イベントと交換
        parameters[1] = 2
        parameters[2] = get_character_id($1)
        parameters[3] = 0
        parameters[4] = parse_direction(values[2])
      end
      add_event_command(202,element.indent,parameters)
      if Log.debug?
        Log.debug(parameters.collect {|p| p.to_s}.join(","))
      end
      return next_accept(element)
    end
    
    # 203 マップのスクロール
    add_visitor_method("マップのスクロール", :visit_map_scroll)
    def visit_map_scroll(element)
      values = element.parameters
      if values.size == 3
        direction = Event_Compiler_Util.parse_direction(values[0])
        distance = values[1].to_i
        speed = values[2].to_i
        add_event_command(203,element.indent,[direction, distance, speed])
      end
      return next_accept(element)
    end
    
    # 204 マップの設定変更
    add_visitor_method("マップの設定変更", :visit_map_config)
    def visit_map_config(element)
      return next_accept(element)
    end
    
    # 205 フォグの色調変更
    add_visitor_method("フォグの色調変更", :visit_fog_tone)
    def visit_fog_tone(element)
      return next_accept(element)
    end
    
    # 206 フォグの不透明度変更
    add_visitor_method("フォグの不透明度変更", :visit_fog_opacity)
    def visit_fog_opacity(element)
      return next_accept(element)
    end
    
    # 207 アニメーションの表示
    add_visitor_method("アニメーションの表示", :visit_animation)
    def visit_animation(element)
      values = element.parameters
      if values.size == 2
        character_id = get_character_id(values[0])
        animation_id = Event_Compiler_Util.parse_animation_id(values[1])
        add_event_command(207,element.indent,[character_id,animation_id])
      end
      return next_accept(element)
    end
    
    # 208 透明状態の変更
    add_visitor_method("透明状態の変更", :visit_transparent)
    def visit_transparent(element)
      return next_accept(element)
    end
    
    # 209 移動ルートの設定
    add_visitor_method("移動ルートの設定", :visit_move_route)
    def visit_move_route(element)
      parameters = element.value.split(/,|、|（|）/)
      # 移動キャラクター
      character = 0
      case parameters[0]
      when "プレイヤー"
        character = -1
      when "このイベント"
        character = 0
      else # イベント指定
        character = get_character_id(parameters[0])
      end
      
      # 移動ルート作成
      move_route = RPG::MoveRoute.new
      
      if parameters.include?("動作を繰り返す")
        move_route.repeat = true # 動作を繰り返す
      else
        move_route.repeat = false # 動作を繰り返えさない
      end
      if parameters.include?("移動できない場合は無視")
        move_route.skippable = true # 移動でいない場合は無視する
      else
        move_route.skippable = false # 移動でいない場合は無視しない
      end
      
      add_event_command(209,element.indent,[character,move_route])
      
      # 移動ルート
      move_route.list = []
      
      element.related_elements.each do |related_element|
        # エレメント修正
         (name,value) = related_element.value.split(/：|:/,2)
        related_element.name = name unless name.nil?
        related_element.value = value unless value.nil?
        # アクセプト
        move_command = related_element.accept(self)
        move_route.list.push move_command
        add_event_command(509,related_element.indent,[move_command])
      end
      
      # 終了判定用
      if move_route.list[-1].code != 0
        move_route.list.push RPG::MoveCommand.new
      end
      
      return next_accept(element)
    end
    
    # 移動ルート
    [
    [0,'◇'],
    [1,'◇下に移動'],
    [2,'◇左に移動'],
    [3,'◇右に移動'],
    [4,'◇上に移動'],
    [5,'◇左下に移動'],
    [6,'◇右下に移動'],
    [7,'◇左上に移動'],
    [8,'◇右上に移動'],
    [9,'◇ランダムに移動'],
    [10,'◇プレイヤーに近づく'],
    [11,'◇プレイヤーから遠ざかる'],
    [12,'◇一歩前進'],
    [13,'◇一歩後退'],
    [14,'◇ジャンプ',:Integer,:Integer],
    [15,'◇ウェイト',:Integer],
    [16,'◇下を向く'],
    [17,'◇左を向く'],
    [18,'◇右を向く'],
    [19,'◇上を向く'],
    [20,'◇右に90度回転'],
    [21,'◇左に90度回転'],
    [22,'◇180度回転'],
    [23,'◇右か左に90度回転'],
    [24,'◇ランダムに方向転換'],
    [25,'◇プレイヤーの方を向く'],
    [26,'◇プレイヤーの逆を向く'],
    [27,'◇スイッチON',proc do |command,values|
      unless values.empty?
        command.parameters.push Event_Compiler_Util.parse_switch_id(values[0])
      end
    end
    ],
    [28,'◇スイッチOFF',proc do |command,values|
      unless values.empty?
        command.parameters.push Event_Compiler_Util.parse_switch_id(values[0])
      end
    end
    ],
    [29,'◇移動速度の変更',:Integer],
    [30,'◇移動頻度の変更',:Integer],
    [31,'◇移動時アニメON'],
    [32,'◇移動時アニメOFF'],
    [33,'◇停止時アニメON'],
    [34,'◇停止時アニメOFF'],
    [35,'◇向き固定ON'],
    [36,'◇向き固定OFF'],
    [37,'◇すり抜けON'],
    [38,'◇すり抜けOFF'],
    [39,'◇最前面に表示ON'],
    [40,'◇最前面に表示OFF'],
    [41,'◇グラフィック変更',proc do |command,values|
      unless values.empty?
        name = ""
        hue = 0
        direction = 2
        pattern = 0
        name = values[0].to_s if values.size >= 1
        hue = values[1].to_i if values.size >= 2
        if values.size >= 3
          direction = Event_Compiler_Util.parse_direction(values[2])
        end
        pattern = values[3].to_i if values.size >= 4
        command.parameters = [name, hue, direction, pattern]
      end
    end
    ],
    [42,'◇不透明度の変更',:Integer],
    [43,'◇合成方法の変更',proc do |command,values|
      unless values.empty?
        blend_type = Event_Compiler_Util.parse_blend_type(values[0])
        command.parameters.push blend_type
      end
    end
    ],
    [44,'◇SEの演奏', proc do |command,values|
      unless values.empty?
        se = RPG::AudioFile.new()
        se.name = values[0].to_s if values.size >= 1
        se.volume = values[1].to_i if values.size >= 2
        se.pitch = values[2].to_i if values.size >= 3
        command.parameters.push se
      end
    end
    ],
    [44,'◇SE', proc do |command,values|
      unless values.empty?
        se = RPG::AudioFile.new()
        se.name = values[0].to_s if values.size >= 1
        se.volume = values[1].to_i if values.size >= 2
        se.pitch = values[2].to_i if values.size >= 3
        command.parameters.push se
      end
    end
    ],
    [45,'◇スクリプト', :String],
    ].each do |id,value,*param_types|
      method_name = "visit_move_route__#{id}".to_sym
      # メソッド追加
      add_visitor_method(value, method_name)
      # メソッド作成
      define_method(method_name) do |element|
        # ルート作成
        command = RPG::MoveCommand.new(id)
        # ルートパラメーターをメソッド定義から作成
        element_values = element.value.split(/、|,/)
        param_types.zip(element_values) do |_type,_value|
          case _type
          when :Integer
            command.parameters.push _value.to_i
          when :String
            command.parameters.push _value.to_s
          when Proc,Method
            _type.call(command,element_values)
          end
        end
        return command
      end
    end
    
    # 210 移動完了までウェイト
    add_visitor_method("移動完了までウェイト", :visit_move_wait)
    def visit_move_wait(element)
      add_event_command(210,element.indent,[])
      return next_accept(element)
    end
    
    # 221 トランジション準備
    add_visitor_method("トランジション準備", :visit_transition_setup)
    def visit_transition_setup(element)
      add_event_command(221,element.indent,[])
      return next_accept(element)
    end
    
    # 222 トランジション実行
    add_visitor_method("トランジション実行", :visit_transition_exec)
    def visit_transition_exec(element)
      add_event_command(222,element.indent,[element.value.strip])
      return next_accept(element)
    end
    
    # 223 画面の色調変更
    add_visitor_method("画面の色調変更", :visit_screen_tone)
    def visit_screen_tone(element)
      p = element.value.delete("()").split(/,|、/).collect {|s| s.to_i}
      if p.size == 5
        tone = Tone.new(p[0],p[1],p[2],p[3])
        time = p[4]
        add_event_command(223,element.indent,[tone,time])
      end
      return next_accept(element)
    end
    
    # 224 画面のフラッシュ
    add_visitor_method("画面のフラッシュ", :visit_screen_flash)
    def visit_screen_flash(element)
      p = element.value.delete("()").split(/,|、/).collect {|s| s.to_i}
      if p.size == 5
        c = Color.new(p[0],p[1],p[2],p[3])
        t = p[4]
        add_event_command(224,element.indent,[c,t])
      end
      return next_accept(element)
    end
    
    # 225 画面のシェイク
    add_visitor_method("画面のシェイク", :visit_screen_shake)
    def visit_screen_shake(element)
      parameters = element.parameters
      if parameters.size == 3
        add_event_command(225,element.indent,parameters.collect {|n|n.to_i})
      end
      return next_accept(element)
    end
    
    # 231 ピクチャの表示
    add_visitor_method("ピクチャの表示", :visit_picture)
    def visit_picture(element)
      return next_accept(element)
    end
    
    # 232 ピクチャの移動
    add_visitor_method("ピクチャの移動", :visit_picture_move)
    def visit_picture_move(element)
      return next_accept(element)
    end
    
    # 233 ピクチャの回転
    add_visitor_method("ピクチャの回転", :visit_picture_angle)
    def visit_picture_angle(element)
      return next_accept(element)
    end
    
    # 234 ピクチャの色調変更
    add_visitor_method("ピクチャの色調変更", :visit_picture_tone)
    def visit_picture_tone(element)
      return next_accept(element)
    end
    
    # 235 ピクチャの消去
    add_visitor_method("ピクチャの消去", :visit_picture_erase)
    def visit_picture_erase(element)
      return next_accept(element)
    end
    
    # 236 天候の設定
    add_visitor_method("天候の設定", :visit_weather)
    def visit_weather(element)
      return next_accept(element)
    end
    
    # 241 BGM の演奏
    add_visitor_method("BGM の演奏", :visit_bgm)
    add_visitor_method("BGMの演奏", :visit_bgm)
    def visit_bgm(element)
      audio = create_audio_parameter(element.value)
      add_event_command(241,element.indent,[audio])
      return next_accept(element)
    end
    
    # 242 BGM のフェードアウト
    add_visitor_method("BGM のフェードアウト", :visit_bgm_fadeout)
    add_visitor_method("BGMのフェードアウト", :visit_bgm_fadeout)
    def visit_bgm_fadeout(element)
      time = 1
      if element.value =~ /([0-9]+?)/
        time = $1.to_i
      end
      add_event_command(242,element.indent,[time])
      return next_accept(element)
    end
    
    # 245 BGS の演奏
    add_visitor_method("BGS の演奏", :visit_bgs)
    def visit_bgs(element)
      return next_accept(element)
    end
    
    # 246 BGS のフェードアウト
    add_visitor_method("BGS のフェードアウト", :visit_bgs_fadeout)
    def visit_bgs_fadeout(element)
      return next_accept(element)
    end
    
    # 247 BGM / BGS の記憶
    add_visitor_method("BGM / BGS の記憶", :visit_bgm_bgs_save)
    def visit_bgm_bgs_save(element)
      return next_accept(element)
    end
    
    # 248 BGM / BGS の復帰
    add_visitor_method("BGM / BGS の復帰", :visit_bgm_bgs_load)
    def visit_bgm_bgs_load(element)
      return next_accept(element)
    end
    
    # 249 ME の演奏
    add_visitor_method("ME の演奏", :visit_me)
    def visit_me(element)
      return next_accept(element)
    end
    
    # 250 SE の演奏
    add_visitor_method("SE の演奏", :visit_se)
    add_visitor_method("SEの演奏", :visit_se)
    add_visitor_method("SE", :visit_se)
    def visit_se(element)
      values = element.parameters
      unless values.empty?
        se = RPG::AudioFile.new()
        se.name = values[0].to_s if values.size >= 1
        se.volume = values[1].to_i if values.size >= 2
        se.pitch = values[2].to_i if values.size >= 3
        add_event_command(250,element.indent,[se])
      end
      return next_accept(element)
    end
    
    # 251 SE の停止
    add_visitor_method("SE の停止", :visit_se_stop)
    def visit_se_stop(element)
      return next_accept(element)
    end
    
    # 301 バトルの処理
    add_visitor_method("バトルの処理", :visit_battle)
    def visit_battle(element)
      values = element.value.split(/,|、/)
      unless values.empty?
        troop_id = get_troop_id(values[0])
        add_event_command(301,element.indent,[troop_id,false,false])
        element.related_elements.zip(element.child_elements) do |related_element,child_element|
          if related_element.nil? or child_element.nil?
            add_event_command(604,element.indent,[])
            break
          end
          related_element.accept(self)
          child_element.accept(self)
        end
      end
      return next_accept(element)
    end
    
    # 601 勝った場合
    add_visitor_method("勝った場合", :visit_battle_win)
    def visit_battle_win(element)
      add_event_command(601,element.indent)
      return next_accept(element)
    end
    
    # 602 逃げた場合
    add_visitor_method("逃げた場合", :visit_battle_escape)
    def visit_battle_escape(element)
      get_event_command(301,element.indent).parameters[1] = true
      add_event_command(602,element.indent)
      return next_accept(element)
    end
    
    # 603 負けた場合
    add_visitor_method("負けた場合", :visit_battle_lose)
    def visit_battle_lose(element)
      get_event_command(301,element.indent).parameters[2] = true
      add_event_command(603,element.indent)
      return next_accept(element)
    end
    
    # 302 ショップの処理
    add_visitor_method("ショップの処理", :visit_shop)
    def visit_shop(element)
      return next_accept(element)
    end
    
    # 303 名前入力の処理
    add_visitor_method("名前入力の処理", :visit_input_name)
    def visit_input_name(element)
      return next_accept(element)
    end
    
    # 311 HP の増減
    add_visitor_method("HP の増減", :visit_increase_or_decrease_hp)
    def visit_increase_or_decrease_hp(element)
      return next_accept(element)
    end
    
    # 312 SP の増減
    add_visitor_method("SP の増減", :visit_increase_or_decrease_sp)
    def visit_increase_or_decrease_sp(element)
      return next_accept(element)
    end
    
    # 313 ステートの変更
    add_visitor_method("ステートの変更", :visit_state_change)
    def visit_state_change(element)
      return next_accept(element)
    end
    
    # 314 全回復
    add_visitor_method("全回復", :visit_recover_all)
    def visit_recover_all(element)
      return next_accept(element)
    end
    
    # 315 EXP の増減
    add_visitor_method("EXP の増減", :visit_increase_or_decrease_exp)
    def visit_increase_or_decrease_exp(element)
      return next_accept(element)
    end
    
    # 316 レベルの増減
    add_visitor_method("レベルの増減", :visit_increase_or_decrease_lebel)
    def visit_increase_or_decrease_lebel(element)
      return next_accept(element)
    end
    
    # 317 パラメータの増減
    add_visitor_method("パラメータの増減", :visit_increase_or_decrease_parameters)
    def visit_increase_or_decrease_parameters(element)
      return next_accept(element)
    end
    
    # 318 スキルの増減
    add_visitor_method("スキルの増減", :visit_increase_or_decrease_skill)
    def visit_increase_or_decrease_skill(element)
      return next_accept(element)
    end
    
    # 319 装備の変更
    add_visitor_method("装備の変更", :visit_equip_change)
    def visit_equip_change(element)
      return next_accept(element)
    end
    
    # 320 アクターの名前変更
    add_visitor_method("アクターの名前変更", :visit_actor_name_change)
    def visit_actor_name_change(element)
      return next_accept(element)
    end
    
    # 321 アクターのクラス変更
    add_visitor_method("アクターのクラス変更", :visit_actor_class_change)
    def visit_actor_class_change(element)
      return next_accept(element)
    end
    
    # 322 アクターのグラフィック変更
    add_visitor_method("アクターのグラフィック変更", :visit_actor_graphics_change)
    def visit_actor_graphics_change(element)
      return next_accept(element)
    end
    
    # 331 エネミーの HP 増減
    add_visitor_method("エネミーの HP 増減", :visit_enemy_increase_or_decrease_hp)
    def visit_enemy_increase_or_decrease_hp(element)
      return next_accept(element)
    end
    
    # 332 エネミーの SP 増減
    add_visitor_method("エネミーの SP 増減", :visit_enemy_increase_or_decrease_sp)
    def visit_enemy_increase_or_decrease_sp(element)
      return next_accept(element)
    end
    
    # 333 エネミーのステート変更
    add_visitor_method("エネミーのステート変更", :visit_enemy_state_change)
    def visit_enemy_state_change(element)
      return next_accept(element)
    end
    
    # 334 エネミーの出現
    add_visitor_method("エネミーの出現", :visit_enemy_appear)
    def visit_enemy_appear(element)
      return next_accept(element)
    end
    
    # 335 エネミーの変身
    add_visitor_method("エネミーの変身", :visit_enemy_metamorphosis)
    def visit_enemy_metamorphosis(element)
      return next_accept(element)
    end
    
    # 336 エネミーの全回復
    add_visitor_method("エネミーの全回復", :visit_enemy_recover_all)
    def visit_enemy_recover_all(element)
      return next_accept(element)
    end
    
    # 337 戦闘アニメーションの表示
    add_visitor_method("戦闘アニメーションの表示", :visit_battle_animation)
    def visit_battle_animation(element)
      return next_accept(element)
    end
    
    # 338 ダメージの処理
    add_visitor_method("ダメージの処理", :visit_battle_damage)
    def visit_battle_damage(element)
      return next_accept(element)
    end
    
    # 339 アクションの強制
    add_visitor_method("アクションの強制", :visit_battle_action)
    def visit_battle_action(element)
      return next_accept(element)
    end
    
    # 340 バトルの中断
    add_visitor_method("バトルの中断", :visit_battle_break)
    def visit_battle_break(element)
      return next_accept(element)
    end
    
    # 351 メニュー画面の呼び出し
    add_visitor_method("メニュー画面の呼び出し", :visit_call_menu)
    def visit_call_menu(element)
      return next_accept(element)
    end
    
    # 352 セーブ画面の呼び出し
    add_visitor_method("セーブ画面の呼び出し", :visit_call_save)
    def visit_call_save(element)
      return next_accept(element)
    end
    
    # 353 ゲームオーバー
    add_visitor_method("ゲームオーバー", :visit_call_gameover)
    def visit_call_gameover(element)
      add_event_command(353,element.indent)
      return next_accept(element)
    end
    
    # 354 タイトル画面に戻す
    add_visitor_method("タイトル画面に戻す", :visit_call_title)
    def visit_call_title(element)
      return next_accept(element)
    end
    
    # 355 スクリプト
    add_visitor_method("スクリプト", :visit_script)
    def visit_script(element)
      add_event_command(355,element.indent,[element.value])
      element.related_elements.each do |related_element|
        add_event_command(655,element.indent,[related_element.value])
      end
      return next_accept(element)
    end
    
    #-------------------------
    # オリジナルイベントコマンド
    #-------------------------
    def add_script_event_command(element,value)
      value.split(/\n/).each_with_index do |script,index|
        if index == 0
          add_event_command(355,element.indent,[script])
        else
          add_event_command(655,element.indent,[script])
        end
      end
      add_event_command(655,element.indent,["true"])
    end
    
    add_visitor_method("イベントのトーン変更", :visit_event_tone_change)
    add_visitor_method("イベントの色調変更", :visit_event_tone_change)
    def visit_event_tone_change(element)
      parameters = element.parameters
      if parameters.empty?
        Log.error("パラメーターがありません。")
        return next_accept(element)
      end
      event_id = get_character_id(parameters[0])
      tone = []
      case parameters.size
      when 4
        tone[0] = parameters[1] # r
        tone[1] = parameters[2] # g
        tone[2] = parameters[3] # b
      when 5
        tone[0] = parameters[1] # r
        tone[1] = parameters[2] # g
        tone[2] = parameters[3] # b
        tone[3] = parameters[4] # b
      else
        Log.error("パラメーター数が合いません。")
        return next_accept(element)
      end
      script = <<_SCRIPT_
get_character(#{event_id}).tone = Tone.new(#{tone.join(',')})
_SCRIPT_
      add_script_event_command(element,script)
      return next_accept(element)
    end
    
    add_visitor_method("イベントのビューポート除外", :visit_event_viewport_remove)
    def visit_event_viewport_remove(element)
      parameters = element.parameters
      if parameters.empty?
        Log.error("イベントのビューポート除外：パラメーターがありません。")
        return next_accept(element)
      end
      event_id_list = []
      parameters.each do |param|
        if param =~ /^[0-9]+$/
          event_id_list.push param.to_i
        elsif param == "プレイヤー"
          event_id_list.push 0
        elsif param == "このイベント"
          event_id_list.push @event.id
        else
          id = get_character_id(param)
          event_id_list.push id if id != 0
        end
      end
      script = <<_SCRIPT_
e = [#{event_id_list.join(',')}]
$scene.change_character_viewport(e)
_SCRIPT_
      add_script_event_command(element,script)
      return next_accept(element)
    end
    
    add_visitor_method("イベントのビューポート復帰", :visit_event_viewport_return)
    def visit_event_viewport_return(element)
      parameters = element.parameters
      if parameters.empty?
        Log.error("イベントのビューポート復帰:パラメーターがありません。")
        return next_accept(element)
      end
      event_id_list = []
      parameters.each do |param|
        if param =~ /^[0-9]+$/
          event_id_list.push param.to_i
        elsif param == "プレイヤー"
          event_id_list.push 0
        elsif param == "このイベント"
          event_id_list.push @event.id
        else
          id = get_character_id(param)
          event_id_list.push id if id != 0
        end
      end
      script = <<_SCRIPT_
e = [#{event_id_list.join(',')}]
$scene.reset_character_viewport(e)
_SCRIPT_
      add_script_event_command(element,script)
      return next_accept(element)
    end
    
    add_visitor_method("パーティの並びを保存", :visit_save_party_order)
    def visit_save_party_order(element)
      add_script_event_command(element,"Event_Script.save_party_order")
      return next_accept(element)
    end
    
    add_visitor_method("パーティの並びを復帰", :visit_load_party_order)
    def visit_load_party_order(element)
      add_script_event_command(element,"Event_Script.load_party_order")
      return next_accept(element)
    end
    
    add_visitor_method("パーティの並びを変更", :visit_change_party_order)
    def visit_change_party_order(element)
      parameters = element.parameters
      script = <<_SCRIPT_
l = [#{parameters.join(',')}]
Event_Script.change_party_order(l)
_SCRIPT_
      add_script_event_command(element,script)
      return next_accept(element)
    end
    
  end
end
