/*
 * Copyright(C) 2007 Jiro Nishiguchi <jiro@cpan.org>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package senna;

/**
 * Wrapper for sen_index
 */
public class Index {

    /**
     * <pre>
     *  call sen_index_create() with below values.
     *  keySize: 0
     *  flags: 0
     *  initialNSegments: 0
     *  encoding: SEN_ENC_DEFAULT
     * </pre>
     * 
     * @param path
     *            path
     * @return created sen_index
     * @throws SennaException
     */
    public static Index create(String path) throws SennaException {
        return create(path, 0, 0, 0, Senna.SEN_ENC_DEFAULT);
    }

    /**
     * <pre>
     *  call sen_index_create() with below values.
     *  flags: 0
     *  initialNSegments: 0
     *  encoding: SEN_ENC_DEFAULT
     * </pre>
     * 
     * @param path
     *            path
     * @param keySize
     *            key size
     * @return created sen_index
     * @throws SennaException
     */
    public static Index create(String path, int keySize) throws SennaException {
        return create(path, keySize, 0, 0, Senna.SEN_ENC_DEFAULT);
    }

    /**
     * <pre>
     *  call sen_index_create() with below values.
     *  initialNSegments: 0
     *  encoding: SEN_ENC_DEFAULT
     * </pre>
     * 
     * @param path
     *            path
     * @param keySize
     *            key size
     * @param flags
     *            flags
     * @return created sen_index
     * @throws SennaException
     */
    public static Index create(String path, int keySize, int flags)
            throws SennaException {
        return create(path, keySize, flags, 0, Senna.SEN_ENC_DEFAULT);
    }

    /**
     * <pre>
     *  call sen_index_create() with below values.
     *  encoding: SEN_ENC_DEFAULT
     * </pre>
     * 
     * @param path
     *            path
     * @param keySize
     *            key size
     * @param flags
     *            flags
     * @param initialNSegments
     *            initialNSegments
     * @return created sen_index
     * @throws SennaException
     */
    public static Index create(String path, int keySize, int flags,
            int initialNSegments) throws SennaException {
        return create(path, keySize, flags, initialNSegments,
                Senna.SEN_ENC_DEFAULT);
    }

    /**
     * call sen_index_create().
     * 
     * @param path
     *            path
     * @param keySize
     *            key size
     * @param flags
     *            flags
     * @param initialNSegments
     *            initialNSegments
     * @param encoding
     *            encoding
     * @return created sen_index
     * @throws SennaException
     */
    public static native Index create(String path, int keySize, int flags,
            int initialNSegments, int encoding) throws SennaException;

    /**
     * call sen_index_rename().
     * 
     * @param oldName
     *            old name
     * @param newName
     *            new name
     * @throws SennaException
     * 
     */
    public static native void rename(String oldName, String newName)
            throws SennaException;

    /**
     * call sen_index_remove().
     * 
     * @param path
     *            path
     * @throws SennaException
     */
    public static native void remove(String path) throws SennaException;

    /* C pointer to sen_index */
    private int ptr;

    /* for create() */
    private Index() {
    }

    /**
     * creates new object with sen_index_open().
     * 
     * @param path
     *            path
     * @throws SennaException
     */
    public Index(String path) throws SennaException {
        open(path);
    }

    /**
     * call sen_index_open().
     * 
     * @param path
     *            path
     * @throws SennaException
     */
    private native void open(String path) throws SennaException;

    /**
     * call sen_index_upd().
     * 
     * @param key
     *            key
     * @param value
     *            value
     * @throws SennaException
     */
    public native void insert(String key, String value) throws SennaException;

    /**
     * call sen_index_upd().
     * 
     * @param key
     *            key
     * @param oldValue
     *            old value
     * @throws SennaException
     */
    public native void delete(String key, String oldValue)
            throws SennaException;

    /**
     * call sen_index_upd().
     * 
     * @param key
     *            key
     * @param oldValue
     *            old value
     * @param newValue
     *            new value
     * @throws SennaException
     */
    public native void upd(String key, String oldValue, String newValue)
            throws SennaException;

    /**
     * call sen_index_info() and print results.
     * 
     * @throws SennaException
     */
    public native void printInfo() throws SennaException;

    /**
     * call sen_index_close().
     * 
     * @throws SennaException
     */
    public native void close() throws SennaException;

    /**
     * call sen_index_sel().
     * 
     * @param query
     *            query
     * @return records
     * @throws SennaException
     */
    public native Records sel(String query) throws SennaException;

}
