# -*- coding: utf-8 -*-

import re
import sys
import time

from trac.perm import PermissionError, PermissionSystem
from trac.resource import ResourceNotFound
from trac.ticket.model import Ticket
from trac.util.compat import set
from trac.util.text import console_print
from trac.web.chrome import Chrome

from genshi.builder import tag
from genshi.template.markup import MarkupTemplate

def to_localdate(utcdate, zoneoffset):
    return time.strftime("%Y/%m/%d %H:%M:%S", time.gmtime(utcdate + zoneoffset))

def get_mlperm_action(mlid):
    return u'MAILARCHIVE_VIEW_%s' % mlid.upper()

def get_accessable_ml(env, req, ml_permission_map):
    perm = PermissionSystem(env)
    actions = perm.get_user_permissions(req.authname)
    view_actions = [x for x in actions if x.startswith('MAILARCHIVE_VIEW_')]
    accessable_mlids = [ml_permission_map.get(x) for x in view_actions]
    return view_actions, accessable_mlids

def has_mailarchive_view_permission(env, req):
    if 'MAILARCHIVE_VIEW' in req.perm('mailarchive'):
        return True
    
    perm = PermissionSystem(env)
    actions = perm.get_user_permissions(req.authname)
    view_actions = [x for x in actions if x.startswith('MAILARCHIVE_VIEW_')]

    if len(view_actions) > 0:
        return True
    else:
        return False
    
def linkify_id(env, req, id=None, mail=None):
    from model import Mail
    try:
        m = None
        if mail:
            m = mail
        else:
            m = Mail(env, id)
        m.assert_permission(req)
        link = tag.a('mail:%s' % m.id,
                     href=req.href.mailarchive(m.id),
                     class_='ticket',
                     title='%s %s %s' % (m.get_fromtext(), m.get_senddate(), m.subject))
        return link
    except (ResourceNotFound, PermissionError), e:
        return id

def linkify_ids(env, req, ids):
    if len(ids) == 0 or ids[0] == '':
        return tag.span()
    
    data = []
    
    from model import MailFinder
    mails = MailFinder.find_mails(env, ids)
    
    for mail in sorted(mails, key=lambda x: int(x.id)):
        data.append(linkify_id(env, req, mail=mail))
        data.append(', ')
    if data:
        del data[-1] # Remove the last comma if needed
    return tag.span(*data)

def get_author(fromname, fromaddr):
    author = fromname
    if fromname == '':
        if re.match('(.+?)@', fromaddr):
            author = re.match('(.+?)@', fromaddr).group(1)
    if author == None or author.strip() == '':
        author = '--'
    return author

def get_related_tickets(env, req, db, id):
    # cached?
    if not hasattr(req, '_related_tikects'):
        cursor = db.cursor()
        
        sql = """SELECT ticket, value
        FROM ticket_custom
        WHERE name='mail_id' AND value <> ''"""
        
        cursor.execute(sql)
        related_tickets = [(ticket_id, [int(x) for x in mail_ids.split(',') if x != '']) for ticket_id, mail_ids in cursor]
        
        req._related_tikects = related_tickets
        
    related_tickets = req._related_tikects
    tickets = []
    for ticket_id, mail_ids in related_tickets:
        if int(id) in mail_ids:
            tickets.append(Ticket(env, ticket_id))
        
    return tickets

def show_thread(env, req, mail_ids):
    from model import MailFinder
    mails = MailFinder.find_mails(env, mail_ids)
    
    template = MarkupTemplate("""
  <div xmlns:py="http://genshi.edgewall.org/">
 
  <script type="text/javascript" src="${req.chrome.htdocs_location}js/folding.js"></script>
  <script type="text/javascript">
    jQuery(document).ready(function($) {
      $("fieldset legend.foldable").enableFolding(false);
      /* Hide the filters for saved queries. */
      $(".options").toggleClass("collapsed");
    });
  </script>    
    
  <py:def function="show_thread(mail, cached_mails)">
  <li class="thread_li">
    <py:choose>
      <py:when test="not mail.has_permission(req)">
        (表示権限がありません)
      </py:when>
      <py:otherwise test="">
        <py:choose>
          <span py:when="mail.has_attachments(req)" class="mail_clickable mail_subject thread_subject_clip">
          [$mail.id] ${mail.get_subject()}</span>
          <span py:otherwise="" class="mail_clickable mail_subject"
            href="#">[$mail.id] ${mail.get_subject()}</span>
          <py:with vars="related_tickets=mail.get_related_tickets(req);">
            <py:if test="len(related_tickets) > 0">
              <span class="related_tickets">
                --- (関連チケット: 
                <py:for each="ticket in related_tickets">
                  <a href="${req.href.ticket(ticket.id)}">#${ticket.id}</a>&nbsp;
                </py:for>
                )
              </span>
            </py:if>
          </py:with>
          &nbsp;&nbsp;<a class="open_mail" target="_brank"
            href="${req.href.mailarchive(mail.id)}" title="別ウィンドウで開く"><span class="icon"/></a>
        </py:choose>
        <br />
        <span class="thread_from">${mail.get_fromtext()}</span>
        <span class="thread_senddate">${mail.get_senddate()}</span>
        <div class="mail_id" style="display:none;">${mail.id}</div>
        <div class="loading" style="display:none;">
          <img src="%s"/>
        </div>
        <div class="mailplaceholder"></div>
      </py:otherwise>
    </py:choose>

    <py:for each="child in mail.get_children(cached_mails=cached_mails)">
      <ul class="thread_ul">
        ${show_thread(child, cached_mails)}
      </ul>
    </py:for>
  </li>
  </py:def>

  <h2>関連メール</h2>
  <div class="mail_threads">
    <py:for each="mail in mails">
      <fieldset class="options">
        <legend class="foldable" style="! important">[${mail.id}] ${mail.subject}</legend>
        <span class="thread_from">${mail.get_fromtext()}</span>
        <span class="thread_senddate">${mail.get_senddate()}</span>
        <div id="mail_thread">
          <ul class="thread_ul">
            ${show_thread(mail.get_thread_root(), mail.get_thread_mails())}
          </ul>
        </div>
      </fieldset>
    </py:for>
  </div>
  </div>""" % (req.href.chrome('mailarchive/png/loading.gif')))
    return template.generate(mails=mails, req=req)

def get_category_href(req, category):
    return req.href.mailarchive() + '?category=%s' % category

def get_item_per_page(env):
    return int(env.config.get('mailarchive', 'items_page','50'))

def get_num_shown_pages(env):
    return int(env.config.get('mailarchive', 'shown_pages','30'))

def month_add(year, month, add_month):
    month = month + add_month
    while month >12 or month <1:
        if month > 12:
            month = month - 12
            year = year + 1
        else :
            month = month + 12
            year = year - 1
        
def is_empty(str):
    return str == None or str.strip() == ''
    
def printout(*args):
    console_print(sys.stdout, *args)

def printerr(*args):
    console_print(sys.stderr, *args)
    
class Logger:
    
    def __init__(self, env):
        try:
            self.log = env.log
        except AttributeError:
            self.log = None
            
    def debug(self, *args):
        if self.log:
            self.log.debug(*args)
        else:
            printout(*args)
            
    def info(self, *args):
        if self.log:
            self.log.info(*args)
        else:
            printout(*args)
            
    def warn(self, *args):
        if self.log:
            self.log.warn(*args)
        else:
            printout(*args)
    
    def error(self, *args):
        if self.log:
            self.log.error(*args)
        else:
            printerr(*args)
        

