# -*- coding: utf-8 -*-

from genshi.builder import tag
from genshi.filters import Transformer

from trac.ticket.report import ReportModule
from trac.util.datefmt import format_datetime, format_date, format_time

class ReportRenderer(object):
    
    def __init__(self, env):
        self.env = env
        
    def render(self, req, params):
        u"""TracReportsの結果をHTMLで返す。オプションによってはテーブル、グラフそれぞれを表示する。
        """
        self.index = req._reportinclude
        
        id, vars, opts = self._parse_params(params)
        report = self._get_report_info(id)
        
        if len(report) == 1:
            title, sql, desc = report[0]
            
            if opts['title'] == '':
                opts['title'] = title
                
            table = self._render_table(req, id, vars, opts, title, sql, desc)
            return self._render_graph(req, opts, table, title, desc)
        else:
            raise ResourceNotFound(
                _('Report [%(num)] does not exist.', num=id),
                _('Invalid Report Number'))

    def _render_table(self, req, id, vars, opts, title, sql, desc):
             
        db = self.env.get_db_cnx()
        
        sql, vars = self._sql_sub_vars(sql, vars, db)
        
        cursor = db.cursor()
        cursor.execute(sql, vars)
        
        if opts['table'] == 'hide':
            table_display = 'none'
        else:
            table_display = 'inline'
        
        table = tag.table(class_="listing tickets",
                          id="reportgraphtable_%d" %(self.index),
                          style="display:%s" %(table_display))
        caption = tag.caption(tag.a('%s(report:%s)' %(title, id), href='%s' % req.href.report(id)),
                              align='center')
        table.append(caption)
        
        tr = tag.tr()
        columns = []
        for desc in cursor.description:
            columns.append(desc[0])
            if desc[0] == '__color__' or desc[0] == '__group__':
                continue
            if desc[0].startswith('_'):
                continue
            
            tr.append(tag.th('%s' % desc[0]))
        
        table.append(tag.thead(tr))
            
        tbody = tag.tbody()
        table.append(tbody)
        
        for idx, row in enumerate(cursor):
            
            odd_or_even = (idx % 2 == 0) and 'odd' or 'even'
            tr = tag.tr()
            css_class = ''
            
            for col_idx, col in enumerate(row):
                column = columns[col_idx]
                if column == '__color__' and css_class is '':
                    css_class = 'color%s-' % col
                    continue
                if column.startswith('_'):
                    continue
                
                if column == 'time':
                    col = col != None and format_time(int(col)) or '--'
                if column in ('date', 'created', 'modified'):
                    col = col != None and format_date(int(col)) or '--'
                if column == 'datetime':
                    col = col != None and format_datetime(int(col)) or '--'
                
                if column.lower() in ('ticket', 'id', u'チケット'):
                    col = tag.a('#' + str(col), title='View Ticket', class_='ticket',
                                href=req.href.ticket(str(col)))
                    
                tr = tr(tag.td(col))
                
            css_class = css_class + odd_or_even
            tr = tr(class_ = css_class)
            tbody.append(tr)
        
        #add_stylesheet(req, 'wikitable/css/wikitable.css')
        return table

    def _render_graph(self, req, opts, table, title, desc):

        if opts['graph'] != 'lines' and opts['graph'] != 'bars':  
            return table

        opttag = tag.div(id="reportgraphopt_%d" % (self.index),
                         style="display:none")

        for opt in opts:
            opttag.append(tag.span(opts[opt], class_=opt))

        script = """
jQuery(document).ready(function($) {
  renderGraph("#reportgraph_%d");
});
        """ % (self.index)

        div = tag.div(
                      tag.div(' ',
                              id="placeholder_%d" % (self.index),
                              style="width:%spx;height:%spx;" %
                              (opts["width"],opts["height"])),
                      opttag,
                      tag.br(),
                      table,
                      tag.script(script, type="text/javascript"),
                      class_="reportgraph",
                      id="reportgraph_%d" % (self.index)
                      )
        return div
    
    def _parse_params(self, params):
        opts = {
               'title':'',
               'renderer':'jqplot',
               'width':'536',
               'height':'300',
               'end':'',
               'start':'',
               'per':'day',
               'graph':'',
               'table':'inline',
               'dateFormat':'yyyy-MM-dd',
               'stack':False,
               'legendLoc':'ne',
               'legendXOffset':'12',
               'legendYOffset':'12',
               'xaxisMin':'null',
               'xaxisMax':'null',
               'yaxisMin':'null',
               'yaxisMax':'null',
               'xaxisFormatString':'',
               'yaxisFormatString':''
               }
        
        vars = {}
        for (index, param) in enumerate(params.split(',')):
            if index == 0:
                id = param
                id, vars = self._parse_vars(id)
                continue
            
            colon_split = param.split(':')
            key = colon_split[0].strip()
            value = ''
            if len(colon_split) > 1:
                value = ':'.join(colon_split[1:])
            else:
                value = True
            opts[key] = value
        return id, vars, opts
    
    def _parse_vars(self, id):
        vars = {}
        
        if id.find('?') == -1:
            return id, vars
        
        id_and_params = id.split('?')
        params = id_and_params[1]
        id = id_and_params[0]
        
        if params.find('&') != -1:
            for (index, param) in enumerate(params.split('&')):
                if param.find('=') == -1:
                    continue
                entry = param.split('=')
                vars[entry[0]] = entry[1]
        elif params.find('=') != -1:
            entry = params.split('=')
            vars[entry[0]] = entry[1]
        
        return id, vars
    
    def _get_report_info(self, id):
        db = self.env.get_db_cnx()
        cursor = db.cursor()
        cursor.execute("SELECT title,query,description from report "
                       "WHERE id=%s", (id,))
        
        report = [(title, sql, desc) for title, sql, desc in cursor]
        return report
    
    def _sql_sub_vars(self, sql, vars, db):
        rm = ReportModule(self.env)
        sql, vars = rm.sql_sub_vars(sql, vars, db)
        return sql, vars
