﻿using NUnit.Framework;
using Meyn.TestLink;
using CookComputing.XmlRpc;

namespace BTS.Trac.Test
{
    using Priority = TracNameAndValueField<IPriority>;

    [TestFixture]
    [TestLinkFixture(
        Url = TestLinkConnect.Url,
        ProjectName = TestLinkConnect.ProjectName,
        UserId = TestLinkConnect.UserId,
        TestPlan = TestLinkConnect.TestPlan,
        TestSuite = TestLinkConnect.TestSuite,
        DevKey = TestLinkConnect.DevKey )]
    public class PriorityTest
    {
        [SetUp]
        public void SetUp()
        {
            Trac.Connect( TracConnect.Url, TracConnect.UserName, TracConnect.Password );

            // 最初にすべて削除しておく
            string[] priorityNames = Priority.GetAll();
            foreach ( string priorityName in priorityNames ) {
                Priority.Delete( priorityName );
            }
        }

        [TearDown]
        public void TearDown()
        {
            Trac.Close();
        }

        [Test]
        public void 優先度の項目を取得する()
        {
            int count = 10;

            // 作成
            for ( int i = 0; i < count; ++i ) {
                string priorityName = "Test" + i.ToString();
                Priority priority = new Priority();
                priority.Value = i.ToString();
                priority.Create( priorityName );
            }

            string[] priorityNames = Priority.GetAll();
            Assert.AreEqual( priorityNames.Length, count );
        }

        [Test]
        public void 優先度の各項目を取得する()
        {
            int count = 10;
            string priorityNameBase = "Test";

            // 作成
            for ( int i = 0; i < count; ++i ) {
                string priorityName = priorityNameBase + i.ToString();
                Priority priority = new Priority();
                priority.Value = i.ToString();
                priority.Create( priorityName );
            }

            // 作成数を確認
            string[] priorityNames = Priority.GetAll();
            Assert.AreEqual( priorityNames.Length, count );

            for ( int i = 0; i < count; ++i ) {
                string priorityName = priorityNameBase + i.ToString();
                Priority priority = new Priority( priorityName );

                // 作成したものと取得したものが一致する
                Assert.AreEqual( priority.Name, priorityName );
                Assert.AreEqual( priority.Value, i.ToString() );
            }
        }

        [Test]
        public void 優先度を作成する()
        {
            string[] before = Priority.GetAll();

            // 作成
            string priorityName = "Test";
            Priority priority = new Priority();
            priority.Value = "10";
            priority.Create( priorityName );

            // 優先度の数が増えていること
            Assert.AreEqual( before.Length + 1, Priority.GetAll().Length );

            // 取得した優先度と一致すること
            Priority getPriority = new Priority( priorityName );
            Assert.AreEqual( priority.Name, getPriority.Name );
            Assert.AreEqual( priority.Value, getPriority.Value );
        }

        [Test]
        public void 優先度を更新する()
        {
            string[] before = Priority.GetAll();

            // 作成
            string priorityName = "Test";
            Priority priority = new Priority();
            priority.Value = "10";
            priority.Create( priorityName );

            // 優先度の数が増えていること
            Assert.AreEqual( before.Length + 1, Priority.GetAll().Length );

            // 更新
            Priority getPriority = new Priority( priorityName );
            getPriority.Value = "11";
            getPriority.Update();

            // 更新した優先度と一致すること
            Priority updatePriority = new Priority( priorityName );
            Assert.AreEqual( getPriority.Name, updatePriority.Name );
            Assert.AreEqual( getPriority.Value, updatePriority.Value );
        }

        [Test]
        public void 優先度を削除する()
        {
            string[] before = Priority.GetAll();

            // 作成
            string priorityName = "Test";
            Priority priority = new Priority();
            priority.Value = "10";
            priority.Create( priorityName );

            // 優先度の数が増えていること
            Assert.AreEqual( before.Length + 1, Priority.GetAll().Length );

            priority.Delete();

            // 優先度の数が戻っていること
            Assert.AreEqual( before.Length, Priority.GetAll().Length );
        }

        [Test]
        public void 優先度を削除するstatic版()
        {
            string[] before = Priority.GetAll();

            // 作成
            string priorityName = "Test";
            Priority priority = new Priority();
            priority.Value = "10";
            priority.Create( priorityName );

            // 優先度の数が増えていること
            Assert.AreEqual( before.Length + 1, Priority.GetAll().Length );

            Priority.Delete( priorityName );

            // 優先度の数が戻っていること
            Assert.AreEqual( before.Length, Priority.GetAll().Length );
        }

        [Test]
        [Ignore]
        public void 同じ順序の優先度を作成すると例外が発生する()
        {
            ///
            /// 今のところ例外が発生せず正常に登録されてしまう
            ///

            // 作成
            string priorityName = "Test";
            Priority priority = new Priority();
            priority.Value = "10";
            priority.Create( priorityName );

            priority.Create( priorityName + "2" );

            // 同じ順序の優先度を作成すると例外が発生する
            Assert.Fail();
        }

        [Test]
        [ExpectedException( typeof( XmlRpcFaultException ) )]
        public void 同じ名称の優先度を作成すると例外が発生する()
        {
            // 作成
            string priorityName = "Test";
            Priority priority = new Priority();
            priority.Value = "10";
            priority.Create( priorityName );

            priority.Value = "11";
            priority.Create( priorityName );

            // 同じ名称の優先度を作成すると例外が発生する
            Assert.Fail();
        }

        [Test]
        [ExpectedException( typeof( XmlRpcFaultException ) )]
        public void 登録されていない名称の優先度を取得すると例外が発生する()
        {
            Priority priority = new Priority( "存在しない優先度" );

            // 登録されていない名称の優先度を取得すると例外が発生する
            Assert.Fail();
        }

        [Test]
        [ExpectedException( typeof( XmlRpcFaultException ) )]
        public void 登録されていない名称の優先度を更新すると例外が発生する()
        {
            Priority priority = new Priority();
            priority.Update();

            // 登録されていない名称の優先度を更新すると例外が発生する
            Assert.Fail();
        }

        [Test]
        [ExpectedException( typeof( XmlRpcFaultException ) )]
        public void 登録されていない名称の優先度を削除すると例外が発生する()
        {
            Priority.Delete( "存在しない優先度" );

            // 登録されていない名称の優先度を削除すると例外が発生する
            Assert.Fail();
        }
    }
}
