﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using NUnit.Framework;
using Meyn.TestLink;
using CookComputing.XmlRpc;

namespace BTS.Trac.Test
{
    using TicketType = TracNameAndValueField<ITicketType>;

    [TestFixture]
    [TestLinkFixture(
        Url = TestLinkConnect.Url,
        ProjectName = TestLinkConnect.ProjectName,
        UserId = TestLinkConnect.UserId,
        TestPlan = TestLinkConnect.TestPlan,
        TestSuite = TestLinkConnect.TestSuite,
        DevKey = TestLinkConnect.DevKey )]
    public class TicketTypeTest
    {
        [SetUp]
        public void SetUp()
        {
            Trac.Connect( TracConnect.Url, TracConnect.UserName, TracConnect.Password );

            // 最初にすべて削除しておく
            string[] ticketTypeNames = TicketType.GetAll();
            foreach ( string ticketTypeName in ticketTypeNames ) {
                TicketType.Delete( ticketTypeName );
            }
        }

        [TearDown]
        public void TearDown()
        {
            Trac.Close();
        }

        [Test]
        public void チケット分類の項目を取得する()
        {
            int count = 10;

            // 作成
            for ( int i = 0; i < count; ++i ) {
                string ticketTypeName = "Test" + i.ToString();
                TicketType ticketType = new TicketType();
                ticketType.Value = i.ToString();
                ticketType.Create( ticketTypeName );
            }

            string[] ticketTypeNames = TicketType.GetAll();
            Assert.AreEqual( ticketTypeNames.Length, count );
        }

        [Test]
        public void チケット分類の各項目を取得する()
        {
            int count = 10;
            string ticketTypeNameBase = "Test";

            // 作成
            for ( int i = 0; i < count; ++i ) {
                string ticketTypeName = ticketTypeNameBase + i.ToString();
                TicketType ticketType = new TicketType();
                ticketType.Value = i.ToString();
                ticketType.Create( ticketTypeName );
            }

            // 作成数を確認
            string[] ticketTypeNames = TicketType.GetAll();
            Assert.AreEqual( ticketTypeNames.Length, count );

            for ( int i = 0; i < count; ++i ) {
                string ticketTypeName = ticketTypeNameBase + i.ToString();
                TicketType ticketType = new TicketType( ticketTypeName );

                // 作成したものと取得したものが一致する
                Assert.AreEqual( ticketType.Name, ticketTypeName );
                Assert.AreEqual( ticketType.Value, i.ToString() );
            }
        }

        [Test]
        public void チケット分類を作成する()
        {
            string[] before = TicketType.GetAll();

            // 作成
            string ticketTypeName = "Test";
            TicketType ticketType = new TicketType();
            ticketType.Value = "10";
            ticketType.Create( ticketTypeName );

            // チケット分類の数が増えていること
            Assert.AreEqual( before.Length + 1, TicketType.GetAll().Length );

            // 取得したチケット分類と一致すること
            TicketType getSeverity = new TicketType( ticketTypeName );
            Assert.AreEqual( ticketType.Name, getSeverity.Name );
            Assert.AreEqual( ticketType.Value, getSeverity.Value );
        }

        [Test]
        public void チケット分類を更新する()
        {
            string[] before = TicketType.GetAll();

            // 作成
            string ticketTypeName = "Test";
            TicketType ticketType = new TicketType();
            ticketType.Value = "10";
            ticketType.Create( ticketTypeName );

            // チケット分類の数が増えていること
            Assert.AreEqual( before.Length + 1, TicketType.GetAll().Length );

            // 更新
            TicketType getSeverity = new TicketType( ticketTypeName );
            getSeverity.Value = "11";
            getSeverity.Update();

            // 更新したチケット分類と一致すること
            TicketType updateSeverity = new TicketType( ticketTypeName );
            Assert.AreEqual( getSeverity.Name, updateSeverity.Name );
            Assert.AreEqual( getSeverity.Value, updateSeverity.Value );
        }

        [Test]
        public void チケット分類を削除する()
        {
            string[] before = TicketType.GetAll();

            // 作成
            string ticketTypeName = "Test";
            TicketType ticketType = new TicketType();
            ticketType.Value = "10";
            ticketType.Create( ticketTypeName );

            // チケット分類の数が増えていること
            Assert.AreEqual( before.Length + 1, TicketType.GetAll().Length );

            ticketType.Delete();

            // チケット分類の数が戻っていること
            Assert.AreEqual( before.Length, TicketType.GetAll().Length );
        }

        [Test]
        public void チケット分類を削除するstatic版()
        {
            string[] before = TicketType.GetAll();

            // 作成
            string ticketTypeName = "Test";
            TicketType ticketType = new TicketType();
            ticketType.Value = "10";
            ticketType.Create( ticketTypeName );

            // チケット分類の数が増えていること
            Assert.AreEqual( before.Length + 1, TicketType.GetAll().Length );

            TicketType.Delete( ticketTypeName );

            // チケット分類の数が戻っていること
            Assert.AreEqual( before.Length, TicketType.GetAll().Length );
        }

        [Test]
        [Ignore]
        public void 同じ順序のチケット分類を作成すると例外が発生する()
        {
            ///
            /// 今のところ例外が発生せず正常に登録されてしまう
            ///

            // 作成
            string ticketTypeName = "Test";
            TicketType TicketType = new TicketType();
            TicketType.Value = "10";
            TicketType.Create( ticketTypeName );

            TicketType.Create( ticketTypeName + "2" );

            // 同じ順序のチケット分類を作成すると例外が発生する
            Assert.Fail();
        }

        [Test]
        [ExpectedException( typeof( XmlRpcFaultException ) )]
        public void 同じ名称のチケット分類を作成すると例外が発生する()
        {
            // 作成
            string ticketTypeName = "Test";
            TicketType TicketType = new TicketType();
            TicketType.Value = "10";
            TicketType.Create( ticketTypeName );

            TicketType.Value = "11";
            TicketType.Create( ticketTypeName );

            // 同じ名称のチケット分類を作成すると例外が発生する
            Assert.Fail();
        }

        [Test]
        [ExpectedException( typeof( XmlRpcFaultException ) )]
        public void 登録されていない名称のチケット分類を取得すると例外が発生する()
        {
            TicketType TicketType = new TicketType( "存在しないチケット分類" );

            // 登録されていない名称のチケット分類を取得すると例外が発生する
            Assert.Fail();
        }

        [Test]
        [ExpectedException( typeof( XmlRpcFaultException ) )]
        public void 登録されていない名称のチケット分類を更新すると例外が発生する()
        {
            TicketType TicketType = new TicketType();
            TicketType.Update();

            // 登録されていない名称のチケット分類を更新すると例外が発生する
            Assert.Fail();
        }

        [Test]
        [ExpectedException( typeof( XmlRpcFaultException ) )]
        public void 登録されていない名称のチケット分類を削除すると例外が発生する()
        {
            TicketType.Delete( "存在しないチケット分類" );

            // 登録されていない名称のチケット分類を削除すると例外が発生する
            Assert.Fail();
        }
    }
}
