# -*- encoding: utf-8 -*-
#   Copyright 2008 Agile42 GmbH, Berlin (Germany)
#   Copyright 2007 Andrea Tomasini <andrea.tomasini_at_agile42.com>
#
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.
#   
#   Author: 
#       - Andrea Tomasini <andrea.tomasini_at_agile42.com>

from datetime import datetime
from random import randint
import unittest

from trac.util.datefmt import FixedOffset, utc

from agilo.core import Field, PersistentObject, PersistentObjectManager, \
    Relation
from agilo.test import TestEnvHelper

# Test PO Object
class MyPO(PersistentObject):
    class Meta(object):
        name = Field(primary_key=True)
        users = Field(type='number')
        amount = Field(type='real')
        description = Field()


class MyPOwithoutPK(PersistentObject):
    class Meta(object):
        name = Field(unique=True)
        description = Field()


class TestCore(unittest.TestCase):
    """Tests agilo core"""
    
    def setUp(self):
        """Setup defines a dummy object and creates an environment"""
        self.teh = TestEnvHelper()
        self.env = self.teh.get_env()
        self.pom = PersistentObjectManager(self.env)
        #FIXME: I don't know why, but putting the test to check if the table
        # is created, after a while gives error. I didn't have the chance to 
        # check it through, but I guess it is related to the :memory: db that
        # SQLite reallocate in the same position, so the new Env (which has 
        # a different Python id) is till pointing to the old InMemoryDatabase
        self.pom.create_table(MyPO)
        self.assertTrue(self.pom.create_table(MyPOwithoutPK))
    
    def tearDown(self):
        self.teh.cleanup()
        self.env= None
    
    def testDefineAndCreatePersistentObject(self):
        """Tests definition and creation of a PersistentObject"""
        class DummyPersistentObject(PersistentObject):
            class Meta(object):
                name = Field(primary_key=True)
                number = Field(type='real')
        
        self.assertTrue(self.pom.create_table(DummyPersistentObject))
        
        # Now instantiate the object
        dpo = DummyPersistentObject(self.env, name='Test', number=2.5)
        self.assertTrue(hasattr(dpo, 'number'))
        self.assertEqual(dpo.number, 2.5)
        self.assertTrue(hasattr(dpo, 'name'))
        self.assertEqual(dpo.name, 'Test')
        # Save the object
        self.assertTrue(dpo.save())
        dpo_copy = DummyPersistentObject(self.env, name='Test')
        self.assertTrue(hasattr(dpo_copy, 'name'))
        self.assertEqual(dpo_copy.name, 'Test')
        self.assertTrue(hasattr(dpo_copy, 'number'))
        self.assertEqual(dpo_copy.number, 2.5)
        
    def testPersistentObjectExistence(self):
        """Tests the persistent object exists method"""
        myPO = MyPO(self.env)
        myPO.name = 'This is my first Po'
        myPO.users = 3
        self.assertTrue(myPO.save())
        self.assertTrue(myPO.exists)
        
        myPO2 = MyPO(self.env)
        self.assertFalse(myPO2.exists)
        
    def testPersistenceWithoutPK(self):
        """Tests the persistent object without the PK"""
        myPO = MyPOwithoutPK(self.env, name='Test')
        self.assertTrue(hasattr(myPO, '_id'))
        self.assertTrue(myPO.save())
        myPOreloaded = MyPOwithoutPK(self.env, name='Test')
        self.assertTrue(myPOreloaded.exists)
        
    def testDeletePersistentObjectWithPK(self):
        """Tests the deletion of a persistent object with PK from the DB"""
        myObj = MyPO(self.env, name='Test', users=2)
        self.assertTrue(myObj.save())
        # Now delete it
        self.assertTrue(myObj.delete())
        myReborn = MyPO(self.env, name='Test')
        self.assertFalse(myReborn.exists)
        
    def testDeletePersistentObjectWithoutPK(self):
        """Tests the deletion of a persistent object without PK from the DB"""
        myObj = MyPOwithoutPK(self.env, name='Test')
        self.assertTrue(myObj.save())
        # Now delete it
        self.assertTrue(myObj.delete())
        
    def testSelectPersistentObject(self):
        """Tests the select of the persistent object"""
        myObj1 = MyPO(self.env, name="Obj1", users=2)
        self.assertTrue(myObj1.save())
        myObj2 = MyPO(self.env, name="Obj2", users=4)
        self.assertTrue(myObj2.save())
        myObj3 = MyPO(self.env, name="Obj3", users=6)
        self.assertTrue(myObj3.save())
        
        objs = MyPO.select(self.env, criteria={'name': 'Obj1'})
        self.assertEqual(len(objs), 1)
        self.assertEqual(objs[0].name, 'Obj1')
        
        objs = MyPO.select(self.env, criteria={"users": "> 3"})
        self.assertEqual(len(objs), 2)
        for obj in objs:
            self.assertTrue(obj.users > 3)
        
        objs = MyPO.select(self.env)
        self.assertEqual(len(objs), 3)
        
    def testPersistentObjectUpdate(self):
        """Tests the Persistent Object Update"""
        myObj = MyPO(self.env, name="Test", users=4)
        self.assertTrue(myObj.save())
        # Now update
        myObj.users = 5
        self.assertTrue(myObj.save())
        # Now change the Primary Key
        myObj.name = "New Test"
        self.assertTrue(myObj.save())
        # Now myObj Test should not exists anymore
        myOldObj = MyPO(self.env, name="Test")
        self.assertFalse(myOldObj.exists, "Created new object instead of updating...")
        
    def checkInstanceVariableAndClassVariable(self):
        """Tests the setting and getting of instance and class variables"""
        myObj1 = MyPO(self.env, name="Test1", users=3)
        self.assertTrue(myObj1.save())
        myObj2 = MyPO(self.env, name="Test2", users=6)
        self.assertTrue(myObj2.save())
        self.assertNotEqual(myObj1.name, myObj2.name)
        self.assertNotEqual(myObj1.users, myObj2.users)
        
    def testSettingOfKeyFieldsToNone(self):
        """Test that the Persistent Object doesn't allow to set key or unique fields to None"""
        myObj = MyPO(self.env, name="Test", users=2)
        self.assertTrue(myObj.save())
        myObj.name = None
        self.assertNotEqual(myObj.name, None)
        
    def testRelationBetweenTwoDifferentPo(self):
        """Tests the relation between 2 different POs"""
        class MyRelatedPO(PersistentObject):
            class Meta(object):
                name = Field(primary_key=True)
                mypo = Relation(MyPO)
        
        self.assertTrue(self.pom.create_table(MyRelatedPO))
        
        myObj = MyPO(self.env, name="Test", users=4)
        self.assertTrue(myObj.save())
        myRelObj = MyRelatedPO(self.env, name="Related Test", mypo=myObj)
        self.assertTrue(myRelObj.save())
        # Test the member is still the object
        self.assertEqual(myRelObj.mypo, myObj)
        # Now reload the object from the database and check if it is an object and is the same
        myRelCopy = MyRelatedPO(self.env, name=myRelObj.name)
        self.assertTrue(isinstance(myRelCopy.mypo, MyPO), "Object not converted! %s" % myRelCopy.mypo)
        self.assertEqual(myRelCopy.mypo.name, myObj.name)
        self.assertEqual(myRelCopy.mypo.users, myObj.users)
        
    def testSettingWrongTypeOnRelation(self):
        """Tests if the Relation is type safe"""
        class MyDummyPO(PersistentObject):
            class Meta(object):
                name = Field(unique=True)
        
        class MyRelatedPO(PersistentObject):
            class Meta(object):
                name = Field(primary_key=True)
                mypo = Relation(MyPO)
        
        self.assertTrue(self.pom.create_table(MyDummyPO))
        self.assertTrue(self.pom.create_table(MyRelatedPO))
        
        myObj = MyPO(self.env, name="Test", users=3)
        self.assertTrue(myObj.save())
        self.assertEqual(myObj.name, "Test")
        self.assertEqual(myObj.users, 3)
        myRelObj = MyRelatedPO(self.env, name="Related Test", mypo=myObj)
        self.assertTrue(myRelObj.save())
        # Test the member is still the object
        self.assertEqual(myRelObj.mypo, myObj)
        # Now create a dummy object and try to set the mypo
        myDummy = MyDummyPO(self.env, name="Dummy")
        self.assertTrue(myDummy.save())
        self.assertEqual(myDummy.name, "Dummy")
        self.assertNotEqual(myDummy, myObj)
        myRelObj.mypo = myDummy
        self.assertNotEqual(myRelObj.mypo, myDummy)
        
    def testSettingNoneOnRelation(self):
        """Tests if the Relation is accepting None"""
        class MyRelatedPO(PersistentObject):
            class Meta(object):
                name = Field(primary_key=True)
                mypo = Relation(MyPO)
        
        self.assertTrue(self.pom.create_table(MyRelatedPO))
        
        myObj = MyPO(self.env, name="Test", users=3)
        self.assertTrue(myObj.save())
        myRelObj = MyRelatedPO(self.env, name="Related Test", mypo=myObj)
        self.assertTrue(myRelObj.save())
        # Test the member is still the object
        self.assertEqual(myRelObj.mypo, myObj)
        # Now set None
        myRelObj.mypo = None
        self.assertTrue(myRelObj.save())
        # Reload
        myRelObj = MyRelatedPO(self.env, name="Related Test")
        self.assertEqual(myRelObj.mypo, None)
    
    def testSettingNoneOnRelationWhenPKIsNumber(self):
        """Tests setting the PK as None on a relation"""
        class MyPOWithNumberId(PersistentObject):
            class Meta(object):
                id = Field(type='number', primary_key=True)
                users = Field(type='number')
        
        class MyRelatedPO(PersistentObject):
            class Meta(object):
                name = Field(primary_key=True)
                mypo = Relation(MyPOWithNumberId)
        
        self.assertTrue(self.pom.create_table(MyPOWithNumberId))
        self.assertTrue(self.pom.create_table(MyRelatedPO))
        
        myObj = MyPOWithNumberId(self.env, id=1, users=3)
        self.assertTrue(myObj.save())
        myRelObj = MyRelatedPO(self.env, name="Related Test", mypo=myObj)
        self.assertTrue(myRelObj.save())
        # Test the member is still the object
        self.assertEqual(myRelObj.mypo, myObj)
        # Now set None
        myRelObj.mypo = None
        self.assertTrue(myRelObj.save())
        # Reload
        myRelObj = MyRelatedPO(self.env, name="Related Test")
        self.assertEqual(myRelObj.mypo, None)
        
    def testSelectWithNoneParameter(self):
        """Tests the select with a parameter None, also on relations"""
        class MyRelatedPO(PersistentObject):
            class Meta(object):
                name = Field(primary_key=True)
                mypo = Relation(MyPO)
        
        self.assertTrue(self.pom.create_table(MyRelatedPO))
        
        myObj = MyPO(self.env, name="Test", users=3)
        self.assertTrue(myObj.save())
        myRelObj = MyRelatedPO(self.env, name="Related Test", mypo=myObj)
        self.assertTrue(myRelObj.save())
        
        # Select MyPO where name is None
        res = MyPO.select(self.env, criteria={'name': None})
        self.assertEqual(len(res), 0)
        
        # Select with relation None
        myRelObj.mypo = None
        self.assertTrue(myRelObj.save())
        MyRelatedPO.select(self.env, criteria={'mypo': None})
    
    def testInsertEscapes(self):
        """Tests if strings containing quotes can be saved and loaded correctly."""
        description="""
                     This is a quote: ' And another "
                     And lots: '', ''', ""
                     ; drop table grades;
        """
        myObj = MyPO(self.env, name='ObjWithDesc', description=description)
        self.assertTrue(myObj.save())
        myObj = MyPO(self.env, name='ObjWithDesc')
        self.assertEquals(myObj.description, description)
        
    def testPersistentObjectComparison(self):
        """Tests the compare between persistent objects"""
        myObj1 = MyPO(self.env, name="TestPOComparison", description="Test me too")
        self.assertTrue(myObj1.save())
        myObj2 = MyPO(self.env, name="TestPOComparison")
        self.assertEquals(myObj1, myObj2)
        
    def testPersistentObjectRealConversion(self):
        """Tests the conversion of real to float of a Persistent Object"""
        myObj1 = MyPO(self.env, name="TestPO", amount=5.6)
        self.assertTrue(myObj1.save())
        myObj2 = MyPO(self.env, name="TestPO")
        self.assertEquals(myObj1, myObj2)
        
    def _assert_po_behaves_as_expected(self, po_class):
        myPO = po_class(self.env)
        myPO.name = 'John Smith'
        myPO.foobar = 'should be ignored'
        self.assertTrue(myPO.save())
        self.assertTrue(myPO.exists)
        
        po = po_class(self.env, name='John Smith')
        self.assertEquals('John Smith', po.name)
        self.assertFalse(hasattr(po, 'foobar'))
        self.assertTrue(myPO.save())
        
        myPO.delete()
        po = po_class(self.env, name='John Smith')
        self.assertFalse(po.exists)
    
    def testDifferentColumnNameInDb(self):
        """Tests that a different db name can be chosen for a column."""
        self._assert_po_behaves_as_expected(MyPO)
    
    def testDifferentColumnNameForPrimaryKeyInDb(self):
        """Tests that a different db name can be chosen for a column which is
        the primary key."""
        class MyPOKey(PersistentObject):
            class Meta(object):
                name = Field(primary_key=True, db_name='mypo_key')
                what = Field()
        self.pom.create_table(MyPOKey)
        self._assert_po_behaves_as_expected(MyPOKey)
    
    def testDifferentColumnNameForUniqueKeyInDb(self):
        """Tests that a different db name can be chosen for a column which is
        marked as unique."""
        class MyPOUnique(PersistentObject):
            class Meta(object):
                name = Field(unique=True, db_name='mypo_key')
                another = Field()
        self.pom.create_table(MyPOUnique)
        self._assert_po_behaves_as_expected(MyPOUnique)
    
    def testSelectWithDifferentColumnNameInDb(self):
        """Tests that a select on a key works even if the db column name is
        different from the Python attribute name."""
        class MyPODiffColName(PersistentObject):
            class Meta(object):
                name = Field(primary_key=True, db_name='foobar')
                
        self.assertEqual('foobar', MyPODiffColName.name.db_name)
        self.assertTrue(self.pom.create_table(MyPODiffColName))
        
        myPO = MyPODiffColName(self.env)
        myPO.name = 'John Smith'
        self.assertTrue(myPO.save())
        self.assertTrue(myPO.exists)
        
        names = [s.name for s in MyPODiffColName.select(self.env)]
        self.assertEqual(['John Smith'], names)
    
    def testRenamePrimaryKeyWithDifferentColumnNameInDb(self):
        """Tests that the primary key can still be renamed even if the column
        name is different from the attribute name."""
        myPO = MyPO(self.env)
        myPO.name = 'John Smith'
        self.assertTrue(myPO.save())
        self.assertTrue(myPO.exists)
        
        myPo2 = MyPO(self.env, name='John Smith')
        myPo2.name = 'John Doe'
        myPo2.save()
        
        names = [s.name for s in MyPO.select(self.env)]
        self.assertEqual(['John Doe'], names)
        
    def testOrderByInSelectQuery(self):
        """Tests the orderby in select query"""
        for i in range(10):
            myPo = MyPO(self.env, name='test%s' % i, users=randint(0, 30))
            myPo.save()
        
        myPos = MyPO.select(self.env, order_by=['users'])
        self.assertEqual(10, len(myPos), 
                         "Found %s items instead of 10!" % len(myPos))
        
        # should be sorted ascending
        last = 0
        for myPo in myPos:
            if last == 0:
                last = myPo.users
            self.assertTrue(myPo.users >= last, 
                            "Items out of order: %s !>= %s" % \
                            (myPo.users, last))
            last = myPo.users
        
        myPos = MyPO.select(self.env, order_by=['-users'])
        self.assertEqual(10, len(myPos), 
                         "Found %s items instead of 10!" % len(myPos))
        # should be sorted descending
        last = 0
        for myPo in myPos:
            if last == 0:
                last = myPo.users
            self.assertTrue(myPo.users <= last, 
                            "Items out of order: %s !<= %s" % \
                            (myPo.users, last))
            last = myPo.users

    def testOrderByInSelectQueryWithDbFieldsName(self):
        """Tests the order_by in select query with specific db fields"""
        for i in range(10):
            myPo = MyPO(self.env, name='test%s' % i, users=randint(0, 30))
            myPo.save()
        
        myPos = MyPO.select(self.env, order_by=['users'])
        self.assertEqual(10, len(myPos), 
                         "Found %s items instead of 10!" % len(myPos))
        
        # should be sorted ascending
        last = 0
        for myPo in myPos:
            if last == 0:
                last = myPo.users
            self.assertTrue(myPo.users >= last, 
                            "Items out of order: %s !>= %s" % \
                            (myPo.users, last))
            last = myPo.users
        
        myPos = MyPO.select(self.env, order_by=['-users'])
        self.assertEqual(10, len(myPos), 
                         "Found %s items instead of 10!" % len(myPos))
        # should be sorted descending
        last = 0
        for myPo in myPos:
            if last == 0:
                last = myPo.users
            self.assertTrue(myPo.users <= last, 
                            "Items out of order: %s !<= %s" % \
                            (myPo.users, last))
            last = myPo.users

    def testLimitInPOQuery(self):
        """Tests the limit in the PersistentObject query"""
        for i in range(10):
            myPo = MyPO(self.env, name='test%s' % i, users=randint(0, 30))
            myPo.save()
        
        myPos = MyPO.select(self.env, limit=3)
        self.assertEqual(3, len(myPos), 
                         "Found %s items instead of 3!" % len(myPos))

        myPos = MyPO.select(self.env)
        self.assertEqual(10, len(myPos), 
                         "Found %s items instead of 10!" % len(myPos))

    def testNotInAndInSelectInPoQuery(self):
        """Tests the criteria 'in' and 'not in' a list of values"""
        some_pos = list()
        for i in range(10):
            myPo = MyPO(self.env, name='test%s' % i, users=i)
            myPo.save()
            if i % 2:
                some_pos.append(myPo.users)
        
        all_pos = MyPO.select(self.env)
        self.assertEqual(10, len(all_pos))
        # Now make a query and get all the Pos which are not included in the list
        in_pos = MyPO.select(self.env, criteria={'users': 'in %s' % some_pos})
        self.assertEqual(len(some_pos), len(in_pos))
        # negative
        not_pos = MyPO.select(self.env, criteria={'users': 'not in %s' % some_pos})
        self.assertEqual(10 - len(some_pos), len(not_pos))
        
        # now try with stings
        class MyStrPO(PersistentObject):
            class Meta(object):
                name = Field(primary_key=True) # , db_name='foobar'
        
        self.assertTrue(self.pom.create_table(MyStrPO))
        
        some_pos = list()
        for i in range(10):
            myPo = MyStrPO(self.env, name='test%s' % i)
            myPo.save()
            if i % 2:
                some_pos.append(myPo.name)
        
        all_pos = MyStrPO.select(self.env)
        self.assertEqual(10, len(all_pos))
        # Now make a query and get all the Pos which are not included in the list
        in_pos = MyStrPO.select(self.env, criteria={'name': 'in %s' % some_pos})
        self.assertEqual(len(some_pos), len(in_pos))
        # negative
        not_pos = MyStrPO.select(self.env, criteria={'name': 'not in %s' % some_pos})
        self.assertEqual(10 - len(some_pos), len(not_pos))
    
    def testPODoesNotLoadFromDatabaseIfNoKeyGiven(self):
        class MyPO(PersistentObject):
            class Meta(object):
                name = Field(primary_key=True)
        # Check that name=None is the same as no name parameter given
        po = MyPO(self.env, name=None)
        self.assertFalse(po.exists)

    def testAsDictUtilityMethod(self):
        """
        Tests the as_dict to transform a PersistentObject into plain
        Python Object
        """
        class MyPO2(PersistentObject):
            class Meta(object):
                name = Field(unique=True)
                prop2 = Relation(MyPO)
        
        self.assertTrue(self.pom.create_table(MyPO2))
        
        myPO = MyPO(self.env, name='TestPO', users=2)
        myPO.save()
        myPO2 = MyPO2(self.env, name='testDictPO', prop2=myPO)
        myPO2.save()
        
        po2_dict = myPO2.as_dict()
        self.assertEqual('testDictPO', po2_dict['name'])
        self.assertEqual(myPO.as_dict(), po2_dict['prop2'])
        po_dict = myPO.as_dict()
        self.assertEqual('TestPO', po_dict['name'])
        self.assertEqual(2, po_dict['users'])
    
    def testDBConvertingData(self):
        """Tests the DB layer converting data of its own."""
        class MyTestPo(PersistentObject):
            class Meta:
                value0 = Field(unique=True)
                value1 = Field(type="real")
                value2 = Field(type="integer")
                value3 = Field()
        
        self.pom.create_table(MyTestPo)
        
        myt = MyTestPo(self.env, value0='TestConversion')
        self.assertTrue(myt.save())
        self.assertEqual(None, myt.value1)
        self.assertEqual(None, myt.value2)
        self.assertEqual(None, myt.value3)
        self.assertEqual(None, myt._get_value_of_field(MyTestPo.value1.field, myt))
        self.assertEqual(None, myt._get_value_of_field(MyTestPo.value2.field, myt))
        self.assertEqual(None, myt._get_value_of_field(MyTestPo.value3.field, myt))
        
        # check what is into the DB
        db = self.env.get_db_cnx()
        cursor = db.cursor()
        cursor.execute('SELECT value1,value2,value3 FROM %s WHERE _id=%s' % \
                       (MyTestPo._table.name, myt._id))
        row = cursor.fetchone()
        self.assertEqual(None, row[0])
        self.assertEqual(None, row[1])
        self.assertEqual(None, row[2])
        
        # Now see what comes back from the DB
        myt1 = MyTestPo(self.env, value0='TestConversion')
        self.assertEqual(None, myt1._get_value_of_field(MyTestPo.value1.field, myt1))
        self.assertEqual(None, myt1._get_value_of_field(MyTestPo.value2.field, myt1))
        self.assertEqual(None, myt1._get_value_of_field(MyTestPo.value3.field, myt1))
        self.assertEqual(None, myt1.value1)
        self.assertEqual(None, myt1.value2)
        self.assertEqual(None, myt1.value3)
    
    def testPODateTimeIsSavedAsUTC(self):
        # Actually we don't have to implement any code - trac stores datetime
        # internally always UTC as already.
        class MyDatetimePO(PersistentObject):
            class Meta(object):
                id = Field(primary_key=True, type='integer')
                start = Field(type='datetime')
        self.pom.create_table(MyDatetimePO)
        
        pdt = FixedOffset(-7*60, 'PDT')
        start = datetime(2008, 10, 4, 12, 42, tzinfo=pdt)
        # utc_timestamp = 1223149320
        
        po = MyDatetimePO(self.env, id=1, start=start)
        po.save()
        
        another_po = MyDatetimePO(self.env, id=1)
        self.assertEqual(start.astimezone(utc), another_po.start)
    
    def testOldValuesAreSetPerInstance(self):
        """This is a bugfix test to check that the _old values for persistent
        objects are set per instance and not on a class level."""
        class MyPOWithOld(PersistentObject):
            class Meta(object):
                name = Field(primary_key=True)
        
        env = self.teh.get_env()
        PersistentObjectManager(env).create_table(MyPOWithOld)
        
        obj1 = MyPOWithOld(env, name='foo')
        obj1.save()
        
        obj2 = MyPOWithOld(env, name='bar')
        obj2.save()
        
        obj2.name = 'baz'
        self.assertEqual({'name': 'bar'}, obj2._old)
        self.assertEqual({'name': None}, obj1._old)
    
    def testCanSelectWithCriteriaNone(self):
        """This is a bugfix test to check that the _old values for persistent
        objects are set per instance and not on a class level."""
        class MyPOWithNone(PersistentObject):
            class Meta(object):
                name = Field(primary_key=True)
                value = Field()
        
        PersistentObjectManager(self.env).create_table(MyPOWithNone)
        
        obj1 = MyPOWithNone(self.env, name='foo')
        obj1.save()
        
        results = obj1.select(self.env, criteria={'value': None})
        self.assertEqual(1, len(results))


if __name__ == '__main__':
    suite = unittest.TestLoader().loadTestsFromTestCase(TestCore)
#    suite = unittest.TestSuite()
#    suite.addTest(TestCore('testNotInAndInSelectInPoQuery'))
    unittest.TextTestRunner(verbosity=0).run(suite)
