/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.main;

import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Enumeration;

import javax.swing.JComponent;
import javax.swing.JTable;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.TableColumnModelEvent;
import javax.swing.event.TableColumnModelListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

/**
 * JTable のカラム幅をデータ内容に応じて自動調整します。<br>
 * JTable構築時の静的データ内容で設定する方法と、 テータモデルのセル値変更や行挿入イベントに応じて動的な内容で設定する方法を提供します。<br>
 * 
 * @author sima
 * 
 */
public class TableColumnFit implements TableColumnModelListener, TableModelListener, MouseMotionListener,
        PropertyChangeListener {

	private final JTable jTable;
	private boolean autoFit = false;

	public TableColumnFit(JTable jTable) {
		this.jTable = jTable;
		this.jTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
	}

	/**
	 * カラム自動調整の実施可否状態
	 * 
	 * @return autoFit
	 */
	public boolean isAutoFit() {
		return autoFit;
	}

	/**
	 * カラム自動調整を中断または再開させる。EDTの中から実行してください。
	 * 
	 * @param autoFit
	 *            false の時は調整を中断する、trueの時は調整を実施する。初期値はtrue。
	 */
	public void setEnableAutoFit(boolean autoFit) {
		if (autoFit) {
			// JTableの各column幅を、列の最大値幅に設定します。
			// 最大幅は見出し/セルをフルスキャンする事で求めます。
			setPreferredWidthFromHeader();
			for (int row = 0; row < jTable.getModel().getRowCount(); row++) {
				setPreferredSizeFromRoW(row);
			}

			// 列幅変更したので画面表示を更新する。
			jTable.doLayout();

			// 値が変わった時だけリスナー登録
			if (this.autoFit != autoFit) {
				JTableHeader jTableHeader = jTable.getTableHeader();
				// Drag%Dropで列幅変更が行われた場合、自動調整を停止させたいので、
				// Jtableヘッダーのマウスモーションを監視する。
				if (jTableHeader != null) {
					jTableHeader.addMouseMotionListener(this);
				}

				// リアルタイムな行追加変更に応じて、列幅を変更したいので、
				// テーブルモデルの値変更を監視する。
				// 但し、テーブルモデルはabstractTableModelを実装し、fireXXXで変更を通知しなけらばならない。
				jTable.getModel().addTableModelListener(this);
				if (jTableHeader != null) {
					jTableHeader.getColumnModel().addColumnModelListener(this);
				}

				// Fontが変更された時再計算したいので、
				// "font"のプロパティ変更を監視する。
				jTable.addPropertyChangeListener("font", this);
			}
		} else {
			// 自動調整停止時は、監視リスナーを削除する。
			jTable.getTableHeader().removeMouseMotionListener(this);
			jTable.getModel().removeTableModelListener(this);
			jTable.getTableHeader().getColumnModel().removeColumnModelListener(this);
			jTable.removePropertyChangeListener(this);
		}
		this.autoFit = autoFit;
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see java.awt.event.MouseMotionListener#mouseDragged(java.awt.event.MouseEvent )
	 */
	@Override
	public void mouseDragged(MouseEvent e) {
		// 列ヘッダーのDrag&Dropが行われたら自動列幅調整は停止する。
		setEnableAutoFit(false);
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see javax.swing.event.TableColumnModelListener#columnAdded(javax.swing.event .TableColumnModelEvent)
	 */
	@Override
	public void columnAdded(TableColumnModelEvent e) {
		if (!autoFit) {
			return;
		}
		setPreferredWidthFromHeader();
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @seejavax.swing.event.TableModelListener#tableChanged(javax.swing.event. TableModelEvent)
	 */
	@Override
	public void tableChanged(TableModelEvent event) {
		if (!autoFit) {
			return;
		}
		switch (event.getType()) {
		case TableModelEvent.UPDATE:

			if (event.getFirstRow() == TableModelEvent.HEADER_ROW) {
				// AbstractTableModel#fireTableStructureChanged の時らしいが、
				// DefaultTableColumnModel では、カラム数=0 で呼ばれる為、その時は意味の無い処理となる。
				// 但し、JTable#setModel の時も呼ばれるがので、その時意味がある。
				setPreferredWidthFromHeader();
			} else {
				int cnt = jTable.getModel().getRowCount();
				if (event.getLastRow() < cnt) {
					cnt = event.getLastRow();
				}
				for (int row = event.getFirstRow(); row < cnt; row++) {
					setPreferredSizeFromRoW(row);
				}
			}
			break;

		case TableModelEvent.INSERT:

			for (int row = event.getFirstRow(); row <= event.getLastRow(); row++) {
				setPreferredSizeFromRoW(row);
			}
			break;

		default:
			break;
		}
	}

	/**
	 * JTableのヘッダcolumn内容でセル幅を設定します。
	 * 
	 * @param jTable
	 *            column幅を変更したいJTable
	 */
	private void setPreferredWidthFromHeader() {
		JTableHeader tableHeader = jTable.getTableHeader();
		TableCellRenderer defrenderer = null;
		if (tableHeader != null) {
			defrenderer = tableHeader.getDefaultRenderer();
		}
		TableColumnModel tcm = jTable.getColumnModel();

		Enumeration<TableColumn> e = tcm.getColumns();
		while (e.hasMoreElements()) {
			TableColumn tableColumn = e.nextElement();

			TableCellRenderer r = tableColumn.getCellRenderer();
			if (r == null) {
				// 個別設定が無いのでデフォルトを使用する
				if (defrenderer != null) {
					r = defrenderer;
				} else {
					return;
				}
			}
			JComponent c = (JComponent) r.getTableCellRendererComponent(jTable,
			                                                            tableColumn.getHeaderValue(),
			                                                            false,
			                                                            false,
			                                                            -1,
			                                                            0);
			int w = preferredWidth(c);
			tableColumn.setPreferredWidth(w);
		}
		jTable.setRowHeight(1);
	}

	/**
	 * JTableの行column内容でセル幅と高さを設定します。
	 * 
	 * @param row
	 *            column幅高さを変更したい行
	 */
	private void setPreferredSizeFromRoW(int row) {
		for (int column = 0; column < jTable.getColumnCount(); column++) {
			TableColumn tableColumn = jTable.getColumnModel().getColumn(column);

			TableCellRenderer r = jTable.getCellRenderer(row, column);
			// cellの値を getValueAt で取得し、レンダラーにて表示後コンポーネントを作成する。
			// 作成したコンポーネントは、勿体無いけど捨ててしまう。
			// JTable#getValueAt は view座標指定なので、TableModel#getValueAt
			// でモデル座標で取得する。
			JComponent c = (JComponent) r.getTableCellRendererComponent(jTable,
			                                                            jTable.getModel().getValueAt(row, column),
			                                                            false,
			                                                            false,
			                                                            row,
			                                                            column);
			int w = preferredWidth(c);
			if (tableColumn.getPreferredWidth() < w) {
				tableColumn.setPreferredWidth(w);
			}

			int h = preferredHeight(c);
			if (jTable.getRowHeight() < h) {
				jTable.setRowHeight(h);
			}
		}
	}

	/**
	 * セルコンポーネントの幅を計算する
	 * 
	 * @param jComponent
	 *            セルコンポーネント
	 * @return セルコンポーネントの幅に余白幅を加算した値
	 */
	private int preferredWidth(JComponent jComponent) {
		int rtn = jComponent.getPreferredSize().width + jComponent.getInsets().left + jComponent.getInsets().right
		        + jTable.getIntercellSpacing().width;
		return rtn;
	}

	/**
	 * セルコンポーネントの高さを計算する
	 * 
	 * @param jComponent
	 *            セルコンポーネント
	 * @return セルコンポーネントの幅に余白幅を加算した値
	 */
	private int preferredHeight(JComponent jComponent) {
		int rtn = jComponent.getPreferredSize().height;
		return rtn;
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see java.beans.PropertyChangeListener#propertyChange(java.beans. PropertyChangeEvent)
	 */
	@Override
	public void propertyChange(PropertyChangeEvent evt) {
		setPreferredWidthFromHeader();
		for (int row = 0; row < jTable.getModel().getRowCount(); row++) {
			setPreferredSizeFromRoW(row);
		}
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see java.awt.event.MouseMotionListener#mouseMoved(java.awt.event.MouseEvent)
	 */
	@Override
	public void mouseMoved(MouseEvent e) {
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see javax.swing.event.TableColumnModelListener#columnMarginChanged(javax. swing.event.ChangeEvent)
	 */
	@Override
	public void columnMarginChanged(ChangeEvent e) {
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see javax.swing.event.TableColumnModelListener#columnMoved(javax.swing.event .TableColumnModelEvent)
	 */
	@Override
	public void columnMoved(TableColumnModelEvent e) {
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see javax.swing.event.TableColumnModelListener#columnRemoved(javax.swing. event.TableColumnModelEvent)
	 */
	@Override
	public void columnRemoved(TableColumnModelEvent e) {
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see javax.swing.event.TableColumnModelListener#columnSelectionChanged(javax .swing.event.ListSelectionEvent)
	 */
	@Override
	public void columnSelectionChanged(ListSelectionEvent e) {
	}

}
