/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.sqlite.jdbc;

import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import org.sqlite.schema.ColumnMetaData;
import org.sqlite.Statement;
import static org.sqlite.swig.SQLite3Constants.SQLITE_INTEGER;
import static org.sqlite.swig.SQLite3Constants.SQLITE_FLOAT;
import static org.sqlite.swig.SQLite3Constants.SQLITE_TEXT;
import static org.sqlite.swig.SQLite3Constants.SQLITE_BLOB;
import static org.sqlite.swig.SQLite3Constants.SQLITE_NULL;

/**
 *
 * @author calico
 */
public class JdbcResultSetMetaData implements ResultSetMetaData {
    private final Statement stmt;

    public JdbcResultSetMetaData(Statement stmt) {
        this.stmt = stmt;
    }
    
    // START implements
    public int getColumnCount() throws SQLException {
        return stmt.getColumnCount();
    }

    /**
     * 
     * @param column
     * @return
     * @throws java.sql.SQLException
     */
    public boolean isAutoIncrement(int column) throws SQLException {
        final ColumnMetaData meta = getColumnMetaData(column);
        return meta.isAutoIncrement;
    }

    /**
     * True when Collation Sequences is 'NOCASE'. When it is not 'NOCASE', false is returned. 
     * @param column
     * @return
     * @throws java.sql.SQLException
     */
    public boolean isCaseSensitive(int column) throws SQLException {
        final ColumnMetaData meta = getColumnMetaData(column);
        return "NOCASE".equalsIgnoreCase(meta.collationSequenceName);
    }

    /**
     * It always returns true.
     * @see org.sqlite.jdbc.JdbcDatabaseMetaData#getTypeInfo()
     * @param column
     * @return true
     * @throws java.sql.SQLException
     */
    public boolean isSearchable(int column) throws SQLException {
        validateColumnIndexRange(column);
        
        return true;
    }

    /**
     * It always returns false.
     * @param column
     * @return false
     * @throws java.sql.SQLException
     */
    public boolean isCurrency(int column) throws SQLException {
        validateColumnIndexRange(column);
        
        return false;
    }

    public int isNullable(int column) throws SQLException {
        final ColumnMetaData meta = getColumnMetaData(column);
        return (meta.isNotNull ? columnNoNulls : columnNullable);
    }

    /**
     * 
     * NOTE: Required to invoke the java.sql.ResultSet#next() method beforehand.
     * @param column
     * @return
     * @throws java.sql.SQLException
     * @see org.sqlite.jdbc.JdbcDatabaseMetaData#getTypeInfo()
     */
    public boolean isSigned(int column) throws SQLException {
        validateColumnIndexRange(column);

        return JdbcDatabaseMetaData.isSigned(stmt.getColumnType(column));
    }

    /**
     * 
     * NOTE: Required to invoke the java.sql.ResultSet#next() method beforehand.
     * @param column
     * @return
     * @throws java.sql.SQLException
     * @see org.sqlite.jdbc.JdbcDatabaseMetaData#getColumnDisplaySize(int)
     * @see org.hsqldb.Types#getMaxDisplaySize(int)
     */
    public int getColumnDisplaySize(int column) throws SQLException {
        validateColumnIndexRange(column);

        return JdbcDatabaseMetaData.getColumnDisplaySize(stmt.getColumnType(column));
    }

    /**
     * 
     * @param column
     * @return
     * @throws java.sql.SQLException
     */
    public String getColumnLabel(int column) throws SQLException {
        validateColumnIndexRange(column);
        
        return stmt.getColumnLabel(column);
    }

    public String getColumnName(int column) throws SQLException {
        validateColumnIndexRange(column);
        
        return stmt.getColumnName(column);
    }

    /**
     * It always returns empty string.
     * @param column
     * @return empty string
     * @throws java.sql.SQLException
     */
    public String getSchemaName(int column) throws SQLException {
        validateColumnIndexRange(column);
        
        return "";
    }

    /**
     * 
     * NOTE: Required to invoke the java.sql.ResultSet#next() method beforehand.
     * @param column
     * @return
     * @throws java.sql.SQLException
     * @see org.sqlite.jdbc.JdbcDatabaseMetaData#getTypeInfo()
     */
    public int getPrecision(int column) throws SQLException {
        validateColumnIndexRange(column);
        
        return JdbcDatabaseMetaData.getPrecision(stmt.getColumnType(column));
    }

    /**
     * It always returns 0.
     * NOTE: Required to invoke the java.sql.ResultSet#next() method beforehand.
     * @param column
     * @return 0
     * @throws java.sql.SQLException
     */
    public int getScale(int column) throws SQLException {
        validateColumnIndexRange(column);
        
        return JdbcDatabaseMetaData.getScale(stmt.getColumnType(column));
    }

    public String getTableName(int column) throws SQLException {
        validateColumnIndexRange(column);
        
        return stmt.getColumnTableName(column);
    }

    /**
     * It always returns empty string.
     * @param column
     * @return empty string
     * @throws java.sql.SQLException
     */
    public String getCatalogName(int column) throws SQLException {
        validateColumnIndexRange(column);
        
        return "";
    }

    /**
     * 
     * NOTE: Required to invoke the java.sql.ResultSet#next() method beforehand.
     * @param column
     * @return
     * @throws java.sql.SQLException
     */
    public int getColumnType(int column) throws SQLException {
        validateColumnIndexRange(column);

        return JdbcDatabaseMetaData.getColumnType(stmt.getColumnType(column));
    }

    public String getColumnTypeName(int column) throws SQLException {
        validateColumnIndexRange(column);
        
        return stmt.getColumnTypeName(column);
    }

    /**
     * It always returns true.
     * @param column
     * @return true
     * @throws java.sql.SQLException
     */
    public boolean isReadOnly(int column) throws SQLException {
        validateColumnIndexRange(column);
        
        return true;
    }

    /**
     * It always returns false.
     * @param column
     * @return false
     * @throws java.sql.SQLException
     */
    public boolean isWritable(int column) throws SQLException {
        validateColumnIndexRange(column);
        
        return false;
    }

    /**
     * It always returns false.
     * @param column
     * @return false
     * @throws java.sql.SQLException
     */
    public boolean isDefinitelyWritable(int column) throws SQLException {
        validateColumnIndexRange(column);
        
        return false;
    }

    /**
     * 
     * NOTE: Required to invoke the java.sql.ResultSet#next() method beforehand.
     * @param column
     * @return
     * @throws java.sql.SQLException
     */
    public String getColumnClassName(int column) throws SQLException {
        validateColumnIndexRange(column);

        return JdbcDatabaseMetaData.getColumnClassName(stmt.getColumnType(column));
    }
    // END implements

    protected void validateColumnIndexRange(int columnIndex) throws SQLException {
        if (columnIndex > getColumnCount() || columnIndex < 1) {
            throw new SQLException("Column index out of range.", "90J01");
        }
    }

    /**
     * 
     * NOTE: Required to invoke the java.sql.ResultSet#next() method beforehand.
     * @param column
     * @return
     * @throws java.sql.SQLException
     */
    public int getSQLiteColumnType(int column) throws SQLException {
        validateColumnIndexRange(column);

        return stmt.getColumnType(column);
    }
    
    protected ColumnMetaData getColumnMetaData(int columnIndex) throws SQLException {
        validateColumnIndexRange(columnIndex);
        
        final String tableName = getTableName(columnIndex);
        final String columnName = getColumnName(columnIndex);
        return stmt.getColumnMetaData(null, tableName, columnName);
    }
    
}
