/* File : sqlite3jni.i */
%module SQLite3
%{
#include <assert.h>
#include "sqlite3.h"

#ifdef __cplusplus
extern "C" {
#endif

/* Field ID of org.sqlite.callback.Callback._this member field */
static jfieldID fid_callback_this = 0;

/* Field ID of org.sqlite.callback.NamedCallback.name member field */
static jfieldID fid_namedcallback_name = 0;

/* org.sqlite.udf.Function class */
static jclass clsFunction = 0;

/* Field ID of org.sqlite.udf.Function member fields  */
static jfieldID fid_function_argc = 0;

/* org.sqlite.udf.AggregateFunction class */
static jclass clsAggregateFunction = 0;

/* org.sqlite.text.Collator class */
static jclass clsCollator = 0;

/* org.sqlite.auth.Authorizer class */
static jclass clsAuthorizer = 0;

/* org.sqlite.event.BusyHandler class */
static jclass clsBusyHandler = 0;

/* org.sqlite.event.ProgressHandler class */
static jclass clsProgressHandler = 0;

JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM *vm, void *reserved) {
    JNIEnv *jenv = NULL;
    // require JNI ver 1.2 or later
    if ((*vm)->GetEnv(vm, (void **)&jenv, JNI_VERSION_1_2) != JNI_OK) {
        return JNI_ERR;
    }
    return JNI_VERSION_1_2;
}

/* delete clsFunction and clsAggregateFunction */
JNIEXPORT void JNICALL JNI_OnUnload(JavaVM *vm, void *reserved) {
    JNIEnv *jenv = NULL;
    if ((*vm)->AttachCurrentThread(vm, (void **)&jenv, 0) == JNI_OK) {
        if (clsFunction) {
            (*jenv)->DeleteGlobalRef(jenv, clsFunction);
            clsFunction = 0;
        }
        if (clsAggregateFunction) {
            (*jenv)->DeleteGlobalRef(jenv, clsAggregateFunction);
            clsAggregateFunction = 0;
        }
        if (clsCollator) {
            (*jenv)->DeleteGlobalRef(jenv, clsCollator);
            clsCollator = 0;
        }
        if (clsAuthorizer) {
            (*jenv)->DeleteGlobalRef(jenv, clsAuthorizer);
            clsAuthorizer = 0;
        }
        if (clsBusyHandler) {
            (*jenv)->DeleteGlobalRef(jenv, clsBusyHandler);
            clsBusyHandler = 0;
        }
        if (clsProgressHandler) {
            (*jenv)->DeleteGlobalRef(jenv, clsProgressHandler);
            clsProgressHandler = 0;
        }
    }
}

/* throw new Exception */
static void JavaThrowException(JNIEnv *jenv, const char *clazz, const char *message) {
    if (jenv) {
        jclass ex;
        char *cls = (char *)clazz;
        char *msg = (char *)message;
        (*jenv)->ExceptionClear(jenv);
        do {
            ex = (*jenv)->FindClass(jenv, cls);
            if (ex) {
                (*jenv)->ThrowNew(jenv, ex, msg);
                (*jenv)->DeleteLocalRef(jenv, ex);
                return;
            }
            // exception class not found
            msg = cls;
            cls = "java/lang/ClassNotFoundException";
        } while (cls != msg);
    }
}

/* sqlite3_column_blob by byte[] */
JNIEXPORT jbyteArray JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1column_1blob_1by_1bytes(JNIEnv *jenv, jclass jcls, jlong stmt, jint col) {
    sqlite3_stmt *pStmt = *(sqlite3_stmt **)&stmt;
    jsize len;
    jbyteArray result;
    jbyte *pSrc;
    const void *blob = sqlite3_column_blob(pStmt, col);

    if (!blob) {
        return NULL;
    }

    len = sqlite3_column_bytes(pStmt, col);
    result = (*jenv)->NewByteArray(jenv, len);
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return NULL;
    }

    pSrc = (*jenv)->GetPrimitiveArrayCritical(jenv, result, 0);
    memcpy(pSrc, blob, len);
    (*jenv)->ReleasePrimitiveArrayCritical(jenv, result, pSrc, 0);

    return result;
}

/* sqlite3_value_blob by byte[] */
JNIEXPORT jbyteArray JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1value_1blob_1by_1bytes(JNIEnv *jenv, jclass jcls, jlong value) {
    sqlite3_value *pVal = *(sqlite3_value **)&value;
    jsize len;
    jbyteArray result;
    jbyte *pSrc;
    const void *blob = sqlite3_value_blob(pVal);

    if (!blob) {
        return NULL;
    }

    len = sqlite3_value_bytes(pVal);
    result = (*jenv)->NewByteArray(jenv, len);
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return NULL;
    }

    pSrc = (*jenv)->GetPrimitiveArrayCritical(jenv, result, 0);
    memcpy(pSrc, blob, len);
    (*jenv)->ReleasePrimitiveArrayCritical(jenv, result, pSrc, 0);

    return result;
}

/* loadl class */
static int LoadClass(JNIEnv *jenv, jclass *clazz, const char* name) {
	if (*clazz || (*clazz = (*jenv)->FindClass(jenv, name))) {
		return 1;
	}

	JavaThrowException(jenv, "java/lang/ClassNotFoundException", name);
    return 0;
}

/* load 'org.sqlite.callback.Callback' class */
static int LoadCallbackClass(JNIEnv *jenv) {
    if (fid_callback_this) {
        // alrealy loaded
        return 1;

	} else {
		jclass clsCallback = 0;
		if (LoadClass(jenv, &clsCallback, "org/sqlite/callback/Callback")) {
			fid_callback_this = (*jenv)->GetFieldID(jenv, clsCallback, "_this", "J");
			return 1;
		}
	}

    return 0;
}

/* load 'org.sqlite.callback.NamedCallback' class */
static int LoadNamedCallbackClass(JNIEnv *jenv) {
    if (fid_namedcallback_name) {
        // alrealy loaded
        return 1;

	} else {
		jclass clsNamedCallback = 0;
		if (LoadCallbackClass(jenv)
				&& LoadClass(jenv, &clsNamedCallback, "org/sqlite/callback/NamedCallback")) {
			fid_namedcallback_name = (*jenv)->GetFieldID(jenv, clsNamedCallback, "name", "Ljava/lang/String;");
			return 1;
		}
	}

    return 0;
}

/* load 'org.sqlite.udf.Function' class */
static int LoadFunctionClass(JNIEnv *jenv) {
    if (clsFunction) {
        // alrealy loaded
        return 1;
    }

	if (LoadNamedCallbackClass(jenv)
			&& LoadClass(jenv, &clsFunction, "org/sqlite/udf/Function")) {
	    clsFunction = (*jenv)->NewGlobalRef(jenv, clsFunction);
		fid_function_argc = (*jenv)->GetFieldID(jenv, clsFunction, "argc", "I");
		return 1;
	}
    
    return 0;
}

/* load 'org.sqlite.udf.AggregateFunction' class */
static int LoadAggregateFunctionClass(JNIEnv *jenv) {
    if (clsAggregateFunction) {
        // alrealy loaded
        return 1;

    } else if (LoadNamedCallbackClass(jenv)
				&& LoadClass(jenv, &clsAggregateFunction, "org/sqlite/udf/AggregateFunction")) {
	    clsAggregateFunction = (*jenv)->NewGlobalRef(jenv, clsAggregateFunction);
		return 1;
	}
    
    return 0;
}

/* load 'org.sqlite.text.Collator' class */
static int LoadCollatorClass(JNIEnv *jenv) {
    if (clsCollator) {
        // alrealy loaded
        return 1;

    } else if (LoadNamedCallbackClass(jenv)
				&& LoadClass(jenv, &clsCollator, "org/sqlite/text/Collator")) {
	    clsCollator = (*jenv)->NewGlobalRef(jenv, clsCollator);
		return 1;
	}

    return 0;
}

/* load 'org.sqlite.auth.Authorizer' class */
static int LoadAuthorizerClass(JNIEnv *jenv) {
    if (clsAuthorizer) {
        // alrealy loaded
        return 1;

    } else if (LoadCallbackClass(jenv)
				&& LoadClass(jenv, &clsAuthorizer, "org/sqlite/auth/Authorizer")) {
	    clsAuthorizer = (*jenv)->NewGlobalRef(jenv, clsAuthorizer);
		return 1;
	}

    return 0;
}

/* load 'org.sqlite.event.BusyHandler' class */
static int LoadBusyHandlerClass(JNIEnv *jenv) {
    if (clsBusyHandler) {
        // alrealy loaded
        return 1;

    } else if (LoadCallbackClass(jenv)
				&& LoadClass(jenv, &clsBusyHandler, "org/sqlite/event/BusyHandler")) {
	    clsBusyHandler = (*jenv)->NewGlobalRef(jenv, clsBusyHandler);
		return 1;
	}

    return 0;
}

/* load 'org.sqlite.event.ProgressHandler' class */
static int LoadProgressHandlerClass(JNIEnv *jenv) {
    if (clsProgressHandler) {
        // alrealy loaded
        return 1;

    } else if (LoadCallbackClass(jenv)
				&& LoadClass(jenv, &clsProgressHandler, "org/sqlite/event/ProgressHandler")) {
	    clsProgressHandler = (*jenv)->NewGlobalRef(jenv, clsProgressHandler);
		return 1;
	}

    return 0;
}

/* convert pointer to jlong */
static jlong _jlong(void *p) {
    jvalue val;
    val.l = p;
    return val.j;
}

/* convert jlong to pointer */
static void * _voidp(jlong j) {
    jvalue val;
    val.j = j;
    return (void *)val.l;
}

/* structure for user-defined function and user-defined collating sequences */
typedef struct _JAVA_OBJECT {
    JavaVM *jvm;
    jobject obj;
} JAVA_OBJECT, *PJAVA_OBJECT;

static void xFunc(sqlite3_context* ctx, int argc, sqlite3_value** value) {
    static jmethodID mid_xFunc = 0;
    JNIEnv *jenv = NULL;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)sqlite3_user_data(ctx);
    if ((*jobj->jvm)->AttachCurrentThread(jobj->jvm, (void **)&jenv, 0) != JNI_OK) {
        // AttachCurrentThread() is not supported
        JavaThrowException(jenv, "java/lang/InternalError", "AttachCurrentThread() failed");
        return;
    }

    if (!mid_xFunc) {
        mid_xFunc = (*jenv)->GetMethodID(jenv, clsFunction, "xFunc", "(JIJ)V");
    }

    (*jenv)->CallVoidMethod(jenv, jobj->obj, mid_xFunc, _jlong(ctx), argc, _jlong(value));
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
}

static void xFinal(sqlite3_context* ctx) {
    static jmethodID mid_xFinal = 0;
    JNIEnv *jenv = NULL;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)sqlite3_user_data(ctx);
    if ((*jobj->jvm)->AttachCurrentThread(jobj->jvm, (void **)&jenv, 0) != JNI_OK) {
        // AttachCurrentThread() is not supported
        JavaThrowException(jenv, "java/lang/InternalError", "AttachCurrentThread() failed");
        return;
    }

    if (!mid_xFinal) {
        mid_xFinal = (*jenv)->GetMethodID(jenv, clsAggregateFunction, "xFinal", "(J)V");
    }

    (*jenv)->CallVoidMethod(jenv, jobj->obj, mid_xFinal, _jlong(ctx));
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
}

/* sqlite3_create_function by org.sqlite.udf.Function */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_create_1function(JNIEnv *jenv, jclass jcls, jlong db, jobject func) {
    PJAVA_OBJECT jobj = NULL;
    jstring jname = NULL;
    char *name = NULL;
    jint argc = 0;
    int isAgFunc = 0;
    jint jresult = SQLITE_ERROR;
    int result;

    if (!clsFunction) {
        if (!LoadFunctionClass(jenv)) {
            // not found 'org.sqlite.udf.Function' class
            return jresult;
        }
        assert(clsFunction);
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, func, clsFunction)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "function is not instanceof 'org.sqlite.udf.Function'.");
        return jresult;
    }

    // get org.sqlite.udf.Function._this value
    if ((*jenv)->GetLongField(jenv, func, fid_callback_this)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "function is already registered.");
        return jresult;
    }

    // get org.sqlite.udf.Function.name value
    jname = (jstring)(*jenv)->GetObjectField(jenv, func, fid_namedcallback_name);
    if (jname) {
        name = (char *)(*jenv)->GetStringUTFChars(jenv, jname, 0);
        if (!name) {
            SWIG_JavaThrowException(jenv, SWIG_JavaRuntimeException, "Failed JNIEnv#GetStringUTFChars().");
            return jresult;
        }
    } else {
        SWIG_JavaThrowException(jenv, SWIG_JavaNullPointerException, "name is null.");
        return jresult;
    }

    // get org.sqlite.udf.Function.argc value
    argc = (*jenv)->GetIntField(jenv, func, fid_function_argc);

    // allocate user data
    jobj = (PJAVA_OBJECT)calloc(1, sizeof(JAVA_OBJECT));
    if (!jobj) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return jresult;
    }
    jobj->obj = (*jenv)->NewGlobalRef(jenv, func);
    (*jenv)->GetJavaVM(jenv, &jobj->jvm);
    
    if (!clsAggregateFunction) {
        if (!LoadAggregateFunctionClass(jenv)) {
            // not found 'org.sqlite.udf.AggregateFunction' class
            return jresult;
        }
        assert(clsAggregateFunction);
    }
    
    // func insetanceof org.sqlite.udf.AggregateFunction
    isAgFunc = (*jenv)->IsInstanceOf(jenv, func, clsAggregateFunction);
    
    // register function
    result = sqlite3_create_function(*(sqlite3 **)&db, name, (int)argc, SQLITE_UTF8, jobj, (isAgFunc ? NULL : &xFunc), (isAgFunc ? &xFunc : NULL), (isAgFunc ? &xFinal : NULL));
    if (result == SQLITE_OK) {
        // set org.sqlite.udf.Function.pUserData value
        (*jenv)->SetLongField(jenv, func, fid_callback_this, _jlong(jobj));

    } else {
        // clean up user data
        (*jenv)->DeleteGlobalRef(jenv, jobj->obj);
        free(jobj);
    }

    if (name) {
        (*jenv)->ReleaseStringUTFChars(jenv, jname, (const char *)name);
    }

    jresult = (jint)result;
    return jresult;
}

/* sqlite3_create_function by org.sqlite.udf.Function */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_drop_1function(JNIEnv *jenv, jclass jcls, jlong db, jobject func) {
    jlong _this;
    jstring jname = NULL;
    char *name = NULL;
    jint jresult = SQLITE_ERROR;
    int result;

    if (!clsFunction) {
        if (!LoadFunctionClass(jenv)) {
            // not found 'org.sqlite.udf.Function' class
            return jresult;
        }
        assert(clsFunction);
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, func, clsFunction)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "function is not instanceof 'org.sqlite.udf.Function'.");
        return jresult;
    }

    // get org.sqlite.udf.Function._this value
    _this = (*jenv)->GetLongField(jenv, func, fid_callback_this);
    if (!_this) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "function is not registered.");
        return jresult;
    }

    // get org.sqlite.udf.Function.name value
    jname = (jstring)(*jenv)->GetObjectField(jenv, func, fid_namedcallback_name);
    if (jname) {
        name = (char *)(*jenv)->GetStringUTFChars(jenv, jname, 0);
        if (!name) {
            SWIG_JavaThrowException(jenv, SWIG_JavaRuntimeException, "Failed JNIEnv#GetStringUTFChars().");
            return jresult;
        }
    } else {
        SWIG_JavaThrowException(jenv, SWIG_JavaNullPointerException, "name is null.");
        return jresult;
    }

    // unregister function
    result = sqlite3_create_function(*(sqlite3 **)&db, name, -1, SQLITE_UTF8, NULL, NULL, NULL, NULL);
    if (result == SQLITE_OK) {
        // clean up user data
        PJAVA_OBJECT jobj = _voidp(_this);
        (*jenv)->DeleteGlobalRef(jenv, jobj->obj);
        free(jobj);
        (*jenv)->SetLongField(jenv, func, fid_callback_this, 0);
    }

    if (name) {
        (*jenv)->ReleaseStringUTFChars(jenv, jname, (const char *)name);
    }

    jresult = (jint)result;
    return jresult;
}

static int xCompare(void *p, int len1, const void *str1, int len2, const void *str2) {
    static jmethodID mid_xCompare = 0;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    JNIEnv *jenv = NULL;
    jstring jstr1 = NULL;
    jstring jstr2 = NULL;
    jint jresult = 0;
    
    if ((*jobj->jvm)->AttachCurrentThread(jobj->jvm, (void **)&jenv, 0) != JNI_OK) {
        // AttachCurrentThread() is not supported
        JavaThrowException(jenv, "java/lang/InternalError", "AttachCurrentThread() failed");
        return jresult;
    }

    if (!mid_xCompare) {
        mid_xCompare = (*jenv)->GetMethodID(jenv, clsCollator, "xCompare", "(Ljava/lang/String;Ljava/lang/String;)I");
    }

    if (str1) {
        if (strlen((const char *)str1) == len1) {
            jstr1 = (*jenv)->NewStringUTF(jenv, (const char *)str1);        
        } else {
            char *str = (char *)calloc(len1 + 1, sizeof(char));
            strncpy(str, str1, len1);
            str[len1] = '\0';
            jstr1 = (*jenv)->NewStringUTF(jenv, (const char *)str);
            free(str);
        }
    }

    if (str2) {
        if (strlen((const char *)str2) == len2) {
            jstr2 = (*jenv)->NewStringUTF(jenv, (const char *)str2);        
        } else {
            char *str = (char *)calloc(len2 + 1, sizeof(char));
            strncpy(str, str2, len2);
            str[len2] = '\0';
            jstr2 = (*jenv)->NewStringUTF(jenv, (const char *)str);
            free(str);
        }
    }
    
    jresult = (*jenv)->CallIntMethod(jenv, jobj->obj, mid_xCompare, jstr1, jstr2);
    
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
    
    return jresult;
}

/* sqlite3_create_collation by org.sqlite.text.Collator */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_create_1collation(JNIEnv *jenv, jclass jcls, jlong db, jobject col) {
    PJAVA_OBJECT jobj = NULL;
    jstring jname = NULL;
    char *name = NULL;
    jint jresult = SQLITE_ERROR;
    int result;

    if (!clsCollator) {
        if (!LoadCollatorClass(jenv)) {
            // not found 'org.sqlite.text.Collator' class
            return jresult;
        }
        assert(clsCollator);
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, col, clsCollator)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "collator is not instanceof 'org.sqlite.text.Collator'.");
        return jresult;
    }

    // get org.sqlite.text.Collator._this value
    if ((*jenv)->GetLongField(jenv, col, fid_callback_this)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "collation is already registered.");
        return jresult;
    }

    // get org.sqlite.text.Collator.name value
    jname = (jstring)(*jenv)->GetObjectField(jenv, col, fid_namedcallback_name);
    if (jname) {
        name = (char *)(*jenv)->GetStringUTFChars(jenv, jname, 0);
        if (!name) {
            SWIG_JavaThrowException(jenv, SWIG_JavaRuntimeException, "Failed JNIEnv#GetStringUTFChars().");
            return jresult;
        }
    } else {
        SWIG_JavaThrowException(jenv, SWIG_JavaNullPointerException, "name is null.");
        return jresult;
    }

    // allocate user data
    jobj = (PJAVA_OBJECT)calloc(1, sizeof(JAVA_OBJECT));
    if (!jobj) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return jresult;
    }
    jobj->obj = (*jenv)->NewGlobalRef(jenv, col);
    (*jenv)->GetJavaVM(jenv, &jobj->jvm);
    
    // register collation
    result = sqlite3_create_collation(*(sqlite3 **)&db, name, SQLITE_UTF8, jobj, &xCompare);
    if (result == SQLITE_OK) {
        // set org.sqlite.text.Collator._this value
        (*jenv)->SetLongField(jenv, col, fid_callback_this, _jlong(jobj));

    } else {
        // clean up user data
        (*jenv)->DeleteGlobalRef(jenv, jobj->obj);
        free(jobj);
    }

    if (name) {
        (*jenv)->ReleaseStringUTFChars(jenv, jname, (const char *)name);
    }

    jresult = (jint)result;
    return jresult;
}

/* sqlite3_create_collation by org.sqlite.text.Collator */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_drop_1collation(JNIEnv *jenv, jclass jcls, jlong db, jobject col) {
    jlong _this;
    jstring jname = NULL;
    char *name = NULL;
    jint jresult = SQLITE_ERROR;
    int result;

    if (!clsCollator) {
        if (!LoadCollatorClass(jenv)) {
            // not found 'org.sqlite.text.Collator' class
            return jresult;
        }
        assert(clsCollator);
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, col, clsCollator)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "collator is not instanceof 'org.sqlite.text.Collator'.");
        return jresult;
    }

    // get org.sqlite.text.Collator._this value
    _this = (*jenv)->GetLongField(jenv, col, fid_callback_this);
    if (!_this) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "collation is not registered.");
        return jresult;
    }

    // get org.sqlite.text.Collator.name value
    jname = (jstring)(*jenv)->GetObjectField(jenv, col, fid_namedcallback_name);
    if (jname) {
        name = (char *)(*jenv)->GetStringUTFChars(jenv, jname, 0);
        if (!name) {
            SWIG_JavaThrowException(jenv, SWIG_JavaRuntimeException, "Failed JNIEnv#GetStringUTFChars().");
            return jresult;
        }
    } else {
        SWIG_JavaThrowException(jenv, SWIG_JavaNullPointerException, "name is null.");
        return jresult;
    }

    // unregister collation
    result = sqlite3_create_collation(*(sqlite3 **)&db, name, SQLITE_UTF8, NULL, NULL);
    if (result == SQLITE_OK) {
        // clean up user data
        PJAVA_OBJECT jobj = _voidp(_this);
        (*jenv)->DeleteGlobalRef(jenv, jobj->obj);
        free(jobj);
        (*jenv)->SetLongField(jenv, col, fid_callback_this, 0);
    }

    if (name) {
        (*jenv)->ReleaseStringUTFChars(jenv, jname, (const char *)name);
    }

    jresult = (jint)result;
    return jresult;
}

static int xAuth(void *p, int action, const char *param3, const char *param4, const char* database, const char* triggerOrView) {
    static jmethodID mid_xAuth = 0;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    JNIEnv *jenv = NULL;
    jint jaction = (jint)action;
    jstring jstr1 = NULL;
    jstring jstr2 = NULL;
    jstring jstr3 = NULL;
    jstring jstr4 = NULL;    
    jint jresult = SQLITE_DENY;
    
    if ((*jobj->jvm)->AttachCurrentThread(jobj->jvm, (void **)&jenv, 0) != JNI_OK) {
        // AttachCurrentThread() is not supported
        JavaThrowException(jenv, "java/lang/InternalError", "AttachCurrentThread() failed");
        return jresult;
    }

    if (!mid_xAuth) {
        mid_xAuth = (*jenv)->GetMethodID(jenv, clsAuthorizer, "xAuth", "(ILjava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;)I");
    }

    if (param3) {
        jstr1 = (*jenv)->NewStringUTF(jenv, param3);        
    }
    if (param4) {
        jstr2 = (*jenv)->NewStringUTF(jenv, param4);        
    }
    if (database) {
        jstr3 = (*jenv)->NewStringUTF(jenv, database);        
    }
    if (triggerOrView) {
        jstr4 = (*jenv)->NewStringUTF(jenv, triggerOrView);        
    }
    
    jresult = (*jenv)->CallIntMethod(jenv, jobj->obj, mid_xAuth, jaction, jstr1, jstr2, jstr3, jstr4);
    
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
    
    return jresult;
}

/* sqlite3_set_authorizer by org.sqlite.auth.Authorizer */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_set_1authorizer(JNIEnv *jenv, jclass jcls, jlong db, jobject auth) {
    PJAVA_OBJECT jobj = NULL;
    jint jresult = SQLITE_ERROR;
    int result;

    if (!clsAuthorizer) {
        if (!LoadAuthorizerClass(jenv)) {
            // not found 'org.sqlite.auth.Authorizer' class
            return jresult;
        }
        assert(clsAuthorizer);
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, auth, clsAuthorizer)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "authorizer is not instanceof 'org.sqlite.auth.Authorizer'.");
        return jresult;
    }

    // get org.sqlite.auth.Authorizer._this value
    if ((*jenv)->GetLongField(jenv, auth, fid_callback_this)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "authorizer has already been set.");
        return jresult;
    }

    // allocate user data
    jobj = (PJAVA_OBJECT)calloc(1, sizeof(JAVA_OBJECT));
    if (!jobj) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return jresult;
    }
    jobj->obj = (*jenv)->NewGlobalRef(jenv, auth);
    (*jenv)->GetJavaVM(jenv, &jobj->jvm);
    
    // set authorizer
    result = sqlite3_set_authorizer(*(sqlite3 **)&db, &xAuth, jobj);
    if (result == SQLITE_OK) {
        // set org.sqlite.auth.Authorizer._this value
        (*jenv)->SetLongField(jenv, auth, fid_callback_this, _jlong(jobj));

    } else {
        // clean up user data
        (*jenv)->DeleteGlobalRef(jenv, jobj->obj);
        free(jobj);
    }

    jresult = (jint)result;
    return jresult;
}

/* sqlite3_set_authorizer by org.sqlite.auth.Authorizer */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_clear_1authorizer(JNIEnv *jenv, jclass jcls, jlong db, jobject auth) {
    jlong _this = 0;
    jint jresult = SQLITE_ERROR;
    int result;

    if (!clsAuthorizer) {
        if (!LoadAuthorizerClass(jenv)) {
            // not found 'org.sqlite.auth.Authorizer' class
            return jresult;
        }
        assert(clsAuthorizer);
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, auth, clsAuthorizer)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "authorizer is not instanceof 'org.sqlite.auth.Authorizer'.");
        return jresult;
    }

    // get org.sqlite.auth.Authorizer._this value
    _this = (*jenv)->GetLongField(jenv, auth, fid_callback_this);
    if (!_this) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "authorizer is not set.");
        return jresult;
    }

    // clear authorizer
    result = sqlite3_set_authorizer(*(sqlite3 **)&db, NULL, NULL);
    if (result == SQLITE_OK) {
        // clean up user data
        PJAVA_OBJECT jobj = _voidp(_this);
        (*jenv)->DeleteGlobalRef(jenv, jobj->obj);
        free(jobj);
        (*jenv)->SetLongField(jenv, auth, fid_callback_this, 0);
    }

    jresult = (jint)result;
    return jresult;
}

static int xBusy(void *p, int count) {
    static jmethodID mid_xBusy = 0;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    JNIEnv *jenv = NULL;
    jint jresult = 0;
    
    if ((*jobj->jvm)->AttachCurrentThread(jobj->jvm, (void **)&jenv, 0) != JNI_OK) {
        // AttachCurrentThread() is not supported
        JavaThrowException(jenv, "java/lang/InternalError", "AttachCurrentThread() failed");
        return jresult;
    }

    if (!mid_xBusy) {
        mid_xBusy = (*jenv)->GetMethodID(jenv, clsBusyHandler, "xBusy", "(I)I");
    }

    jresult = (*jenv)->CallIntMethod(jenv, jobj->obj, mid_xBusy, (jint)count);
    
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
    
    return jresult;
}

/* sqlite3_busy_handler by org.sqlite.event.BusyHandler */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_set_1busy_1handler(JNIEnv *jenv, jclass jcls, jlong db, jobject busy) {
    PJAVA_OBJECT jobj = NULL;
    jint jresult = SQLITE_ERROR;
    int result;

    if (!clsBusyHandler) {
        if (!LoadBusyHandlerClass(jenv)) {
            // not found 'org.sqlite.event.BusyHandler' class
            return jresult;
        }
        assert(clsBusyHandler);
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, busy, clsBusyHandler)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "busy handler is not instanceof 'org.sqlite.event.BusyHandler'.");
        return jresult;
    }

    // get org.sqlite.event.BusyHandler._this value
    if ((*jenv)->GetLongField(jenv, busy, fid_callback_this)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "busy handler has already been set.");
        return jresult;
    }

    // allocate user data
    jobj = (PJAVA_OBJECT)calloc(1, sizeof(JAVA_OBJECT));
    if (!jobj) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return jresult;
    }
    jobj->obj = (*jenv)->NewGlobalRef(jenv, busy);
    (*jenv)->GetJavaVM(jenv, &jobj->jvm);
    
    // set busy handler
    result = sqlite3_busy_handler(*(sqlite3 **)&db, &xBusy, jobj);
    if (result == SQLITE_OK) {
        // set org.sqlite.event.BusyHandler._this value
        (*jenv)->SetLongField(jenv, busy, fid_callback_this, _jlong(jobj));

    } else {
        // clean up user data
        (*jenv)->DeleteGlobalRef(jenv, jobj->obj);
        free(jobj);
    }

    jresult = (jint)result;
    return jresult;
}

/* sqlite3_busy_handler by org.sqlite.event.BusyHandler */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_clear_1busy_1handler(JNIEnv *jenv, jclass jcls, jlong db, jobject busy) {
    jlong _this = 0;
    jint jresult = SQLITE_ERROR;
    int result;

    if (!clsBusyHandler) {
        if (!LoadBusyHandlerClass(jenv)) {
            // not found 'org.sqlite.event.BusyHandler' class
            return jresult;
        }
        assert(clsBusyHandler);
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, busy, clsBusyHandler)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "busy handler is not instanceof 'org.sqlite.event.BusyHandler'.");
        return jresult;
    }

    // get org.sqlite.event.BusyHandler._this value
    _this = (*jenv)->GetLongField(jenv, busy, fid_callback_this);
    if (!_this) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "busy handler is not set.");
        return jresult;
    }

    // clear busy handler
    result = sqlite3_busy_handler(*(sqlite3 **)&db, NULL, NULL);
    if (result == SQLITE_OK) {
        // clean up user data
        PJAVA_OBJECT jobj = _voidp(_this);
        (*jenv)->DeleteGlobalRef(jenv, jobj->obj);
        free(jobj);
        (*jenv)->SetLongField(jenv, busy, fid_callback_this, 0);
    }

    jresult = (jint)result;
    return jresult;
}

static int xProgress(void *p) {
    static jmethodID mid_xProgress = 0;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    JNIEnv *jenv = NULL;
    jint jresult = 0;
    
    if ((*jobj->jvm)->AttachCurrentThread(jobj->jvm, (void **)&jenv, 0) != JNI_OK) {
        // AttachCurrentThread() is not supported
        JavaThrowException(jenv, "java/lang/InternalError", "AttachCurrentThread() failed");
        return jresult;
    }

    if (!mid_xProgress) {
        mid_xProgress = (*jenv)->GetMethodID(jenv, clsProgressHandler, "xProgress", "()I");
    }

    jresult = (*jenv)->CallIntMethod(jenv, jobj->obj, mid_xProgress);
    
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
    
    return jresult;
}

/* sqlite3_progress_handler by org.sqlite.event.ProgressHandler */
JNIEXPORT void JNICALL Java_org_sqlite_swig_SQLite3JNI_set_1progress_1handler(JNIEnv *jenv, jclass jcls, jlong db, jobject prog) {
    static jfieldID fid_progresshandler_opecodes = 0;
    jint opecodes = 0;
    PJAVA_OBJECT jobj = NULL;

    if (!clsProgressHandler) {
        if (!LoadProgressHandlerClass(jenv)) {
            // not found 'org.sqlite.event.ProgressHandler' class
            return;
        }
        assert(clsProgressHandler);

		fid_progresshandler_opecodes = (*jenv)->GetFieldID(jenv, clsProgressHandler, "opecodes", "I");
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, prog, clsProgressHandler)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "progress handler is not instanceof 'org.sqlite.event.ProgressHandler'.");
        return;
    }

    // get org.sqlite.event.ProgressHandler._this value
    if ((*jenv)->GetLongField(jenv, prog, fid_callback_this)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "progress handler has already been set.");
        return;
    }

    // get org.sqlite.event.ProgressHandler.opecodes value
    opecodes = (*jenv)->GetIntField(jenv, prog, fid_progresshandler_opecodes);
    
    // allocate user data
    jobj = (PJAVA_OBJECT)calloc(1, sizeof(JAVA_OBJECT));
    if (!jobj) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return;
    }
    jobj->obj = (*jenv)->NewGlobalRef(jenv, prog);
    (*jenv)->GetJavaVM(jenv, &jobj->jvm);
    
    // set progress handler
    sqlite3_progress_handler(*(sqlite3 **)&db, (int)opecodes, &xProgress, jobj);

    // set org.sqlite.event.ProgressHandler._this value
    (*jenv)->SetLongField(jenv, prog, fid_callback_this, _jlong(jobj));
}

/* sqlite3_progress_handler by org.sqlite.event.ProgressHandler */
JNIEXPORT void JNICALL Java_org_sqlite_swig_SQLite3JNI_clear_1progress_1handler(JNIEnv *jenv, jclass jcls, jlong db, jobject prog) {
    jlong _this = 0;
    PJAVA_OBJECT jobj = NULL;

    if (!clsProgressHandler) {
        if (!LoadProgressHandlerClass(jenv)) {
            // not found 'org.sqlite.event.ProgressHandler' class
            return;
        }
        assert(clsProgressHandler);
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, prog, clsProgressHandler)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "progress handler is not instanceof 'org.sqlite.event.ProgressHandler'.");
        return;
    }

    // get org.sqlite.event.ProgressHandler._this value
    _this = (*jenv)->GetLongField(jenv, prog, fid_callback_this);
    if (!_this) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "progress handler is not set.");
        return;
    }

    // clear progress handler
    sqlite3_progress_handler(*(sqlite3 **)&db, 0, NULL, NULL);
    
    // clean up user data
    jobj = _voidp(_this);
    (*jenv)->DeleteGlobalRef(jenv, jobj->obj);
    free(jobj);
    (*jenv)->SetLongField(jenv, prog, fid_callback_this, 0);
}


#ifdef __cplusplus
}
#endif
%}

/* typemap for 'const unsigned char *' to 'java.lang.String' */
%typemap(jni) const unsigned char *    "jstring"
%typemap(jtype) const unsigned char *  "String"
%typemap(jstype) const unsigned char * "String"
%typemap(out, noblock=1) const unsigned char * { if($1) $result = JCALL1(NewStringUTF, jenv, (const char *)$1); }
%typemap(javaout) const unsigned char * {
    return $jnicall;
  }

/*
%pragma(java) jniclassclassmodifiers="public final class"
%pragma(java) moduleimports="import org.sqlite.udf.Function;\n";
*/
%javamethodmodifiers sqlite3_bind_blob_by_bytes "private";
%javamethodmodifiers create_function "private";
%javamethodmodifiers drop_function "private";
%javamethodmodifiers create_collation "private";
%javamethodmodifiers drop_collation "private";
%javamethodmodifiers set_authorizer "private";
%javamethodmodifiers clear_authorizer "private";
%javamethodmodifiers set_busy_handler "private";
%javamethodmodifiers clear_busy_handler "private";
%javamethodmodifiers set_progress_handler "private";
%javamethodmodifiers clear_progress_handler "private";
%pragma(java) modulecode="
  public static final int SQLITE_TRANSIENT = -1;

  /**
   * Returns in-memory filename.
   * @return \":memory:\"
   * @see <a href=\"http://sqlite.org/c3ref/open.html\">Opening A New Database Connection</a>
   */
  public static String getInMemoryFileName() {
    return \":memory:\";
  }

  /**
   * Returns date format pattern.
   * @return \"yyyy-MM-dd\"
   * @see <a href=\"http://sqlite.org/lang_createtable.html\">CREATE TABLE</a>
   */
  public static String getDateFormatPattern() {
    return \"yyyy-MM-dd\";
  }

  /**
   * Returns time format pattern.
   * @return \"HH:mm:ss\"
   * @see <a href=\"http://sqlite.org/lang_createtable.html\">CREATE TABLE</a>
   */
  public static String getTimeFormatPattern() {
    return \"HH:mm:ss\";
  }

  /**
   * Returns timestamp format pattern.
   * @return \"yyyy-MM-dd HH:mm:ss\"
   * @see <a href=\"http://sqlite.org/lang_createtable.html\">CREATE TABLE</a>
   */
  public static String getTimestampFormatPattern() {
    return \"yyyy-MM-dd HH:mm:ss\";
  }
  
  public static String format(String pattern, java.util.Date x) {
    return new java.text.SimpleDateFormat(pattern).format(x);
  }
  
  public static String format(java.sql.Date x) {
    return format(getDateFormatPattern(), x);
  }
  
  public static String format(java.sql.Time x) {
    return format(getTimeFormatPattern(), x);
  }
  
  public static String format(java.sql.Timestamp x) {
    return format(getTimestampFormatPattern(), x);
  }

  public static long parse(String pattern, String x) throws java.sql.SQLException {
    final java.text.DateFormat formatter = new java.text.SimpleDateFormat(pattern);
    formatter.setLenient(false);
    final java.text.ParsePosition position = new java.text.ParsePosition(0);
    final java.util.Date date = formatter.parse(x, position);
    if (position.getErrorIndex() != -1 || position.getIndex() != x.length()) {
        // parse failed
        throw new java.sql.SQLException(\"Format error.\", \"90J09\");
    }
    return date.getTime();
  }
  
  public static long parseDate(String x) throws java.sql.SQLException {
    return parse(getDateFormatPattern(), x);
  }
  
  public static long parseTime(String x) throws java.sql.SQLException {
    return parse(getTimeFormatPattern(), x);
  }
  
  public static long parseTimestamp(String x) throws java.sql.SQLException {
    return parse(getTimestampFormatPattern(), x);
  }

  public static long addressOf(SWIGTYPE_p_sqlite3_stmt stmt) {
    return SWIGTYPE_p_sqlite3_stmt.getCPtr(stmt);
  }

  public static int sqlite3_exec(SWIGTYPE_p_sqlite3 db, String sql) {
    return SQLite3JNI.sqlite3_exec(SWIGTYPE_p_sqlite3.getCPtr(db), sql, 0, 0, 0);
  }

  public static int sqlite3_bind_blob(SWIGTYPE_p_sqlite3_stmt stmt, int parameterIndex, byte[] val, int len) {
    return SQLite3JNI.sqlite3_bind_blob_by_bytes(SWIGTYPE_p_sqlite3_stmt.getCPtr(stmt), parameterIndex, val, len, SQLITE_TRANSIENT);
  }

  public static int sqlite3_bind_text(SWIGTYPE_p_sqlite3_stmt stmt, int parameterIndex, String val) {
    return SQLite3JNI.sqlite3_bind_text(SWIGTYPE_p_sqlite3_stmt.getCPtr(stmt), parameterIndex, val, -1, SQLITE_TRANSIENT);
  }

  public static int create_function(SWIGTYPE_p_sqlite3 db, org.sqlite.udf.Function func) {
    return SQLite3JNI.create_function(SWIGTYPE_p_sqlite3.getCPtr(db), func);
  }

  public static int drop_function(SWIGTYPE_p_sqlite3 db, org.sqlite.udf.Function func) {
    return SQLite3JNI.drop_function(SWIGTYPE_p_sqlite3.getCPtr(db), func);
  }

  public static void sqlite3_result_blob(SWIGTYPE_p_sqlite3_context ctx, SWIGTYPE_p_void blob, int len) {
    SQLite3JNI.sqlite3_result_blob(SWIGTYPE_p_sqlite3_context.getCPtr(ctx), SWIGTYPE_p_void.getCPtr(blob), len, SQLITE_TRANSIENT);
  }

  public static void sqlite3_result_blob(SWIGTYPE_p_sqlite3_context ctx, byte[] val, int len) {
    SQLite3JNI.sqlite3_result_blob_by_bytes(SWIGTYPE_p_sqlite3_context.getCPtr(ctx), val, len, SQLITE_TRANSIENT);
  }

  public static void sqlite3_result_error(SWIGTYPE_p_sqlite3_context ctx, String message) {
    SQLite3JNI.sqlite3_result_error(SWIGTYPE_p_sqlite3_context.getCPtr(ctx), message, -1);
  }

  public static void sqlite3_result_text(SWIGTYPE_p_sqlite3_context ctx, String val) {
    SQLite3JNI.sqlite3_result_text(SWIGTYPE_p_sqlite3_context.getCPtr(ctx), val, -1, SQLITE_TRANSIENT);
  }

  public static int create_collation(SWIGTYPE_p_sqlite3 db, org.sqlite.text.Collator col) {
    return SQLite3JNI.create_collation(SWIGTYPE_p_sqlite3.getCPtr(db), col);
  }

  public static int drop_collation(SWIGTYPE_p_sqlite3 db, org.sqlite.text.Collator col) {
    return SQLite3JNI.drop_collation(SWIGTYPE_p_sqlite3.getCPtr(db), col);
  }

  public static int set_authorizer(SWIGTYPE_p_sqlite3 db, org.sqlite.auth.Authorizer auth) {
    return SQLite3JNI.set_authorizer(SWIGTYPE_p_sqlite3.getCPtr(db), auth);
  }

  public static int clear_authorizer(SWIGTYPE_p_sqlite3 db, org.sqlite.auth.Authorizer auth) {
    return SQLite3JNI.clear_authorizer(SWIGTYPE_p_sqlite3.getCPtr(db), auth);
  }

  public static int set_busy_handler(SWIGTYPE_p_sqlite3 db, org.sqlite.event.BusyHandler busy) {
    return SQLite3JNI.set_busy_handler(SWIGTYPE_p_sqlite3.getCPtr(db), busy);
  }

  public static int clear_busy_handler(SWIGTYPE_p_sqlite3 db, org.sqlite.event.BusyHandler busy) {
    return SQLite3JNI.clear_busy_handler(SWIGTYPE_p_sqlite3.getCPtr(db), busy);
  }

  public static void set_progress_handler(SWIGTYPE_p_sqlite3 db, org.sqlite.event.ProgressHandler prog) {
    SQLite3JNI.set_progress_handler(SWIGTYPE_p_sqlite3.getCPtr(db), prog);
  }

  public static void clear_progress_handler(SWIGTYPE_p_sqlite3 db, org.sqlite.event.ProgressHandler prog) {
    SQLite3JNI.clear_progress_handler(SWIGTYPE_p_sqlite3.getCPtr(db), prog);
  }

  public static int sqlite3_clear_bindings(SWIGTYPE_p_sqlite3_stmt stmt) {
    final int max = sqlite3_bind_parameter_count(stmt) + 1;
    int ret = SQLITE_OK;
    for (int i = 1; ret == SQLITE_OK && i < max; ++i) {
      ret = sqlite3_bind_null(stmt, i);
    }
    return ret;
  }
"

%javaconst(1);
%javaconst(0) SQLITE_VERSION;
%javaconst(0) SQLITE_VERSION_NUMBER;
%include "sqlite3.h"
%include "various.i"
%include "cpointer.i"

/*******************************************************************/
%exception new_p_p_sqlite3 {
    $action
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return $null;
    }
}

%exception new_p_p_sqlite3_stmt {
    $action
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return $null;
    }
}

%exception new_p_p_char {
    $action
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return $null;
    }
}

%exception new_p_int {
    $action
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return $null;
    }
}

/* Create some functions for working with "sqlite3**" */
%pointer_functions(sqlite3*, p_p_sqlite3);

/* Create some functions for working with "sqlite3_stmt**" */
%pointer_functions(sqlite3_stmt*, p_p_sqlite3_stmt);

/* Create some functions for working with "char const*" */
%pointer_functions(char const*, p_p_char);

/* Create some functions for working with "int*" */
%pointer_functions(int, p_int);

%inline %{
/* sqlite3_bind_blob by byte[] */
static int sqlite3_bind_blob_by_bytes(sqlite3_stmt *pStmt, int parameterIndex, char *BYTE, int length, void (*xDel)(void*)) {
    return sqlite3_bind_blob(pStmt, parameterIndex, BYTE, length, xDel);
}

/* sqlite3_column_blob by java.sql.Blob */
static void read_blob(const void *blob, long long pos, char *BYTE, int offset, int len) {
    memcpy(&BYTE[offset], &((const char *)blob)[pos], len);
}

/* get (sqlite3_value*)sqlite3_value**[i] */
static sqlite3_value* get_p_sqlite3_value(sqlite3_value** value, int i) {
    return value[i];
}

/* sqlite3_result_blob by byte[] */
static void sqlite3_result_blob_by_bytes(sqlite3_context *pCtx, char *BYTE, int length, void (*xDel)(void*)) {
    sqlite3_result_blob(pCtx, BYTE, length, xDel);
}

%}

/* sqlite3_column_blob by byte[] */
%native(sqlite3_column_blob_by_bytes) jbyteArray sqlite3_column_blob_by_bytes(sqlite3_stmt*, int);

/* sqlite3_value_blob by byte[] */
%native(sqlite3_value_blob_by_bytes) jbyteArray sqlite3_value_blob_by_bytes(sqlite3_value*);

/* create_function */
%native(create_function) jint create_function(sqlite3*, jobject);

/* drop_function */
%native(drop_function) jint drop_function(sqlite3*, jobject);

/* create_collation */
%native(create_collation) jint create_collation(sqlite3*, jobject);

/* drop_collation */
%native(drop_collation) jint drop_collation(sqlite3*, jobject);

/* set_authorizer */
%native(set_authorizer) jint set_authorizer(sqlite3*, jobject);

/* clear_authorizer */
%native(clear_authorizer) jint clear_authorizer(sqlite3*, jobject);

/* set_busy_handler */
%native(set_busy_handler) jint set_busy_handler(sqlite3*, jobject);

/* clear_busy_handler */
%native(clear_busy_handler) jint clear_busy_handler(sqlite3*, jobject);

/* set_progress_handler */
%native(set_progress_handler) void set_progress_handler(sqlite3*, jobject);

/* clear_progress_handler */
%native(clear_progress_handler) void clear_progress_handler(sqlite3*, jobject);

/*******************************************************************/
