package org.sqlite;

import org.sqlite.jdbc.TransactionType;
import org.sqlite.schema.ColumnMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Logger;
import org.sqlite.auth.Authorizer;
import org.sqlite.event.BusyHandler;
import org.sqlite.event.CommitHook;
import org.sqlite.event.ProgressHandler;
import org.sqlite.event.RollbackHook;
import org.sqlite.event.UpdateHook;
import org.sqlite.jdbc.JdbcSQLException;
import org.sqlite.swig.SWIGTYPE_p_int;
import org.sqlite.swig.SWIGTYPE_p_p_char;
import org.sqlite.swig.SWIGTYPE_p_p_sqlite3;
import org.sqlite.swig.SWIGTYPE_p_sqlite3;
import org.sqlite.text.Collator;
import org.sqlite.types.SQLite3StmtPtrPtr;
import org.sqlite.udf.Function;
import static org.sqlite.swig.SQLite3.*;

/**
 * sqlite3 wrapper class.<br/>
 * NOTE: SQLite 3.3.5 based.
 * @author calico
 */
public class Database {
    /** sqlite3** wrapper class */
    private static class SQLite3PtrPtr extends SWIGTYPE_p_p_sqlite3 {
        private boolean isDeleted = false;
        
        public SQLite3PtrPtr() {
            super(getCPtr(new_p_p_sqlite3()), false);
        }

        public boolean isDeleted() {
            return isDeleted;
        }
        
        public synchronized void delete() {
            if (!isDeleted) {
                delete_p_p_sqlite3(this);
                isDeleted = true;
            }
        }

        public SWIGTYPE_p_sqlite3 getSQLite3Ptr() {
            if (isDeleted) {
                throw new IllegalStateException("SQLite3PtrPtr is already deleted.");
            }
            return p_p_sqlite3_value(this);
        }

        @Override
        protected void finalize() throws Throwable {
            delete();
            super.finalize();
        }
    }
    
    protected final Properties info;
    private final boolean isInMemory;
    private final SQLite3PtrPtr ppDb = new SQLite3PtrPtr();
    private Map<Long, Statement> statements;
    private Map<String, Function> functions;
    private Map<String, Collator> collators;
    private Authorizer authorizer;
    private BusyHandler busyHandler;
    private ProgressHandler progressHandler;
    private CommitHook commitHook;
    private RollbackHook rollbackHook;
    private UpdateHook updateHook;
    
    /** timeout(ms) : sqlite3_busy_timeout */
    private int timeout;
    
    public Database(String filename, Properties info) throws SQLException {
        this.info = info;
        this.isInMemory
                = (filename == null || getInMemoryFileName().equals(filename));
        open(filename);
    }
    
    /**
     * It always returns "SQLite".
     * @return "SQLite"
     */
    public String getProductName() {
        return "SQLite";
    }
    
    public boolean isInMemoryMode() {
        return isInMemory;
    }
    
    private SWIGTYPE_p_sqlite3 getInstance() {
        return ppDb.getSQLite3Ptr();
    }
    
    /**
     * invoke sqlite3_open() function.
     * @param filename  filename
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/open.html">Opening A New Database Connection</a>
     */
    private void open(String filename) throws SQLException {
        int ret = sqlite3_open(filename, ppDb);
        if (ret != SQLITE_OK) {
            SWIGTYPE_p_sqlite3 db = getInstance();
            SQLException ex = new JdbcSQLException(db);
            ppDb.delete();
            throw ex;
        }
    }
    
    public boolean isReadOnly() {
        return false;
    }
    
    /**
     * Retrieves whether this Database object has been closed.
     * @return true if this Database object is closed. false if it is still open.
     */
    public boolean isClosed() {
        return (ppDb.isDeleted());
    }
    
    /**
     * invoke sqlite3_close() function.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/close.html">Closing A Database Connection</a>
     */
    public void close() throws SQLException {
        if (!isClosed()) {
            // close all statements
            closeStatements();
            // unregister all functions
            dropFunctions();
            // unregister all collators
            dropCollators();
            // clear authorizer
            internalClearAuthorizer();
            // clear busy handler
            internalClearBusyHandler();
            // clear progress handler
            clearProgressHandler();
            // clear commit hook
            clearCommitHook();
            // clear rollback hook
            clearRollbackHook();
            // clear update hook
            clearUpdateHook();
            
            final SWIGTYPE_p_sqlite3 db = getInstance();
            int ret = sqlite3_close(db);
            if (ret != SQLITE_OK) {
                throw new JdbcSQLException(db);
            }
            ppDb.delete();
        }
    }
    
    /**
     * invoke sqlite3_get_autocommit() function.
     * @return  true if auto commit mode.
     * @see <a href="http://sqlite.org/c3ref/get_autocommit.html">Test To See If The Database Is In Auto-Commit Mode</a>
     */
    public boolean getAutoCommit() {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        return (sqlite3_get_autocommit(db) != 0);
    }
    
    /**
     * invoke sqlite3_busy_timeout() function.
     * @param ms    milliseconds
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/busy_timeout.html">Set A Busy Timeout</a>
     */
    public void setBusyTimeout(int ms) throws SQLException {
        clearBusyHandler();

        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = sqlite3_busy_timeout(db, ms);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        timeout = (ms < 1 ? 0 : ms);
    }

    /**
     * Retrieves the timeout(ms) value.
     * @return  timeout(ms) value.
     */
    public int getBusyTimeout() {
        return timeout;
    }
    
    /**
     * invoke sqlite3_exec() function.
     * @param sql
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/exec.html">One-Step Query Execution Interface</a>
     */
    public int execute(String sql) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = 0;
        if (timeout == 0) {
            // no limit
            while ((ret = sqlite3_exec(db, sql)) == SQLITE_BUSY) {
                // waiting...
            }
        } else {
            ret = sqlite3_exec(db, sql);
            if (ret == SQLITE_BUSY) {
                // timeout
                throw new JdbcSQLException("Timeout expired.", SQLITE_BUSY);
            }
        }
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        return ret;
    }

    /**
     * execute PRAGMA commands by sqlite3_exec() finction.
     * @param commands  command list
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/pragma.html">Pragma statements supported by SQLite</a>
     */
    public void pragma(String[] commands) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        for (final String cmd : commands) {
            int ret = sqlite3_exec(db, "PRAGMA " + cmd);
            if (ret != SQLITE_OK) {
                throw new JdbcSQLException(db);
            }
        }
    }
    
    /**
     * begin transaction.
     * @param type  transaction type.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/lang_transaction.html">BEGIN TRANSACTION</a>
     */
    public void beginTransaction(TransactionType type) throws SQLException {
        closeStatements();
        if (type == null) {
            execute("BEGIN");
        } else {
            execute("BEGIN " + type);
        }
    }
    
    /**
     * commit toransaction.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/lang_transaction.html">BEGIN TRANSACTION</a>
     */
    public void commitTransaction() throws SQLException {
        closeStatements();
        execute("COMMIT");
    }
    
    /**
     * rollback transaction.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/lang_transaction.html">BEGIN TRANSACTION</a>
     */
    public void rollbackTransaction() throws SQLException {
        closeStatements();
        execute("ROLLBACK");
    }
    
    /**
     * create MANAGED Statement instance.
     * @param sql
     * @param ppStmt
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/prepare.html">Compiling An SQL Statement</a>
     */
    public Statement prepare(String sql, SQLite3StmtPtrPtr ppStmt) throws SQLException {
        if (sql == null) {
            throw new NullPointerException("sql is null.");
        }
        if (ppStmt == null) {
            throw new NullPointerException("ppStmt is null.");
        }
        
        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = sqlite3_prepare(db, sql, -1, ppStmt);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        return new Statement(this, ppStmt.getSQLite3StmtPtr());
    }
    
    /**
     * create UNMANAGED Statement instance.
     * @param sql
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/prepare.html">Compiling An SQL Statement</a>
     */
    public Statement prepare(String sql) throws SQLException {
        if (sql == null) {
            throw new NullPointerException("sql is null.");
        }

        final SWIGTYPE_p_sqlite3 db = getInstance();
        final SQLite3StmtPtrPtr ppStmt = new SQLite3StmtPtrPtr();
        int ret = sqlite3_prepare(db, sql, -1, ppStmt);
        if (ret != SQLITE_OK) {
            ppStmt.delete();
            throw new JdbcSQLException(db);
        }
        return new Statement(this, ppStmt);
    }
    
    /**
     * create multiple UNMANAGED Statement instance.
     * @param sql
     * @return array of Statement
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/prepare.html">Compiling An SQL Statement</a>
     */
    public List<Statement> prepareMultiple(String sql) throws SQLException {
        if (sql == null) {
            throw new NullPointerException("sql is null.");
        }

        final SWIGTYPE_p_sqlite3 db = getInstance();
        final List<SQLite3StmtPtrPtr> stmts = new ArrayList<SQLite3StmtPtrPtr>();
        final String[] tail = new String[1];
        do {
            final SQLite3StmtPtrPtr ppStmt = new SQLite3StmtPtrPtr();
            stmts.add(ppStmt);
            final int ret = sqlite3_prepare(db, sql, -1, ppStmt, tail);
            if (ret != SQLITE_OK) {
                for (final SQLite3StmtPtrPtr stmt : stmts) {
                    stmt.delete();
                }
                throw new JdbcSQLException(db);
            }
            sql = tail[0].trim();
        } while (sql.length() > 0);
        
        final List<Statement> ret = new  ArrayList<Statement>(stmts.size());
        for (final SQLite3StmtPtrPtr stmt : stmts) {
            ret.add(new Statement(this, stmt));
        }
        return ret;
    }
    
    /**
     * invoke sqlite3_interrupt() function.
     * @see <a href="http://sqlite.org/c3ref/interrupt.html">Interrupt A Long-Running Query</a>
     */
    public void interrupt() throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        sqlite3_interrupt(db);
        closeStatements();
    }
    
    /**
     * invoke sqlite3_changes() function.
     * @return
     * @see <a href="http://sqlite.org/c3ref/changes.html">Count The Number Of Rows Modified</a>
     */
    public int changes() {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        return sqlite3_changes(db);
    }
    
    /**
     * invoke sqlite3_total_changes() function.
     * @return
     * @see <a href="http://sqlite.org/c3ref/total_changes.html">Total Number Of Rows Modified</a>
     */
    public int totalChanges() {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        return sqlite3_total_changes(db);
    }
    
    /**
     * invoke sqlite3_last_insert_rowid() function.
     * @return
     * @see <a href="http://sqlite.org/c3ref/last_insert_rowid.html">Last Insert Rowid</a>
     */
    public long lastInsertRowId() {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        return sqlite3_last_insert_rowid(db);
    }
    
    /**
     * invoke sqlite3_table_column_metadata() function.
     * @param dbName
     * @param tableName
     * @param columnName
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://www.sqlite.org/c3ref/table_column_metadata.html">Extract Metadata About A Column Of A Table</a>
     */
    public ColumnMetaData getColumnMetaData(String dbName, String tableName, String columnName) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        SWIGTYPE_p_p_char dataType = null;
        SWIGTYPE_p_p_char collSeq = null;
        SWIGTYPE_p_int notNull = null;
        SWIGTYPE_p_int primaryKey = null;
        SWIGTYPE_p_int autoInc = null;
        try {
            dataType = new_p_p_char();
            collSeq = new_p_p_char();
            notNull = new_p_int();
            primaryKey = new_p_int();
            autoInc = new_p_int();
        
            int ret = sqlite3_table_column_metadata(
                            db, dbName, tableName, columnName,
                            dataType, collSeq, notNull, primaryKey, autoInc
                        );
            if (ret != SQLITE_OK) {
                throw new JdbcSQLException(db);
            }
            
            return new ColumnMetaData(
                            p_p_char_value(dataType),
                            p_p_char_value(collSeq),
                            p_int_value(notNull),
                            p_int_value(primaryKey),
                            p_int_value(autoInc)
                        );
        } finally {
            if (dataType != null) {
                delete_p_p_char(dataType);
            }
            if (collSeq != null) {
                delete_p_p_char(collSeq);
            }
            if (notNull != null) {
                delete_p_int(notNull);
            }
            if (primaryKey != null) {
                delete_p_int(primaryKey);
            }
            if (autoInc != null) {
                delete_p_int(autoInc);
            }
        }
    }
    
    void addStatement(Statement stmt) throws SQLException {
        if (statements == null) {
            statements = new HashMap<Long, Statement>();
        }
        final long key = stmt.getHandle();
        if (statements.containsKey(key)) {
            throw new SQLException("Duplicate sqlite3_stmt handle error.", "90J31");
        }
        statements.put(key, stmt);
    }
    
    void removeStatement(Statement stmt) throws SQLException {
        if (statements != null) {
            final long key = stmt.getHandle();
            if (statements.remove(key) == null) {
                throw new SQLException("Unmanaged sqlite3_stmt handle error.", "90J32");
            }
        }
    }
    
    private void closeStatements() {
        if (statements != null) {
            final Collection<Statement> list = statements.values();
            statements = null;
            for (final Statement stmt : list) {
                try {
                    stmt.close();
                } catch (SQLException ex) {
                    Logger.getLogger(Database.class.getName()).info(ex.toString());
                }
            }
        }
    }
    
    private void addFunction(Function func) {
        if (functions == null) {
            functions = new  HashMap<String, Function>();
        }
        functions.put(func.getName().toUpperCase(), func);
    }
    
    private void removeFunction(Function func) {
        if (functions != null) {
            functions.remove(func.getName().toUpperCase());
        }
    }
    
    private void dropFunctions() {
        if (functions != null) {
            final Collection<Function> list = functions.values();
            functions = null;
            for (final Function func : list) {
                if (func.isRegistered()) {
                    try {
                        dropFunction(func);
                    } catch (SQLException ex) {
                        Logger.getLogger(Database.class.getName()).info(ex.toString());
                    }
                }
            }
        }
    }
    
    /**
     * invoke sqlite3_create_function() function.
     * @param func User-Defined function
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/create_function.html">Create Or Redefine SQL Functions</a>
     */
    public void createFunction(Function func) throws SQLException {
        if (functions != null) {
            final Function old = functions.get(func.getName().toUpperCase());
            if (old != null) {
                // drop old function
                dropFunction(old);
            }
        }
        
        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = create_function(db, func);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        addFunction(func);
    }
    
    /**
     * invoke sqlite3_create_function() function.
     * @param func User-Defined function
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/create_function.html">Create Or Redefine SQL Functions</a>
     */
    public void dropFunction(Function func) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = drop_function(db, func);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        removeFunction(func);
    }
    
    private void addCollator(Collator col) {
        if (collators == null) {
            collators = new  HashMap<String, Collator>();
        }
        collators.put(col.getName().toUpperCase(), col);
    }
    
    private void removeCollator(Collator col) {
        if (collators != null) {
            collators.remove(col.getName().toUpperCase());
        }
    }
    
    private void dropCollators() {
        if (collators != null) {
            final Collection<Collator> list = collators.values();
            collators = null;
            for (final Collator col : list) {
                if (col.isRegistered()) {
                    try {
                        dropCollation(col);
                    } catch (SQLException ex) {
                        Logger.getLogger(Database.class.getName()).info(ex.toString());
                    }
                }
            }
        }
    }
    
    /**
     * invoke sqlite3_create_collation() function.
     * @param col User-Defined Collating Sequences
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/create_collation.html">Define New Collating Sequences</a>
     */
    public void createCollation(Collator col) throws SQLException {
        if (collators != null) {
            final Collator old = collators.get(col.getName().toUpperCase());
            if (old != null) {
                // drop old collation
                dropCollation(old);
            }
        }
        
        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = create_collation(db, col);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        addCollator(col);
    }
    
    /**
     * invoke sqlite3_create_function() function.
     * @param col User-Defined Collating Sequences
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/create_collation.html">Define New Collating Sequences</a>
     */
    public void dropCollation(Collator col) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = drop_collation(db, col);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        removeCollator(col);
    }
    
    /**
     * invoke sqlite3_set_authorizer() function.
     * @param auth
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/set_authorizer.html">Compile-Time Authorization Callbacks</a>
     */
    public void setAuthorizer(Authorizer auth) throws SQLException {
        clearAuthorizer();
        
        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = set_authorizer(db, auth);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        authorizer = auth;
    }
    
    /**
     * invoke sqlite3_set_authorizer() function.
     * @param auth
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/set_authorizer.html">Compile-Time Authorization Callbacks</a>
     */
    public void clearAuthorizer(Authorizer auth) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = clear_authorizer(db, auth);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
    }
    
    /**
     * invoke sqlite3_set_authorizer() function.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/set_authorizer.html">Compile-Time Authorization Callbacks</a>
     */
    public void clearAuthorizer() throws SQLException {
        if (authorizer != null) {
            clearAuthorizer(authorizer);
            authorizer = null;
        }
    }

    private void internalClearAuthorizer() {
        try {
            clearAuthorizer();
        } catch (SQLException ex) {
            Logger.getLogger(Database.class.getName()).info(ex.toString());
        }
    }
    
    /**
     * invoke sqlite3_busy_handler() function.
     * @param busy busy handler
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/busy_handler.html">Register A Callback To Handle SQLITE_BUSY Errors</a>
     */
    public void setBusyHandler(BusyHandler busy) throws SQLException {
        clearBusyHandler();
        
        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = set_busy_handler(db, busy);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        busyHandler = busy;
        timeout = -1;
    }
    
    /**
     * invoke sqlite3_busy_handler() function.
     * @param busy busy handler
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/busy_handler.html">Register A Callback To Handle SQLITE_BUSY Errors</a>
     */
    public void clearBusyHandler(BusyHandler busy) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = clear_busy_handler(db, busy);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
    }
    
    /**
     * invoke sqlite3_set_authorizer() function.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/busy_handler.html">Register A Callback To Handle SQLITE_BUSY Errors</a>
     */
    public void clearBusyHandler() throws SQLException {
        if (busyHandler != null) {
            clearBusyHandler(busyHandler);
            busyHandler = null;
        }
    }

    private void internalClearBusyHandler() {
        try {
            clearBusyHandler();
        } catch (SQLException ex) {
            Logger.getLogger(Database.class.getName()).info(ex.toString());
        }
    }
    
    /**
     * invoke sqlite3_progress_handler() function.
     * @param prog progress handler
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/progress_handler.html">Query Progress Callbacks</a>
     */
    public void setProgressHandler(ProgressHandler prog) {
        clearProgressHandler();
        
        final SWIGTYPE_p_sqlite3 db = getInstance();
        set_progress_handler(db, prog);
        progressHandler = prog;
    }
    
    /**
     * invoke sqlite3_progress_handler() function.
     * @param prog progress handler
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/progress_handler.html">Query Progress Callbacks</a>
     */
    public void clearProgressHandler(ProgressHandler prog) {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        clear_progress_handler(db, prog);
    }
    
    /**
     * invoke sqlite3_progress_handler() function.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/progress_handler.html">Query Progress Callbacks</a>
     */
    public void clearProgressHandler() {
        if (progressHandler != null) {
            clearProgressHandler(progressHandler);
            progressHandler = null;
        }
    }
    
    /**
     * invoke sqlite3_commit_hook() function.
     * @param hook commit hoot
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/commit_hook.html">Commit And Rollback Notification Callbacks</a>
     */
    public void setCommitHook(CommitHook hook) {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        set_commit_hook(db, hook);
        commitHook = hook;
    }
    
    /**
     * invoke sqlite3_commit_hook() function.
     * @param hook commit hoot
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/commit_hook.html">Commit And Rollback Notification Callbacks</a>
     */
    public void clearCommitHook(CommitHook hook) {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        clear_commit_hook(db, hook);
    }
    
    /**
     * invoke sqlite3_commit_hook() function.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/commit_hook.html">Commit And Rollback Notification Callbacks</a>
     */
    public void clearCommitHook() {
        if (commitHook != null) {
            if (commitHook.isRegistered()) {
                clearCommitHook(commitHook);
            }
            commitHook = null;
        }
    }
    
    /**
     * invoke sqlite3_rollback_hook() function.
     * @param hook rollback hoot
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/commit_hook.html">Commit And Rollback Notification Callbacks</a>
     */
    public void setRollbackHook(RollbackHook hook) {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        set_rollback_hook(db, hook);
        rollbackHook = hook;
    }
    
    /**
     * invoke sqlite3_rollback_hook() function.
     * @param hook rollback hoot
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/commit_hook.html">Commit And Rollback Notification Callbacks</a>
     */
    public void clearRollbackHook(RollbackHook hook) {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        clear_rollback_hook(db, hook);
    }
    
    /**
     * invoke sqlite3_rollback_hook() function.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/commit_hook.html">Commit And Rollback Notification Callbacks</a>
     */
    public void clearRollbackHook() {
        if (rollbackHook != null) {
            if (rollbackHook.isRegistered()) {
                clearRollbackHook(rollbackHook);
            }
            rollbackHook = null;
        }
    }
    
    /**
     * invoke sqlite3_update_hook() function.
     * @param hook update hoot
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/update_hook.html">Data Change Notification Callbacks</a>
     */
    public void setUpdateHook(UpdateHook hook) {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        set_update_hook(db, hook);
        updateHook = hook;
    }
    
    /**
     * invoke sqlite3_update_hook() function.
     * @param hook update hoot
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/update_hook.html">Data Change Notification Callbacks</a>
     */
    public void clearUpdateHook(UpdateHook hook) {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        clear_update_hook(db, hook);
    }
    
    /**
     * invoke sqlite3_update_hook() function.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/update_hook.html">Data Change Notification Callbacks</a>
     */
    public void clearUpdateHook() {
        if (updateHook != null) {
            if (updateHook.isRegistered()) {
                clearUpdateHook(updateHook);
            }
            updateHook = null;
        }
    }
    
    /**
     * invoke sqlite3_enable_shared_cache(true) function.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/enable_shared_cache.html">Enable Or Disable Shared Pager Cache</a>
     */
    public void enableSharedCache() throws SQLException {
        int ret = sqlite3_enable_shared_cache(1);
        if (ret != SQLITE_OK) {
            final SWIGTYPE_p_sqlite3 db = getInstance();
            throw new JdbcSQLException(db);
        }
    }
    
    /**
     * invoke sqlite3_enable_shared_cache(false) function.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/enable_shared_cache.html">Enable Or Disable Shared Pager Cache</a>
     */
    public void disableSharedCache() throws SQLException {
        int ret = sqlite3_enable_shared_cache(0);
        if (ret != SQLITE_OK) {
            final SWIGTYPE_p_sqlite3 db = getInstance();
            throw new JdbcSQLException(db);
        }
    }
    
    // TODO Statement毎にtimeoutが設定できるようにする！
    
    @Override
    protected void finalize() throws Throwable {
        if (!isClosed()) {
            // sqlite3_openを呼び出したスレッドと異なるスレッドから
            // sqlite3_closeを呼び出すことは出来ない（呼び出すとJVMが異常終了する）
            Logger.getLogger("global").severe("Database connection has leaked!");
        }
        super.finalize();
    }
    
}