/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.sqlite.jdbc;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.MalformedURLException;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Map;
import java.util.logging.Logger;
import static org.sqlite.swig.SQLite3Constants.*;

/**
 *
 * @author calico
 */
public class JdbcResultSet implements ResultSet {

    private final Statement owner;
    private final org.sqlite.Statement stmt;
    private boolean isClosed = false;
    private boolean isBeforeFirst = true;
    private boolean isFirst = false;
    private boolean isAfterLast = false;
    private int currentRow = 0;
    private int cntColumns = -1;
    private int cntRows = -1;
    private String[] columnLabels;
    private int[] columnTypes;
    private int lastColumnReadIndex;
    private JdbcBlob blob;
    
    public JdbcResultSet(Statement owner, org.sqlite.Statement stmt) throws SQLException  {
        this.owner = owner;
        this.stmt = stmt;
        stmt.reset();
    }

    // START implements
    public boolean next() throws SQLException {
        validateResultSetOpen();

        int ret = stmt.step();
        
        // close the Blob that has opened.
        if (blob != null) {
            blob.close();
            blob = null;
        }
        
        // TODO do implements!
        if (ret == SQLITE_ROW) {
            // has more records
            if (isBeforeFirst) {
                // first record
                isBeforeFirst = false;
                isFirst = true;
                if (cntColumns == -1) {
                    final int cnt = stmt.getDataCount();
                    cntColumns = cnt;
                    columnLabels = new String[cnt];
                    for (int i = 0; i < cnt; ++i) {
                        columnLabels[i] = stmt.getColumnLabel(i + 1);
                    }
                }
            } else {
                isFirst = false;
            }
            ++currentRow;
            return true;
            
        } else if (ret == SQLITE_DONE) {
            // no more records
            isAfterLast = true;
            isFirst = false;
            cntRows = currentRow;
        }
        
        return false;
    }

    public void close() throws SQLException {
        if (!isClosed) {
            isClosed = true;
            if (!(owner instanceof JdbcPreparedStatement)) {
                // ownerがJdbcPreparedStatementの場合はstmtをcloseしてしまうと
                // ownerも一緒にcloseされてしまうための対策
                stmt.close();
            }
        }
    }

    public boolean wasNull() throws SQLException {
        validateResultSetOpen();
        validateColumnIndexRange(lastColumnReadIndex);
        
        return (stmt.getString(lastColumnReadIndex) != null);
    }

    public String getString(int columnIndex) throws SQLException {
        validateResultSetOpen();
        validateColumnIndexRange(columnIndex);

        lastColumnReadIndex = columnIndex;
        return stmt.getString(columnIndex);
    }

    public boolean getBoolean(int columnIndex) throws SQLException {
        validateResultSetOpen();
        validateColumnIndexRange(columnIndex);

        lastColumnReadIndex = columnIndex;
        return (stmt.getInt(columnIndex) != 0);
    }

    public byte getByte(int columnIndex) throws SQLException {
        validateResultSetOpen();
        validateColumnIndexRange(columnIndex);

        lastColumnReadIndex = columnIndex;
        return (byte) stmt.getInt(columnIndex);
    }

    public short getShort(int columnIndex) throws SQLException {
        validateResultSetOpen();
        validateColumnIndexRange(columnIndex);

        lastColumnReadIndex = columnIndex;
        return (short) stmt.getInt(columnIndex);
    }

    public int getInt(int columnIndex) throws SQLException {
        validateResultSetOpen();
        validateColumnIndexRange(columnIndex);

        lastColumnReadIndex = columnIndex;
        return stmt.getInt(columnIndex);
    }

    public long getLong(int columnIndex) throws SQLException {
        validateResultSetOpen();
        validateColumnIndexRange(columnIndex);

        lastColumnReadIndex = columnIndex;
        return stmt.getLong(columnIndex);
    }

    public float getFloat(int columnIndex) throws SQLException {
        validateResultSetOpen();
        validateColumnIndexRange(columnIndex);

        lastColumnReadIndex = columnIndex;
        return (float) stmt.getDouble(columnIndex);
    }

    public double getDouble(int columnIndex) throws SQLException {
        validateResultSetOpen();
        validateColumnIndexRange(columnIndex);

        lastColumnReadIndex = columnIndex;
        return stmt.getDouble(columnIndex);
    }

    public BigDecimal getBigDecimal(int columnIndex, int scale) throws SQLException {
        validateResultSetOpen();
        validateColumnIndexRange(columnIndex);

        lastColumnReadIndex = columnIndex;
        BigDecimal ret = null;
        final String val = stmt.getString(columnIndex);
        if (val != null) {
            ret = new BigDecimal(val).setScale(scale);
        }
        return ret;
    }

    public byte[] getBytes(int columnIndex) throws SQLException {
        validateResultSetOpen();
        validateColumnIndexRange(columnIndex);

        lastColumnReadIndex = columnIndex;
        return stmt.getBytes(columnIndex);
    }

    public Date getDate(int columnIndex) throws SQLException {
        validateResultSetOpen();
        validateColumnIndexRange(columnIndex);

        lastColumnReadIndex = columnIndex;
        final String val = stmt.getString(columnIndex);
        Date ret = null;
        if (val != null) {
            ret = new Date(parseDateTimeString("yyyy-MM-dd", val));
        }
        return ret;
    }

    public Time getTime(int columnIndex) throws SQLException {
        validateResultSetOpen();
        validateColumnIndexRange(columnIndex);

        lastColumnReadIndex = columnIndex;
        final String val = stmt.getString(columnIndex);
        Time ret = null;
        if (val != null) {
            ret = new Time(parseDateTimeString("HH:mm:ss", val));
        }
        return ret;
    }

    public Timestamp getTimestamp(int columnIndex) throws SQLException {
        validateResultSetOpen();
        validateColumnIndexRange(columnIndex);

        lastColumnReadIndex = columnIndex;
        final String val = stmt.getString(columnIndex);
        Timestamp ret = null;
        if (val != null) {
            ret = new Timestamp(parseDateTimeString("yyyy-MM-dd HH:mm:ss", val));
        }
        return ret;
    }

    public InputStream getAsciiStream(int columnIndex) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public InputStream getUnicodeStream(int columnIndex) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public InputStream getBinaryStream(int columnIndex) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public String getString(String columnLabel) throws SQLException {
        return getString(findColumn(columnLabel));
    }

    public boolean getBoolean(String columnLabel) throws SQLException {
        return getBoolean(findColumn(columnLabel));
    }

    public byte getByte(String columnLabel) throws SQLException {
        return getByte(findColumn(columnLabel));
    }

    public short getShort(String columnLabel) throws SQLException {
        return getShort(findColumn(columnLabel));
    }

    public int getInt(String columnLabel) throws SQLException {
        return getInt(findColumn(columnLabel));
    }

    public long getLong(String columnLabel) throws SQLException {
        return getLong(findColumn(columnLabel));
    }

    public float getFloat(String columnLabel) throws SQLException {
        return getFloat(findColumn(columnLabel));
    }

    public double getDouble(String columnLabel) throws SQLException {
        return getDouble(findColumn(columnLabel));
    }

    public BigDecimal getBigDecimal(String columnLabel, int arg1) throws SQLException {
        return getBigDecimal(findColumn(columnLabel));
    }

    public byte[] getBytes(String columnLabel) throws SQLException {
        return getBytes(findColumn(columnLabel));
    }

    public Date getDate(String columnLabel) throws SQLException {
        return getDate(findColumn(columnLabel));
    }

    public Time getTime(String columnLabel) throws SQLException {
        return getTime(findColumn(columnLabel));
    }

    public Timestamp getTimestamp(String columnLabel) throws SQLException {
        return getTimestamp(findColumn(columnLabel));
    }

    public InputStream getAsciiStream(String columnLabel) throws SQLException {
        return getAsciiStream(findColumn(columnLabel));
    }

    public InputStream getUnicodeStream(String columnLabel) throws SQLException {
        return getUnicodeStream(findColumn(columnLabel));
    }

    public InputStream getBinaryStream(String columnLabel) throws SQLException {
        return getBinaryStream(findColumn(columnLabel));
    }

    public SQLWarning getWarnings() throws SQLException {
        validateResultSetOpen();

        return null;
    }

    public void clearWarnings() throws SQLException {
        validateResultSetOpen();
        
        // nothing
    }

    public String getCursorName() throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public ResultSetMetaData getMetaData() throws SQLException {
        return new JdbcResultSetMetaData(stmt);
    }

    public Object getObject(int columnIndex) throws SQLException {
        validateResultSetOpen();
        validateColumnIndexRange(columnIndex);

        if (columnTypes == null) {
            final int cnt = stmt.getColumnCount();
            columnTypes = new int[cnt];
            for (int i = 0; i < cnt; ++i) {
                columnTypes[i] = stmt.getColumnType(i + 1);
            }
        }
        
        final int type = columnTypes[columnIndex - 1];
        Object ret = null;
        String val = null;
        switch (type) {
            case SQLITE_INTEGER:
                val = getString(columnIndex);
                if (val != null) {
                    ret = Integer.parseInt(val);
                }
                break;

            case SQLITE_FLOAT:
                val = getString(columnIndex);
                if (val != null) {
                    ret = Double.parseDouble(val);
                }
                break;

            case SQLITE_TEXT:
                ret = getString(columnIndex);
                break;

            case SQLITE_BLOB:
                ret = getBlob(columnIndex);
                break;

            case SQLITE_NULL:
                // nothing
                break;

            default:
                throw new SQLException("Unknown column type '" + type + "'.");
        }
        
        return ret;
    }

    public Object getObject(String columnLabel) throws SQLException {
        return getObject(findColumn(columnLabel));
    }

    public int findColumn(String columnLabel) throws SQLException {
        for (int i = 0; i < columnLabels.length; ++i) {
            if (columnLabels[i].equalsIgnoreCase(columnLabel)) {
                return i + 1;
            }
        }
        throw new SQLException("Not found column '" + columnLabel + "'.");
    }

    public Reader getCharacterStream(int columnIndex) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public Reader getCharacterStream(String columnLabel) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public BigDecimal getBigDecimal(int columnIndex) throws SQLException {
        validateResultSetOpen();
        validateColumnIndexRange(columnIndex);

        lastColumnReadIndex = columnIndex;
        BigDecimal ret = null;
        final String val = stmt.getString(columnIndex);
        if (val != null) {
            ret = new BigDecimal(val);
        }
        return ret;
    }

    public BigDecimal getBigDecimal(String columnLabel) throws SQLException {
        return getBigDecimal(findColumn(columnLabel));
    }

    public boolean isBeforeFirst() throws SQLException {
        validateResultSetOpen();

        return isBeforeFirst;
    }

    public boolean isAfterLast() throws SQLException {
        validateResultSetOpen();

        return isAfterLast;
    }

    public boolean isFirst() throws SQLException {
        validateResultSetOpen();

        return isFirst;
    }

    public boolean isLast() throws SQLException {
        // TODO do implements!
        if (cntRows == -1) {
            final int row = currentRow;
            afterLast();
            absolute(row);
        }
        
        return (cntRows == currentRow);
    }

    public void beforeFirst() throws SQLException {
        validateResultSetOpen();

        if (!isBeforeFirst) {
            stmt.reset();
            isBeforeFirst = true;
            isFirst = false;
            isAfterLast = false;
            currentRow = 0;
        }
    }

    public void afterLast() throws SQLException {
        while (!next()) {
            // nothing
        }
    }

    public boolean first() throws SQLException {
        beforeFirst();
        return next();
    }

    public boolean last() throws SQLException {
        // TODO do implements!
        if (cntRows == -1) {
            afterLast();
        }
        
        return absolute(cntRows);
    }

    public int getRow() throws SQLException {
        validateResultSetOpen();

        return currentRow;
    }

    public boolean absolute(int row) throws SQLException {
        validateResultSetOpen();
        
        if (row > 0) {
            if (row < currentRow) {
                beforeFirst();
            }

            boolean ret = false;
            while (row > currentRow && (ret = next())) {
                // skip
            }
            return ret;
            
        } else {
            // TODO do implements!
            throw new UnsupportedOperationException("Not implemented yet.");            
        }
    }

    public boolean relative(int columnIndex) throws SQLException {
        // TODO do implements!
        throw new UnsupportedOperationException("Not implemented yet.");
    }

    public boolean previous() throws SQLException {
        // TODO do implements!
        throw new UnsupportedOperationException("Not implemented yet.");
    }

    public void setFetchDirection(int columnIndex) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public int getFetchDirection() throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void setFetchSize(int columnIndex) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public int getFetchSize() throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public int getType() throws SQLException {
        validateResultSetOpen();

        return TYPE_FORWARD_ONLY;
//        return TYPE_SCROLL_INSENSITIVE;
    }

    public int getConcurrency() throws SQLException {
        validateResultSetOpen();
        
        return CONCUR_READ_ONLY;
    }

    public boolean rowUpdated() throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public boolean rowInserted() throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public boolean rowDeleted() throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateNull(int columnIndex) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateBoolean(int columnIndex, boolean arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateByte(int columnIndex, byte arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateShort(int columnIndex, short arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateInt(int columnIndex, int arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateLong(int columnIndex, long arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateFloat(int columnIndex, float arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateDouble(int columnIndex, double arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateBigDecimal(int columnIndex, BigDecimal arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateString(int columnIndex, String arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateBytes(int columnIndex, byte[] arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateDate(int columnIndex, Date arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateTime(int columnIndex, Time arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateTimestamp(int columnIndex, Timestamp arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateAsciiStream(int columnIndex, InputStream arg1, int arg2) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateBinaryStream(int columnIndex, InputStream arg1, int arg2) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateCharacterStream(int columnIndex, Reader arg1, int arg2) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateObject(int columnIndex, Object arg1, int arg2) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateObject(int columnIndex, Object arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateNull(String columnLabel) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateBoolean(String columnLabel, boolean arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateByte(String columnLabel, byte arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateShort(String columnLabel, short arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateInt(String columnLabel, int arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateLong(String columnLabel, long arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateFloat(String columnLabel, float arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateDouble(String columnLabel, double arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateBigDecimal(String columnLabel, BigDecimal arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateString(String columnLabel, String arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateBytes(String columnLabel, byte[] arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateDate(String columnLabel, Date arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateTime(String columnLabel, Time arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateTimestamp(String columnLabel, Timestamp arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateAsciiStream(String columnLabel, InputStream arg1, int arg2) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateBinaryStream(String columnLabel, InputStream arg1, int arg2) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateCharacterStream(String columnLabel, Reader arg1, int arg2) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateObject(String columnLabel, Object arg1, int arg2) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateObject(String columnLabel, Object arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void insertRow() throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateRow() throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void deleteRow() throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void refreshRow() throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void cancelRowUpdates() throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void moveToInsertRow() throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void moveToCurrentRow() throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public Statement getStatement() throws SQLException {
        validateResultSetOpen();
        
        return owner;
    }

    public Object getObject(int columnIndex, Map<String, Class<?>> map) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public Ref getRef(int columnIndex) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public Blob getBlob(int columnIndex) throws SQLException {
        validateResultSetOpen();
        validateColumnIndexRange(columnIndex);
        
        // close the Blob that has opened.
        if (blob != null) {
            blob.close();
        }
        
        return (blob = new JdbcBlob(stmt, columnIndex));
    }

    public Clob getClob(int columnIndex) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public Array getArray(int columnIndex) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public Object getObject(String columnLabel, Map<String, Class<?>> map) throws SQLException {
        return getObject(findColumn(columnLabel), map);
    }

    public Ref getRef(String columnLabel) throws SQLException {
        return getRef(findColumn(columnLabel));
    }

    public Blob getBlob(String columnLabel) throws SQLException {
        return getBlob(findColumn(columnLabel));
    }

    public Clob getClob(String columnLabel) throws SQLException {
        return getClob(findColumn(columnLabel));
    }

    public Array getArray(String columnLabel) throws SQLException {
        return getArray(findColumn(columnLabel));
    }

    public Date getDate(int columnIndex, Calendar cal) throws SQLException {
        final Date val = getDate(columnIndex);
        Date ret = null;
        if (val != null) {
            final Calendar calendar = (Calendar) cal.clone();
            calendar.setTime(ret);
            ret = new Date(calendar.getTime().getTime());
        }
        return ret;
    }

    public Date getDate(String columnLabel, Calendar cal) throws SQLException {
        return getDate(findColumn(columnLabel), cal);
    }

    public Time getTime(int columnIndex, Calendar cal) throws SQLException {
        final Time val = getTime(columnIndex);
        Time ret = null;
        if (val != null) {
            final Calendar calendar = (Calendar) cal.clone();
            calendar.setTime(ret);
            ret = new Time(calendar.getTime().getTime());
        }
        return ret;
    }

    public Time getTime(String columnLabel, Calendar cal) throws SQLException {
        return getTime(findColumn(columnLabel), cal);
    }

    public Timestamp getTimestamp(int columnIndex, Calendar cal) throws SQLException {
        final Timestamp val = getTimestamp(columnIndex);
        Timestamp ret = null;
        if (val != null) {
            final Calendar calendar = (Calendar) cal.clone();
            calendar.setTime(ret);
            ret = new Timestamp(calendar.getTime().getTime());
        }
        return ret;
    }

    public Timestamp getTimestamp(String columnLabel, Calendar cal) throws SQLException {
        return getTimestamp(findColumn(columnLabel), cal);
    }

    public URL getURL(int columnIndex) throws SQLException {
        final String val = getString(columnIndex);
        URL ret = null;
        if (val != null) {
            try {
                ret = new URL(val);
            } catch (MalformedURLException ex) {
                Logger.getLogger(JdbcResultSet.class.getName()).throwing(JdbcResultSet.class.getName(), "getURL(int)", ex);
                throw new SQLException(ex.getMessage());
            }
        }
        return ret;
    }

    public URL getURL(String columnLabel) throws SQLException {
        return getURL(findColumn(columnLabel));
    }

    public void updateRef(int columnIndex, Ref arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateRef(String columnLabel, Ref arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateBlob(int columnIndex, Blob arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateBlob(String columnLabel, Blob arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateClob(int columnIndex, Clob arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateClob(String columnLabel, Clob arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateArray(int columnIndex, Array arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void updateArray(String columnLabel, Array arg1) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }
    // END implements

    public boolean isClosed() throws SQLException {
        return (isClosed || stmt.isClosed());
    }
    
    protected void validateResultSetOpen() throws SQLException {
        if (isClosed()) {
            throw new SQLException("ResultSet is already closed.");
        }
    }
    
    public static void validateResultSetType(int resultSetType) throws SQLException {
        if (resultSetType != JdbcResultSet.TYPE_FORWARD_ONLY
                && resultSetType != JdbcResultSet.TYPE_SCROLL_INSENSITIVE
                && resultSetType != JdbcResultSet.TYPE_SCROLL_SENSITIVE) {
            // 無効な値が指定されていた場合
            throw new SQLException("Not supported result set type.");
        }        
    }

    public static void validateResultSetConcurrency(int resultSetConcurrency) throws SQLException {
        if (resultSetConcurrency != JdbcResultSet.CONCUR_READ_ONLY
                && resultSetConcurrency != JdbcResultSet.CONCUR_UPDATABLE) {
            // 無効な値が指定されていた場合
            throw new SQLException("Not supported result set concurrency.");
        }
    }

    public static void validateResultSetHoldability(int resultSetHoldability) throws SQLException {
        if (resultSetHoldability != JdbcResultSet. HOLD_CURSORS_OVER_COMMIT) {
            // 無効な値が指定されていた場合
            throw new SQLException("Not supported result set holdability.");
        }
    }

    protected void validateColumnIndexRange(int columnIndex) throws SQLException {
        if (columnIndex > cntColumns || columnIndex < 1) {
            throw new SQLException("Column index out of range.");
        }
    }
    
    private static long parseDateTimeString(String pattern, String source) throws SQLException {
        final SimpleDateFormat formatter = new SimpleDateFormat(pattern);
        formatter.setLenient(false);
        final ParsePosition position = new ParsePosition(0);
        final java.util.Date date = formatter.parse(source, position);
        if (position.getErrorIndex() != -1
                || position.getIndex() != source.length()) {
            // パース失敗
            throw new SQLException("Format error.");
        }
        return date.getTime();
    }

}
