/* File : sqlite3jni.i */
%module SQLite3
%{
#include "../src/sqlite3.h"

#ifdef __cplusplus
extern "C" {
#endif

/* sqlite3_column_blob by byte[] */
JNIEXPORT jbyteArray JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1column_1blob_1by_1bytes(JNIEnv *jenv, jclass jcls, jlong stmt, jint col) {
    sqlite3_stmt *pStmt = *(sqlite3_stmt **)&stmt;
    jsize len;
    jbyteArray result;
    jbyte *pSrc;
    const void *blob = sqlite3_column_blob(pStmt, col);

    if (!blob) {
        return NULL;
    }

    len = sqlite3_column_bytes(pStmt, col);
    result = (*jenv)->NewByteArray(jenv, len);
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return NULL;
    }

    pSrc = (*jenv)->GetPrimitiveArrayCritical(jenv, result, 0);
    memcpy(pSrc, blob, len);
    (*jenv)->ReleasePrimitiveArrayCritical(jenv, result, pSrc, 0);

    return result;
}

#ifdef __cplusplus
}
#endif
%}

/* typemap for 'const unsigned char *' to 'java.lang.String' */
%typemap(jni) const unsigned char *    "jstring"
%typemap(jtype) const unsigned char *  "String"
%typemap(jstype) const unsigned char * "String"
%typemap(out, noblock=1) const unsigned char * { if($1) $result = JCALL1(NewStringUTF, jenv, (const char *)$1); }
%typemap(javaout) const unsigned char * {
    return $jnicall;
  }

/*
%pragma(java) jniclassclassmodifiers="public final class"
*/
%pragma(java) modulecode="
  public static final int SQLITE_TRANSIENT = -1;

  /**
   * Returns in-memory filename.
   * @return \":memory:\"
   * @see <a href=\"http://sqlite.org/c3ref/open.html\">Opening A New Database Connection</a>
   */
  public static String getInMemoryFileName() {
    return \":memory:\";
  }

  public static long addressOf(SWIGTYPE_p_sqlite3_stmt stmt) {
    return SWIGTYPE_p_sqlite3_stmt.getCPtr(stmt);
  }

  public static int sqlite3_exec(SWIGTYPE_p_sqlite3 db, String sql) {
    return SQLite3JNI.sqlite3_exec(SWIGTYPE_p_sqlite3.getCPtr(db), sql, 0, 0, 0);
  }

  public static int sqlite3_bind_text(SWIGTYPE_p_sqlite3_stmt stmt, int parameterIndex, String val) {
    return SQLite3JNI.sqlite3_bind_text(SWIGTYPE_p_sqlite3_stmt.getCPtr(stmt), parameterIndex, val, -1, SQLITE_TRANSIENT);
  }
"

%javaconst(1);
%javaconst(0) SQLITE_VERSION;
%javaconst(0) SQLITE_VERSION_NUMBER;
%include "../src/sqlite3.h"
%include "various.i"
%include "cpointer.i"

/*******************************************************************/
%exception new_p_p_sqlite3 {
    $action
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return $null;
    }
}

%exception new_p_p_sqlite3_stmt {
    $action
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return $null;
    }
}

/* Create some functions for working with "sqlite3**" */
%pointer_functions(sqlite3*, p_p_sqlite3);

/* Create some functions for working with "sqlite3_stmt**" */
%pointer_functions(sqlite3_stmt*, p_p_sqlite3_stmt);

%inline %{
typedef struct {
    sqlite3 *instance;   /* sqlite3 instance */
    JNIEnv *jenv;        /* Java environment for callbacks */
//    jobject callback;    /* sqlite3_callback listener object */
} SQLite3Handle;

typedef struct {
  char *dataType;               /* OUTPUT: Declared data type */
  char *collationSequenceName;  /* OUTPUT: Collation sequence name */
  jboolean notNull;             /* OUTPUT: True if NOT NULL constraint exists */
  jboolean primaryKey;          /* OUTPUT: True if column part of PK */
  jboolean autoIncrement;       /* OUTPUT: True if colums is auto-increment */
} SQLite3ColumnMetaData;

/* new SQLite3Handle structure  */
/*SQLite3Handle* new_SQLite3Handle() {
    return (SQLite3Handle*)malloc(sizeof(SQLite3Handle));
}*/

/* address-of operator function */
/* jlong address_of(jlong any) { */
/* void* address_of(void *any) {
    return (void*)(&any);
}
*/

static sqlite3** get_sqlite3_ptr_ptr(SQLite3Handle *handle) {
    return &(handle->instance);
}

/*
sqlite3** new_sqlite3_ptr() {
    return (sqlite3**)malloc(sizeof(sqlite3*));
}

sqlite3* get_sqlite3_ptr(sqlite3 **sqlite3) {
    return *sqlite3;
}

void delete_sqlite3_ptr(sqlite3 **sqlite3) {
    if (sqlite3) {
        free(sqlite3);
        sqlite3 = NULL;
    }
}

sqlite3_stmt** new_sqlite3_stmt_ptr() {
    return (sqlite3_stmt**)malloc(sizeof(sqlite3_stmt*));
}

sqlite3_stmt* get_sqlite3_stmt_ptr(sqlite3_stmt **stmt) {
    return *stmt;
}

void delete_sqlite3_stmt_ptr(sqlite3_stmt **stmt) {
    if (stmt) {
        free(stmt);
        stmt = NULL;
    }
}
*/

/* sqlite3_bind_blob by byte[] */
static int sqlite3_bind_blob_by_bytes(sqlite3_stmt *pStmt, int parameterIndex, char *BYTE, int length) {
    return sqlite3_bind_blob(pStmt, parameterIndex, BYTE, length, SQLITE_TRANSIENT);
}

/* sqlite3_table_column_metadata by SQLite3ColumnMetaData struct */
static int sqlite3_table_column_metadata_by_struct(sqlite3 *db, const char *zDbName, const char *zTableName, const char *zColumnName, SQLite3ColumnMetaData *meta) {
    return sqlite3_table_column_metadata(db, zDbName, zTableName, zColumnName, &(meta->dataType), &(meta->collationSequenceName), &(meta->notNull), &(meta->primaryKey), &(meta->autoIncrement));
}

/* sqlite3_column_blob by java.sql.Blob */
static void read_blob(const void *blob, long long pos, char *BYTE, int offset, int len) {
    memcpy(&BYTE[offset], &((const char *)blob)[pos], len);
}

/*
int get_sqlite3_magic(SQLite3Handle *handle) {
    return (handle->instance->magic);
}

sqlite3_stmt* get_sqlite3_stmt(SQLite3Handle *handle) {
    return (handle->stmt);
}
*/

%}

/* sqlite3_column_blob by byte[] */
%native(sqlite3_column_blob_by_bytes) jbyteArray sqlite3_column_blob_by_bytes(sqlite3_stmt*, int);

/*******************************************************************/
