package org.sqlite.jdbc;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Savepoint;
import org.sqlite.Database;
import java.util.Map;
import org.sqlite.TransactionType;

/**
 *
 * @author calico
 */
public class JdbcConnection implements Connection {

    private Database db;
    private TransactionType type;
    private final String url;
    
    public JdbcConnection(Database db, String url) throws SQLException {
        this.db = db;
        db.pragma(
                new String[] {
                    "encoding = \"UTF-8\"",
                    "count_changes = off",
                    "case_sensitive_like = on",
                    "short_column_names = on",
                    "empty_result_callbacks = off",
//                    "short_column_names = off",
//                    "full_column_names = on",
//                    "show_datatypes = on", for SQLite ver 2.x
                }
            );
        this.url = url;
    }
    
    // START implements
    public JdbcStatement createStatement() throws SQLException {
        validateConnectionOpen();
        
        return new JdbcStatement(db, this);
    }

    public JdbcPreparedStatement prepareStatement(String sql) throws SQLException {
        validateConnectionOpen();
        
        return new JdbcPreparedStatement(db, this, sql);
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public CallableStatement prepareCall(String sql) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public String nativeSQL(String sql) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void setAutoCommit(boolean autoCommit) throws SQLException {
        validateConnectionOpen();
        
        if (!db.getAutoCommit()) {
            rollback();
        }
        if (!autoCommit && db.getAutoCommit()) {
            db.beginTransaction(type);
        }
    }

    public boolean getAutoCommit() throws SQLException {
        validateConnectionOpen();
        
        return db.getAutoCommit();
    }

    public void commit() throws SQLException {
        validateConnectionOpen();
        
        if (db.getAutoCommit()) {
            throw new SQLException("Connection is auto commit mode.");
        }
        db.commitTransaction();
    }

    public void rollback() throws SQLException {
        validateConnectionOpen();
        
        if (db.getAutoCommit()) {
            throw new SQLException("Connection is auto commit mode.");
        }
        db.rollbackTransaction();
    }

    public void close() throws SQLException {
        if (!isClosed()) {
            if (!db.getAutoCommit()) {
                rollback();
            }
            db.close();
            db = null;
        }
    }

    public boolean isClosed() throws SQLException {
        return (db == null || db.isClosed());
    }

    public JdbcDatabaseMetaData getMetaData() throws SQLException {
        return new JdbcDatabaseMetaData(db, this, url);
    }

    public void setReadOnly(boolean readOnly) throws SQLException {
        // TODO バージョン 3.5.0 以降は sqlite3_open_v2() 関数で読取専用でオープンすることができる
        validateConnectionOpen();
        
        if (!db.getAutoCommit()) {
            throw new SQLException("Connection is transaction mode.");
        }
        if (readOnly) {
            throw new UnsupportedOperationException("Not supported yet.");
        }
    }

    public boolean isReadOnly() throws SQLException {
        validateConnectionOpen();
        
        return db.isReadOnly();
    }

    public void setCatalog(String catalog) throws SQLException {
        validateConnectionOpen();
        
        // nothing
    }

    /**
     * Catalog is not supported yet.
     * It always returns null.
     * @return null
     * @throws java.sql.SQLException
     */
    public String getCatalog() throws SQLException {
        validateConnectionOpen();
        
        return null;
    }

    public void setTransactionIsolation(int level) throws SQLException {
        validateConnectionOpen();
        
        if (level != TRANSACTION_SERIALIZABLE) {
            throw new SQLException("Not supported isolation level.");
        }
        // nothing
    }

    /**
     * It always returns TRANSACTION_SERIALIZABLE.
     * @return java.sql.Connection.TRANSACTION_SERIALIZABLE
     * @throws java.sql.SQLException
     */
    public int getTransactionIsolation() throws SQLException {
        validateConnectionOpen();
        
        return TRANSACTION_SERIALIZABLE;
    }

    /**
     * It always returns null.
     * @return null
     * @throws java.sql.SQLException
     */
    public SQLWarning getWarnings() throws SQLException {
        validateConnectionOpen();
        
        return null;
    }

    public void clearWarnings() throws SQLException {
        validateConnectionOpen();
        
        // nothing
    }

    public JdbcStatement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
        JdbcResultSet.validateResultSetType(resultSetType);
        JdbcResultSet.validateResultSetConcurrency(resultSetConcurrency);
        validateConnectionOpen();
        
        return new JdbcStatement(db, this);
    }

    public JdbcPreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        JdbcResultSet.validateResultSetType(resultSetType);
        JdbcResultSet.validateResultSetConcurrency(resultSetConcurrency);
        validateConnectionOpen();
        
        return new JdbcPreparedStatement(db, this, sql);        
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public Map<String, Class<?>> getTypeMap() throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public void setTypeMap(Map<String, Class<?>> map) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void setHoldability(int holdability) throws SQLException {
        validateConnectionOpen();

        if (holdability != ResultSet.CLOSE_CURSORS_AT_COMMIT) {
            throw new SQLException("Not supported holdability.");
        }
        // nothing
    }

    /**
     * It always returns CLOSE_CURSORS_AT_COMMIT.
     * @return java.sql.ResultSet.CLOSE_CURSORS_AT_COMMIT
     * @throws java.sql.SQLException
     */
    public int getHoldability() throws SQLException {
        validateConnectionOpen();

        return ResultSet.CLOSE_CURSORS_AT_COMMIT;
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public Savepoint setSavepoint() throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public Savepoint setSavepoint(String name) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public void rollback(Savepoint savepoint) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public void releaseSavepoint(Savepoint savepoint) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public JdbcStatement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        JdbcResultSet.validateResultSetType(resultSetType);
        JdbcResultSet.validateResultSetConcurrency(resultSetConcurrency);
        JdbcResultSet.validateResultSetHoldability(resultSetHoldability);
        validateConnectionOpen();

        return new JdbcStatement(db, this);
    }

    public JdbcPreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        JdbcResultSet.validateResultSetType(resultSetType);
        JdbcResultSet.validateResultSetConcurrency(resultSetConcurrency);
        JdbcResultSet.validateResultSetHoldability(resultSetHoldability);
        validateConnectionOpen();
        
        return new JdbcPreparedStatement(db, this, sql);        
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public JdbcPreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
        JdbcStatement.validateAutoGeneretedKeys(autoGeneratedKeys);
        validateConnectionOpen();

        return new JdbcPreparedStatement(db, this, sql);
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public JdbcPreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public JdbcPreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }
    // END implements
    
    protected void validateConnectionOpen() throws SQLException {
        if (isClosed()) {
            throw new SQLException("Connection is already closed.");
        }    
    }

    /**
     * 
     * @param type  'DEFERRED', 'IMMEDIATE', 'EXCLUSIVE'
     */
    public void setTransactionType(TransactionType type) {
        this.type = type;
    }
}
