using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Text;

using SystemNeo.IO;
using SystemNeo.Text;

using SystemNeo.Drawing.Imaging.Icc;

namespace SystemNeo.Drawing.Imaging.Tiff
{
	/// <summary>
	/// 
	/// </summary>
	public sealed class IfdTag
	{
		#region private static fields
		private const int codePageNameLength = 8;
		private static readonly IDictionary<string, Encoding> encodingDic
				= new Dictionary<string, Encoding>() {
					{"ASCII", Encoding.ASCII},
					{"JIS", Encoding.GetEncoding(Encodings.ISO_2022_JP)},
					{"UNICODE", Encoding.Unicode}
				};
		private static readonly IDictionary<int, IfdTag> tagDic;
		#endregion

		#region private fields
		private readonly object conversionParam;
		private readonly IfdConversionType conversionType;
		#endregion

		// public vpeB //

		/// <summary>
		/// 
		/// </summary>
		public ushort Id { get; private set; }

		/// <summary>
		/// 
		/// </summary>
		public string Name { get; private set; }

		// static RXgN^ //

		/// <summary>
		/// 
		/// </summary>
		static IfdTag()
		{
			tagDic = CreateTagDictionary();
		}

		// private RXgN^ //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="id"></param>
		/// <param name="name"></param>
		private IfdTag(ushort id, string name) : this(id, name, IfdConversionType.None) {}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="id"></param>
		/// <param name="name"></param>
		/// <param name="conversionType"></param>
		private IfdTag(ushort id, string name, IfdConversionType conversionType)
		{
			this.Id = id;
			this.Name = name;
			this.conversionType = conversionType;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="id"></param>
		/// <param name="name"></param>
		/// <param name="enumType"></param>
		private IfdTag(ushort id, string name, Type enumType)
				: this(id, name, IfdConversionType.Enum)
		{
			this.conversionParam = enumType;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="id"></param>
		/// <param name="name"></param>
		/// <param name="encoding"></param>
		private IfdTag(ushort id, string name, Encoding encoding)
				: this(id, name, IfdConversionType.Text)
		{
			this.conversionParam = encoding;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="id"></param>
		/// <param name="name"></param>
		/// <param name="ifdName"></param>
		private IfdTag(ushort id, string name, string ifdName)
				: this(id, name, IfdConversionType.Ifd)
		{
			this.conversionParam = ifdName;
		}

		// public static \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="id"></param>
		/// <returns></returns>
		public static IfdTag FromId(ushort id)
		{
			IfdTag tag;
			try {
				tag = tagDic[id];
			} catch (KeyNotFoundException) {
				tag = new IfdTag(id, null);
				tagDic.Add((int)id, tag);
			}
			return tag;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="id"></param>
		/// <returns></returns>
		public static IfdTag FromId(int id)
		{
			return FromId((ushort)id);
		}

		// public \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public override string ToString()
		{
			var sw = new StringWriter();
			sw.Write("0x");
			sw.Write(this.Id.ToString("X4"));
			if (this.Name != null) {
				sw.Write("=");
				sw.Write(this.Name);
			}
			return sw.ToString();
		}

		// internal \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="reader"></param>
		/// <param name="ifdList"></param>
		/// <param name="value"></param>
		/// <returns></returns>
		internal object ConvertValue(BinaryReaderNeo reader, IList<Ifd> ifdList, object value)
		{
			switch (this.conversionType) {
			case IfdConversionType.Enum:
				return Enum.ToObject((Type)this.conversionParam, value);
			case IfdConversionType.Text:
				Encoding encoding = (Encoding)this.conversionParam;
				return encoding.GetString((byte[])value).TrimEnd('\0');
			case IfdConversionType.EncodedText:
				if (value is byte[]) {
					return DecodeString((byte[])value);
				}
				break;
			case IfdConversionType.Ifd:
				string ifdName = (string)this.conversionParam;
				reader.Position = (uint)value;
				return Ifd.ReadIfd(reader, ifdName, ifdList);
			}
			return value;
		}

		// private static \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		private static IDictionary<int, IfdTag> CreateTagDictionary()
		{
			var dic = new Dictionary<int, IfdTag>();
			dic[0x0001] = new IfdTag(0x0001, "InteroperabilityIndex");
			dic[0x0002] = new IfdTag(0x0002, "InteroperabilityVersion", Encoding.ASCII);
			dic[0x000B] = new IfdTag(0x000B, "ProcessingSoftware");
			dic[0x00FE] = new IfdTag(0x00FE, "NewSubfileType");
			dic[0x00FF] = new IfdTag(0x00FF, "SubfileType", typeof(SubfileType));
			dic[0x0100] = new IfdTag(0x0100, "ImageWidth");
			dic[0x0101] = new IfdTag(0x0101, "ImageHeight");
			dic[0x0102] = new IfdTag(0x0102, "BitsPerSample");
			dic[0x0103] = new IfdTag(0x0103, "Compression", typeof(Compression));
			dic[0x0106] = new IfdTag(0x0106, "PhotometricInterpretation", typeof(PhotometricInterpretation));
			dic[0x0107] = new IfdTag(0x0107, "Thresholding", typeof(Thresholding));
			dic[0x0108] = new IfdTag(0x0108, "CellWidth");
			dic[0x0109] = new IfdTag(0x0109, "CellLength");
			dic[0x010A] = new IfdTag(0x010A, "FillOrder", typeof(FillOrder));
			dic[0x010D] = new IfdTag(0x010D, "DocumentName");
			dic[0x010E] = new IfdTag(0x010E, "ImageDescription");
			dic[0x010F] = new IfdTag(0x010F, "Make");
			dic[0x0110] = new IfdTag(0x0110, "Model");
			dic[0x0111] = new IfdTag(0x0111, "StripOffsets");
			dic[0x0112] = new IfdTag(0x0112, "Orientation", typeof(Orientation));
			dic[0x0115] = new IfdTag(0x0115, "SamplesPerPixel");
			dic[0x0116] = new IfdTag(0x0116, "RowsPerStrip");
			dic[0x0117] = new IfdTag(0x0117, "StripByteCounts");
			dic[0x0118] = new IfdTag(0x0118, "MinSampleValue");
			dic[0x0119] = new IfdTag(0x0119, "MaxSampleValue");
			dic[0x011A] = new IfdTag(0x011A, "XResolution");
			dic[0x011B] = new IfdTag(0x011B, "YResolution");
			dic[0x011C] = new IfdTag(0x011C, "PlanarConfiguration", typeof(PlanarConfiguration));
			dic[0x011D] = new IfdTag(0x011D, "PageName");
			dic[0x011E] = new IfdTag(0x011E, "XPosition");
			dic[0x011F] = new IfdTag(0x011F, "YPosition");
			dic[0x0120] = new IfdTag(0x0120, "FreeOffsets");
			dic[0x0121] = new IfdTag(0x0121, "FreeByteCounts");
			dic[0x0122] = new IfdTag(0x0122, "GrayResponseUnit");
			dic[0x0123] = new IfdTag(0x0123, "GrayResponseCurve");
			dic[0x0124] = new IfdTag(0x0124, "T4Options");
			dic[0x0125] = new IfdTag(0x0125, "T6Options");
			dic[0x0128] = new IfdTag(0x0128, "ResolutionUnit", typeof(ResolutionUnit));
			dic[0x0129] = new IfdTag(0x0129, "PageNumber");
			dic[0x012C] = new IfdTag(0x012C, "ColorResponseUnit");
			dic[0x012D] = new IfdTag(0x012D, "TransferFunction");
			dic[0x0131] = new IfdTag(0x0131, "Software");
			dic[0x0132] = new IfdTag(0x0132, "ModifyDate");
			dic[0x013B] = new IfdTag(0x013B, "Artist");
			dic[0x013C] = new IfdTag(0x013C, "HostComputer");
			dic[0x013D] = new IfdTag(0x013D, "Predictor");
			dic[0x013E] = new IfdTag(0x013E, "WhitePoint");
			dic[0x013F] = new IfdTag(0x013F, "PrimaryChromaticities");
			dic[0x0140] = new IfdTag(0x0140, "ColorMap");
			dic[0x0141] = new IfdTag(0x0141, "HalftoneHints");
			dic[0x0142] = new IfdTag(0x0142, "TileWidth");
			dic[0x0143] = new IfdTag(0x0143, "TileLength");
			dic[0x0144] = new IfdTag(0x0144, "TileOffsets");
			dic[0x0145] = new IfdTag(0x0145, "TileByteCounts");
			dic[0x0146] = new IfdTag(0x0146, "BadFaxLines");
			dic[0x0147] = new IfdTag(0x0147, "CleanFaxData");
			dic[0x0148] = new IfdTag(0x0148, "ConsecutiveBadFaxLines");
			dic[0x014A] = new IfdTag(0x014A, "SubIFD", "SubIFD");
			dic[0x014C] = new IfdTag(0x014C, "InkSet");
			dic[0x014D] = new IfdTag(0x014D, "InkNames");
			dic[0x014E] = new IfdTag(0x014E, "NumberOfInks");
			dic[0x0150] = new IfdTag(0x0150, "DotRange");
			dic[0x0151] = new IfdTag(0x0151, "TargetPrinter");
			dic[0x0152] = new IfdTag(0x0152, "ExtraSamples");
			dic[0x0153] = new IfdTag(0x0153, "SampleFormat");
			dic[0x0154] = new IfdTag(0x0154, "SMinSampleValue");
			dic[0x0155] = new IfdTag(0x0155, "SMaxSampleValue");
			dic[0x0156] = new IfdTag(0x0156, "TransferRange");
			dic[0x0157] = new IfdTag(0x0157, "ClipPath");
			dic[0x0158] = new IfdTag(0x0158, "XClipPathUnits");
			dic[0x0159] = new IfdTag(0x0159, "YClipPathUnits");
			dic[0x015A] = new IfdTag(0x015A, "Indexed");
			dic[0x015B] = new IfdTag(0x015B, "JPEGTables");
			dic[0x015F] = new IfdTag(0x015F, "OPIProxy");
			dic[0x0190] = new IfdTag(0x0190, "GlobalParametersIFD", "GlobalParametersIFD");
			dic[0x0191] = new IfdTag(0x0191, "ProfileType");
			dic[0x0192] = new IfdTag(0x0192, "FaxProfile");
			dic[0x0193] = new IfdTag(0x0193, "CodingMethods");
			dic[0x0194] = new IfdTag(0x0194, "VersionYear");
			dic[0x0195] = new IfdTag(0x0195, "ModeNumber");
			dic[0x01B1] = new IfdTag(0x01B1, "Decode");
			dic[0x01B2] = new IfdTag(0x01B2, "DefaultImageColor");
			dic[0x0200] = new IfdTag(0x0200, "JPEGProc", typeof(JpegProc));
			dic[0x0201] = new IfdTag(0x0201, "JPEGInterchangeFormatOffset");
			dic[0x0202] = new IfdTag(0x0202, "JPEGInterchangeFormatByteCount");
			dic[0x0203] = new IfdTag(0x0203, "JPEGRestartInterval");
			dic[0x0205] = new IfdTag(0x0205, "JPEGLosslessPredictors");
			dic[0x0206] = new IfdTag(0x0206, "JPEGPointTransforms");
			dic[0x0207] = new IfdTag(0x0207, "JPEGQTables");
			dic[0x0208] = new IfdTag(0x0208, "JPEGDCTables");
			dic[0x0209] = new IfdTag(0x0209, "JPEGACTables");
			dic[0x0211] = new IfdTag(0x0211, "YCbCrCoefficients");
			dic[0x0212] = new IfdTag(0x0212, "YCbCrSubSampling");
			dic[0x0213] = new IfdTag(0x0213, "YCbCrPositioning", typeof(YCbCrPositioning));
			dic[0x0214] = new IfdTag(0x0214, "ReferenceBlackWhite");
			dic[0x022F] = new IfdTag(0x022F, "StripRowCounts");
			dic[0x02BC] = new IfdTag(0x02BC, "ApplicationNotes");
			dic[0x1000] = new IfdTag(0x1000, "RelatedImageFileFormat");
			dic[0x1001] = new IfdTag(0x1001, "RelatedImageWidth");
			dic[0x1002] = new IfdTag(0x1002, "RelatedImageHeight");
			dic[0x4746] = new IfdTag(0x4746, "Rating");
			dic[0x4749] = new IfdTag(0x4749, "RatingPercent");
			dic[0x800D] = new IfdTag(0x800D, "ImageID");
			dic[0x80A4] = new IfdTag(0x80A4, "WangAnnotation");
			dic[0x80E3] = new IfdTag(0x80E3, "Matteing");
			dic[0x80E4] = new IfdTag(0x80E4, "DataType");
			dic[0x80E5] = new IfdTag(0x80E5, "ImageDepth");
			dic[0x80E6] = new IfdTag(0x80E6, "TileDepth");
			dic[0x827D] = new IfdTag(0x827D, "Model2");
			dic[0x828D] = new IfdTag(0x828D, "CFARepeatPatternDim");
			dic[0x828E] = new IfdTag(0x828E, "CFAPattern2");
			dic[0x828F] = new IfdTag(0x828F, "BatteryLevel");
			dic[0x8298] = new IfdTag(0x8298, "Copyright");
			dic[0x829A] = new IfdTag(0x829A, "ExposureTime");
			dic[0x829D] = new IfdTag(0x829D, "FNumber");
			dic[0x82A5] = new IfdTag(0x82A5, "MDFileTag");
			dic[0x82A6] = new IfdTag(0x82A6, "MDScalePixel");
			dic[0x82A7] = new IfdTag(0x82A7, "MDColorTable");
			dic[0x82A8] = new IfdTag(0x82A8, "MDLabName");
			dic[0x82A9] = new IfdTag(0x82A9, "MDSampleInfo");
			dic[0x82AA] = new IfdTag(0x82AA, "MDPrepDate");
			dic[0x82AB] = new IfdTag(0x82AB, "MDPrepTime");
			dic[0x82AC] = new IfdTag(0x82AC, "MDFileUnits");
			dic[0x830E] = new IfdTag(0x830E, "PixelScale");
			dic[0x83BB] = new IfdTag(0x83BB, "IPTC-NAA");
			dic[0x847E] = new IfdTag(0x847E, "IntergraphPacketData");
			dic[0x847F] = new IfdTag(0x847F, "IntergraphFlagRegisters");
			dic[0x8480] = new IfdTag(0x8480, "IntergraphMatrix");
			dic[0x8482] = new IfdTag(0x8482, "ModelTiePoint");
			dic[0x84E0] = new IfdTag(0x84E0, "Site");
			dic[0x84E1] = new IfdTag(0x84E1, "ColorSequence");
			dic[0x84E2] = new IfdTag(0x84E2, "IT8Header");
			dic[0x84E3] = new IfdTag(0x84E3, "RasterPadding");
			dic[0x84E4] = new IfdTag(0x84E4, "BitsPerRunLength");
			dic[0x84E5] = new IfdTag(0x84E5, "BitsPerExtendedRunLength");
			dic[0x84E6] = new IfdTag(0x84E6, "ColorTable");
			dic[0x84E7] = new IfdTag(0x84E7, "ImageColorIndicator");
			dic[0x84E8] = new IfdTag(0x84E8, "BackgroundColorIndicator");
			dic[0x84E9] = new IfdTag(0x84E9, "ImageColorValue");
			dic[0x84EA] = new IfdTag(0x84EA, "BackgroundColorValue");
			dic[0x84EB] = new IfdTag(0x84EB, "PixelIntensityRange");
			dic[0x84EC] = new IfdTag(0x84EC, "TransparencyIndicator");
			dic[0x84ED] = new IfdTag(0x84ED, "ColorCharacterization");
			dic[0x84EE] = new IfdTag(0x84EE, "HCUsage");
			dic[0x8546] = new IfdTag(0x8546, "SEMInfo");
			dic[0x8568] = new IfdTag(0x8568, "AFCP_IPTC");
			dic[0x85D8] = new IfdTag(0x85D8, "ModelTransform");
			dic[0x8606] = new IfdTag(0x8606, "LeafData");
			dic[0x8649] = new IfdTag(0x8649, "PhotoshopSettings");
			dic[0x8769] = new IfdTag(0x8769, "ExifIFDPointer", "ExifIFD");
			dic[0x8773] = new IfdTag(0x8773, "ICC_Profile");
			dic[0x87AC] = new IfdTag(0x87AC, "ImageLayer");
			dic[0x87AF] = new IfdTag(0x87AF, "GeoTiffDirectory");
			dic[0x87B0] = new IfdTag(0x87B0, "GeoTiffDoubleParams");
			dic[0x87B1] = new IfdTag(0x87B1, "GeoTiffAsciiParams");
			dic[0x8822] = new IfdTag(0x8822, "ExposureProgram", typeof(ExposureProgram));
			dic[0x8824] = new IfdTag(0x8824, "SpectralSensitivity");
			dic[0x8825] = new IfdTag(0x8825, "GPSInfo", "GPSInfoIFD");
			dic[0x8827] = new IfdTag(0x8827, "ISO");
			dic[0x8828] = new IfdTag(0x8828, "Opto-ElectricConvFactor");
			dic[0x8829] = new IfdTag(0x8829, "Interlace");
			dic[0x882A] = new IfdTag(0x882A, "TimeZoneOffset");
			dic[0x882B] = new IfdTag(0x882B, "SelfTimerMode");
			dic[0x885C] = new IfdTag(0x885C, "FaxRecvParams");
			dic[0x885D] = new IfdTag(0x885D, "FaxSubAddress");
			dic[0x885E] = new IfdTag(0x885E, "FaxRecvTime");
			dic[0x888A] = new IfdTag(0x888A, "LeafSubIFD", "LeafSubIFD");
			dic[0x9000] = new IfdTag(0x9000, "ExifVersion", Encoding.ASCII);
			dic[0x9003] = new IfdTag(0x9003, "DateTimeOriginal");
			dic[0x9004] = new IfdTag(0x9004, "DateTimeDigitized");
			dic[0x9101] = new IfdTag(0x9101, "ComponentsConfiguration");
			dic[0x9102] = new IfdTag(0x9102, "CompressedBitsPerPixel");
			dic[0x9201] = new IfdTag(0x9201, "ShutterSpeedValue");
			dic[0x9202] = new IfdTag(0x9202, "ApertureValue");
			dic[0x9203] = new IfdTag(0x9203, "BrightnessValue");
			dic[0x9204] = new IfdTag(0x9204, "ExposureBiasValue");
			dic[0x9205] = new IfdTag(0x9205, "MaxApertureValue");
			dic[0x9206] = new IfdTag(0x9206, "SubjectDistance");
			dic[0x9207] = new IfdTag(0x9207, "MeteringMode", typeof(MeteringMode));
			dic[0x9208] = new IfdTag(0x9208, "LightSource", typeof(LightSource));
			dic[0x9209] = new IfdTag(0x9209, "Flash", typeof(Flash));
			dic[0x920A] = new IfdTag(0x920A, "FocalLength");
			dic[0x920B] = new IfdTag(0x920B, "FlashEnergy");
			dic[0x920C] = new IfdTag(0x920C, "SpecialFrequencyResponse");
			dic[0x920D] = new IfdTag(0x920D, "Noise");
			dic[0x920E] = new IfdTag(0x920E, "FocalPlaneXResolution");
			dic[0x920F] = new IfdTag(0x920F, "FocalPlaneYResolution");
			dic[0x9210] = new IfdTag(0x9210, "FocalPlaneResolutionUnit", typeof(ResolutionUnit));
			dic[0x9211] = new IfdTag(0x9211, "ImageNumber");
			dic[0x9212] = new IfdTag(0x9212, "SecurityClassification");
			dic[0x9213] = new IfdTag(0x9213, "ImageHistory");
			dic[0x9214] = new IfdTag(0x9214, "SubjectLocation");
			dic[0x9215] = new IfdTag(0x9215, "ExposureIndex");
			dic[0x9216] = new IfdTag(0x9216, "TIFF-EPStandardID");
			dic[0x9217] = new IfdTag(0x9217, "SensingMethod", typeof(SensingMethod));
			dic[0x923F] = new IfdTag(0x923F, "StoNits");
			dic[0x927C] = new IfdTag(0x927C, "MakerNote");
			dic[0x9286] = new IfdTag(0x9286, "UserComment", IfdConversionType.EncodedText);
			dic[0x9290] = new IfdTag(0x9290, "SubSecTime");
			dic[0x9291] = new IfdTag(0x9291, "SubSecTimeOriginal");
			dic[0x9292] = new IfdTag(0x9292, "SubSecTimeDigitized");
			dic[0x935C] = new IfdTag(0x935C, "ImageSourceData");
			dic[0x9C9B] = new IfdTag(0x9C9B, "XPTitle", Encoding.Unicode);
			dic[0x9C9C] = new IfdTag(0x9C9C, "XPComponent", Encoding.Unicode);
			dic[0x9C9D] = new IfdTag(0x9C9D, "XPAuthor", Encoding.Unicode);
			dic[0x9C9E] = new IfdTag(0x9C9E, "XPKeywords", Encoding.Unicode);
			dic[0x9C9F] = new IfdTag(0x9C9F, "XPSubject", Encoding.Unicode);
			dic[0xA000] = new IfdTag(0xA000, "FlashPixVersion", Encoding.ASCII);
			dic[0xA001] = new IfdTag(0xA001, "ColorSpace", typeof(ColorSpace));
			dic[0xA002] = new IfdTag(0xA002, "ExifImageWidth");
			dic[0xA003] = new IfdTag(0xA003, "ExifImageHeight");
			dic[0xA004] = new IfdTag(0xA004, "RelatedSoundFile");
			dic[0xA005] = new IfdTag(0xA005, "InteroperabilityIFDPointer", "InteroperabilityIFD");
			dic[0xA20E] = new IfdTag(0xA20E, "FocalPlaneXResolution");
			dic[0xA20F] = new IfdTag(0xA20F, "FocalPlaneYResolution");
			dic[0xA210] = new IfdTag(0xA210, "FocalPlaneResolutionUnit", typeof(ResolutionUnit));
			dic[0xA211] = new IfdTag(0xA211, "ImageNumber");
			dic[0xA212] = new IfdTag(0xA212, "SecurityClassification");
			dic[0xA213] = new IfdTag(0xA213, "ImageHistory");
			dic[0xA214] = new IfdTag(0xA214, "SubjectLocation");
			dic[0xA215] = new IfdTag(0xA215, "ExposureIndex");
			dic[0xA216] = new IfdTag(0xA216, "TIFF-EPStandardID");
			dic[0xA217] = new IfdTag(0xA217, "SensingMethod", typeof(SensingMethod));
			dic[0xA300] = new IfdTag(0xA300, "FileSource");
			dic[0xA301] = new IfdTag(0xA301, "SceneType");
			dic[0xA302] = new IfdTag(0xA302, "CFAPattern");
			dic[0xA401] = new IfdTag(0xA401, "CustomImageProcessing", typeof(CustomImageProcessing));
			dic[0xA402] = new IfdTag(0xA402, "ExposureMode", typeof(ExposureMode));
			dic[0xA403] = new IfdTag(0xA403, "WhiteBalance", typeof(WhiteBalance));
			dic[0xA404] = new IfdTag(0xA404, "DigitalZoomRatio");
			dic[0xA405] = new IfdTag(0xA405, "FocalLengthIn35mmFilm");
			dic[0xA406] = new IfdTag(0xA406, "SceneCaptureType", typeof(SceneCaptureType));
			dic[0xA407] = new IfdTag(0xA407, "GainControl", typeof(GainControl));
			dic[0xA408] = new IfdTag(0xA408, "Contrast", typeof(Contrast));
			dic[0xA409] = new IfdTag(0xA409, "Saturation", typeof(Saturation));
			dic[0xA40A] = new IfdTag(0xA40A, "Sharpness", typeof(Sharpness));
			dic[0xA40B] = new IfdTag(0xA40B, "DeviceSettingsDescription");
			dic[0xA40C] = new IfdTag(0xA40C, "SubjectDistanceRange", typeof(SubjectDistanceRange));
			dic[0xA420] = new IfdTag(0xA420, "ImageUniqueID");
			dic[0xA480] = new IfdTag(0xA480, "GDALMetaData");
			dic[0xA481] = new IfdTag(0xA481, "GDALNoData");
			dic[0xA500] = new IfdTag(0xA500, "Gamma");
			return dic;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="bytes"></param>
		/// <returns></returns>
		private static string DecodeString(byte[] bytes)
		{
			string s = Encoding.ASCII.GetString(bytes, 0, codePageNameLength);
			string codePageName = s.TrimEnd('\0', ' ');
			if (codePageName.Length == 0) {
				return string.Empty;
			} else {
				var encoding = (Encoding)encodingDic[codePageName];
				if (encoding == null) {
					string msg = string.Format(
							"R[hy[W '{0}' ̓T|[g܂B", codePageName);
					throw new NotSupportedException(msg);
				}
				int length = bytes.Length - codePageNameLength;
				return encoding.GetString(bytes, codePageNameLength, length);
			}
		}
	}
}
