using System;
using SystemNeo.Text;
using NUnit.Framework;

namespace SystemNeoTest.Text.StringReaderNeoTest
{
	[TestFixture]
	public class Position
	{
		[Test]
		public void Normal()
		{
			StringReaderNeo sr = new StringReaderNeo("ABCD");
			Assert.AreEqual(0, sr.Position);
			sr.Read();
			Assert.AreEqual(1, sr.Position);
			sr.Read();
			Assert.AreEqual(2, sr.Position);
			sr.Read(2);
			Assert.AreEqual(4, sr.Position);
			sr.Read(2);
			Assert.AreEqual(4, sr.Position);
			sr.Seek(-1);
			Assert.AreEqual(3, sr.Position);
		}
	}

	[TestFixture]
	public class Value
	{
		[Test]
		public void Normal()
		{
			StringReaderNeo sr = new StringReaderNeo("ABCD");
			Assert.AreEqual("ABCD", sr.Value);
		}
	}

	[TestFixture]
	public class Read
	{
		[Test]
		public void Normal()
		{
			StringReaderNeo sr = new StringReaderNeo("ABCD");
			Assert.AreEqual((int)'A', sr.Read());
			Assert.AreEqual((int)'B', sr.Read());
			Assert.AreEqual((int)'C', sr.Read());
			Assert.AreEqual((int)'D', sr.Read());
			Assert.AreEqual(-1, sr.Read());
			Assert.AreEqual(-1, sr.Read());
		}
	}

	[TestFixture]
	public class Read_Int32
	{
		[Test]
		public void Normal()
		{
			StringReaderNeo sr = new StringReaderNeo("ABCD");
			Assert.AreEqual("",   sr.Read(0));
			Assert.AreEqual("A",  sr.Read(1));
			Assert.AreEqual("BC", sr.Read(2));
			Assert.AreEqual("D",  sr.Read(2));
			Assert.AreEqual("",   sr.Read(2));
		}
	}

	[TestFixture]
	public class Read_CharFilter
	{
		[Test]
		public void Normal()
		{
			StringReaderNeo sr = new StringReaderNeo("ABC123");
			Assert.AreEqual("",    sr.Read(new Predicate<char>(char.IsDigit)));
			Assert.AreEqual("ABC", sr.Read(new Predicate<char>(char.IsLetter)));
			Assert.AreEqual("",    sr.Read(new Predicate<char>(char.IsLetter)));
			Assert.AreEqual("123", sr.Read(new Predicate<char>(char.IsDigit)));
			Assert.AreEqual("",    sr.Read(new Predicate<char>(char.IsDigit)));
		}
	}

	[TestFixture]
	public class Peek
	{
		[Test]
		public void Normal()
		{
			StringReaderNeo sr = new StringReaderNeo("ABCD");
			Assert.AreEqual((int)'A', sr.Peek());
			Assert.AreEqual((int)'A', sr.Peek());
			sr.Read(2);
			Assert.AreEqual((int)'C', sr.Peek());
			Assert.AreEqual((int)'C', sr.Peek());
			sr.Read(2);
			Assert.AreEqual(-1, sr.Peek());
		}
	}

	[TestFixture]
	public class Peek_Int32
	{
		[Test]
		public void Normal()
		{
			StringReaderNeo sr = new StringReaderNeo("ABCD");
			Assert.AreEqual("", sr.Peek(0));
			Assert.AreEqual("A", sr.Peek(1));
			Assert.AreEqual("AB", sr.Peek(2));
			Assert.AreEqual("ABCD", sr.Peek(10));
			sr.Read(2);
			Assert.AreEqual("CD", sr.Read(2));
			sr.Read(2);
			Assert.AreEqual("", sr.Read(2));
		}
	}

	[TestFixture]
	public class Peek_PredicateChar
	{
		[Test]
		public void Normal()
		{
			StringReaderNeo sr = new StringReaderNeo("ABC 123");
			Assert.AreEqual("",    sr.Peek(new Predicate<char>(char.IsDigit)));
			Assert.AreEqual("ABC", sr.Peek(new Predicate<char>(char.IsLetter)));
			sr.Seek(3);
			Assert.AreEqual("",    sr.Peek(new Predicate<char>(char.IsLetter)));
			Assert.AreEqual(" ",   sr.Peek(new Predicate<char>(char.IsWhiteSpace)));
			sr.Seek(1);
			Assert.AreEqual("",    sr.Peek(new Predicate<char>(char.IsWhiteSpace)));
			Assert.AreEqual("123", sr.Peek(new Predicate<char>(char.IsDigit)));
			sr.Seek(3);
			Assert.AreEqual("",    sr.Peek(new Predicate<char>(char.IsDigit)));
		}
	}

	[TestFixture]
	public class FindPeek
	{
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void @null()
		{
			StringReaderNeo sr = new StringReaderNeo("ABCD");
			sr.FindPeek(null);
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void 󕶎()
		{
			StringReaderNeo sr = new StringReaderNeo("ABCD");
			sr.FindPeek("");
		}

		[Test]
		public void Normal()
		{
			StringReaderNeo sr = new StringReaderNeo("ABCD");
			Assert.AreEqual("A",  sr.FindPeek("BC"));
			Assert.AreEqual("AB", sr.FindPeek("C"));
			Assert.AreEqual("",   sr.FindPeek("BCG"));
			sr.Seek(1);
			Assert.AreEqual("",   sr.FindPeek("AB"));
			Assert.AreEqual("",   sr.FindPeek("BC"));
			Assert.AreEqual("B",  sr.FindPeek("C"));
			Assert.AreEqual("BC", sr.FindPeek("D"));
			sr.Seek(3);
			Assert.AreEqual("",   sr.FindPeek("E"));
		}
	}

	[TestFixture]
	public class Seek
	{
		[Test]
		public void Normal()
		{
			StringReaderNeo sr = new StringReaderNeo("ABCD");
			Assert.AreEqual(0, sr.Position);
			sr.Seek(1);
			Assert.AreEqual(1, sr.Position);
			sr.Seek(2);
			Assert.AreEqual(3, sr.Position);
			sr.Seek(2);
			Assert.AreEqual(4, sr.Position);
			sr.Seek(-2);
			Assert.AreEqual(2, sr.Position);
			sr.Seek(-4);
			Assert.AreEqual(0, sr.Position);
		}
	}

	[TestFixture]
	public class StartsWith
	{
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void 󕶎()
		{
			StringReaderNeo sr = new StringReaderNeo("ABCD");
			Assert.False(sr.StartsWith(""));
		}

		[Test]
		public void Normal()
		{
			StringReaderNeo sr = new StringReaderNeo("ABCD");
			Assert.True(sr.StartsWith("A"));
			Assert.True(sr.StartsWith("ABC"));
			Assert.False(sr.StartsWith("ABCDE"));
			sr.Seek(2);
			Assert.False(sr.StartsWith("A"));
			Assert.True(sr.StartsWith("CD"));
		}
	}
}
