/*
 * Copyright (c) 2010-2013 Yuichi Watanabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of Yuichi Watanabe nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <core/assert.h>
#include <core/cpu.h>
#include <core/extint.h>
#include <core/printf.h>
#include "apic.h"
#include "bitmap.h"
#include "cpu_emul.h"
#include "constants.h"
#include "current.h"
#include "panic.h"

#define EXTINT_DEBUG 1
#ifdef EXTINT_DEBUG
#define EXTINT_DBG(level, ...) \
	do {							\
		if (level <= EXTINT_DEBUG) {			\
			printf(__VA_ARGS__);			\
		}						\
	} while (0)
#else
#define EXTINT_DBG(...)
#endif

void
extint_pend_apic_int(vector_t vector)
{
	long bsr;

	EXTINT_DBG(2, "R%d %d\n", get_cpu_id(), vector);

	bsr = BITMAP_BSR(&current->extint.pending_apic_int);
	if (vector <= bsr) {
		panic ("A apic interrupt %d received, though a interrupt %ld is pending\n",
		       vector, bsr);
	}

	bsr = BITMAP_BSR(&current->extint.inservice_apic_int);
	if (vector <= bsr) {
		panic ("A apic interrupt %d received, though a interrupt %ld is in-service\n",
		       vector, bsr);
	}

	if (BITMAP_SET(&current->extint.pending_apic_int, vector)) {
		EXTINT_DBG(1, "A external interrupt %d (apic) received while the same interrupt is pending.\n",
			   vector);
	}
	current->halt = false;
}

void
extint_pend_8259a_int(vector_t vector)
{
	EXTINT_DBG(2, "R%d 8259a %d\n", get_cpu_id(), vector);

	if (BITMAP_SET(&current->extint.pending_8259a_int, vector)) {
		EXTINT_DBG(2, "A external interrupt %d (8259a) received while the same interrupt is pending.\n",
			   vector);
	}
	current->halt = false;
}

void
extint_pend_interrupt(vector_t vector)
{
	if (current == NULL) {
		panic ("current is NULL in extint_pend");
	}

	if (!current->vbsp) {
		extint_pend_apic_int(vector);
	} else if (apic_check_apic_interrupt(vector)) {
		extint_pend_apic_int(vector);
	} else {
		extint_pend_8259a_int(vector);
	}
}

/*
 * The caller should disable interrupts (CLI) before calling this
 * function until vmlaunch/vmresume to avoid losing interrupts.
 */
void
extint_check_interrupt(void)
{
	long vector = -1;
	bool apic_int = false;
	long inservice;
	ulong cr0;
	vmmerr_t err;

	/*
	 * Get the highest priority interrupt. We assume 8259a
	 * interrupts are higher than apic interrupts.
	 */
	if (current->vbsp) {
		vector = BITMAP_BSR(&current->extint.pending_8259a_int);
	}
	if (vector < 0) {
		apic_int = true;
		vector = BITMAP_BSR(&current->extint.pending_apic_int);
		if (vector >= 0) {
			inservice = BITMAP_BSR(
				&current->extint.inservice_apic_int);
			if (vector <= inservice) {
				/*
				 * A higher priority interrupt is in service.
				 * Pend the lower interrupt until a guest
				 * software writes EOI.
				 */
				vector = -1;
			}
		}
	}
	if (vector < 0) {
		/*
		 * There is no pending external interrupt to inject as
		 * soon as possible.
		 */
		current->vmctl.extint_pending(false);
		return;
	}
	if (current->vmctl.extint_is_blocked ()) {
		/*
		 * A external interrupt is blocked. Make VM-exit occur
		 * at the beginning of next interrupt window.
		 */
		current->vmctl.extint_pending(true);
		return;
	}

	/*
	 * A external interrupt is not blocked. Inject it.
	 */
	if (apic_int) {
		BITMAP_CLEAR(&current->extint.pending_apic_int, vector);
	} else {
		BITMAP_CLEAR(&current->extint.pending_8259a_int, vector);
	}

	current->vmctl.read_control_reg(CONTROL_REG_CR0,
					 &cr0);
	if (cr0 & CR0_PE_BIT) {
		current->vmctl.generate_external_int(vector);
	} else {
		err = cpu_emul_realmode_int(vector);
		if (err != VMMERR_SUCCESS) {
			panic("vt_check_extint: cpu_emul_realmode_int retured error %d",
			      err);
		}
	}
	if (apic_int) {
		EXTINT_DBG(2, "I%d apic %ld\n", get_cpu_id(), vector);
		/*
		 * The injected interrupt is apic interrupt. A guest
		 * software will write EOI to local APIC so that the
		 * VMM can inject a next interrupt on EOI.
		 */
		current->vmctl.extint_pending(false);
		/*
		 * Recode the interrupt as in-service to manage EOI.
		 */
		BITMAP_SET(&current->extint.inservice_apic_int, vector);
	} else {
		EXTINT_DBG(2, "I%d 8259a %ld\n", get_cpu_id(), vector);
		vector = BITMAP_BSR (&current->extint.pending_8259a_int);
		if (vector < 0) {
			vector = BITMAP_BSR (
				&current->extint.pending_apic_int);
		}
		if (vector >= 0) {
			/*
			 * There are some pending interrupts and the
			 * injected interrupt is 8259a interrupt. A
			 * guest software will not write EOI to local
			 * APIC. Make VM-exit occur at the beginning
			 * of next interrupt window to inject a next
			 * interrupt.
			 */
			current->vmctl.extint_pending(true);
		} else {
			current->vmctl.extint_pending(false);
		}
	}
}

void
extint_apic_eoi(void)
{
	long eoi_vector, pending_vector;

	eoi_vector = BITMAP_BSR(&current->extint.inservice_apic_int);
	if (eoi_vector >= 0) {
		BITMAP_CLEAR(&current->extint.inservice_apic_int, eoi_vector);
		BITMAP_SET(&current->extint.pending_eoi, eoi_vector);
	} else {
		EXTINT_DBG(1, "Null EOI on cpu%d\n", get_cpu_id());
	}

	pending_vector = BITMAP_BSR(&current->extint.pending_apic_int);
	while ((eoi_vector = BITMAP_BSR(&current->extint.pending_eoi)) >= 0) {
		if (eoi_vector <= pending_vector) {
			break;
		}
		EXTINT_DBG(2, "E%d apic %ld\n", get_cpu_id(), eoi_vector);
		BITMAP_CLEAR(&current->extint.pending_eoi, eoi_vector);
		apic_write_eoi();
	}
}

void
extint_send_apic_int(vector_t vector, apic_id_t dest)
{
	apic_send_ipi(vector, dest);
}

void
extint_reset_8259a(void)
{
	ASSERT(current->vbsp);

	EXTINT_DBG(2, "Reset 8259a");
	BITMAP_INIT(&current->extint.pending_8259a_int);
}

void
extint_reset_apic(void)
{
	EXTINT_DBG(2, "Reset APIC");

	long vector;

	while ((vector = BITMAP_BSR(&current->extint.inservice_apic_int))
	       >= 0) {
		BITMAP_CLEAR(&current->extint.inservice_apic_int,
			     vector);
		apic_write_eoi();
	}

	while ((vector = BITMAP_BSR(&current->extint.pending_eoi)) >= 0) {
		BITMAP_CLEAR(&current->extint.pending_eoi, vector);
		apic_write_eoi();
	}

	while ((vector = BITMAP_BSR(&current->extint.pending_apic_int))
	       >= 0) {
		BITMAP_CLEAR(&current->extint.pending_apic_int, vector);
		apic_write_eoi();
	}
}
