/**
 * @file vector.c
 * @date 2010/08/08
 * @brief  CORTEX-M3用の例外ベクトル。
 * @author takemasa
 * @details
 * FLASH ROMへ書き込む値はこのベクトルで宣言する。ここに書き込んだ初期値は、
 * FLASH ROMの先頭番地へ書き込まれる。先頭番地を強制するために、セクション
 * .vectorを使っている。このセクションに対する配置はLDスクリプトで制御する。
 */
#include "LPC17xx.h"
#include "vector.h"

/**
 * @brief 例外ベクトル
 * @details
 * CORTEX-M3の例外ベクトルにかぶせて使う変数。SRAM版では任意の(256バイト境界の)
 * アドレスにおいて使う。ROM版でも同じだが、SRAMにコピーして使う。コピーのための
 * コードがstart.Sにおいてある。また、そのための仕掛けがLDファイルにある。
 *
 * 初期値として持っているのは最低限リセット後のブートに必要なmaster_stackのポインタ
 * 値と、startアドレスのみである。それ以外のベクトル・エントリは実行中に設定している。
 * ただし、初期値として持つことも可能である。
 */
unsigned int exception_vector[64] __attribute__ ((section (".vector")))=
		{
				(unsigned int)&_master_stack,	/* CORTEX-M3の最初のベクトルはスタック・トップ */
				(unsigned int)&_start			/* CORTEX-M3の２番目のベクトルは実行開始番地 */
		};

/**
 * @brief 割り込みの初期化
 * @details
 * NVICのベクトルにハンドラを登録して、割り込み準備を行う。VTORレジスタは常にSRAM上の
 * 割り込みベクトルに設定するので、このベクトルは正しく配置するよう気を遣わなければならない。
 */
void init_interrupt(void)
{
		// CORTEX-M3コアの割り込みを禁止する
	asm volatile(	"mov r0, #1 \n"
					"msr primask, r0 \n"	// すべての割り込みを禁止する
					"mov r0, #0 \n"
					"msr basepri, r0"		// 割り込みレベルによる禁止はしない
					:
					:
					: "r0" );
		// 例外ベクトルをexception_vectorに重ねる。
		// 内容のコピーはstart.Sですでに行っている。
	SCB->VTOR = (unsigned int)exception_vector;
		// UART0受信およびtimer0割り込みハンドラの設定
		// 添え字の"+16"は、cortex-m3のNVICがペリフェラル割り込みにはかせる下駄である。
		// 詳しくはcortex-m3のマニュアル参照
	exception_vector[UART0_IRQn+16] = (unsigned int)uart0_rx_handler;
	exception_vector[TIMER0_IRQn+16]= (unsigned int)timer0_handler;

		// 最初にすべての割り込みをディセーブルにする
	NVIC->ICER[0] = 0xFFFFFFFF;
	NVIC->ICER[1] = 0xFFFFFFFF;

		// UARTおよびTIMER0割り込みをイネーブル
	NVIC->ISER[0] = 1<<UART0_IRQn | 1<<TIMER0_IRQn;

		// CORTEX-M3コアの割り込みを許可する
	asm volatile(	"dmb \n"				// 命令同期バリアは、例外ベクトル変更後に必要
					"mov r0, #0 \n"
					"msr primask, r0 \n"	// すべての割り込みを禁止する
					:
					:
					: "r0" );

}
