/**********************************************************************
 * l7vs_module.h                                               May 2007 
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/
#ifndef L7VS_MODULE_H
#define L7VS_MODULE_H

#include <stdint.h>
#include <glib.h>
#include "l7vs.h"
#include "l7vs_service.h"
#include "logger_wrapper.h"

struct l7vs_conn;
struct l7vs_service;
struct l7vs_service_arg;
struct l7vs_dest;

struct l7vs_protomod{
    void* handle;
    char  modname[L7VS_MODNAME_LEN+1];
    int   refcnt;
    int   fast_schedule;

    //member functions
    int   ( *create )( void*, uint32_t );
    int   ( *compare )( uint32_t, uint32_t );
    int   ( *select_dest )( struct l7vs_service*, struct l7vs_conn*, char*, size_t*, struct l7vs_dest** );
    int   ( *analyze_cldata )( struct l7vs_service*, struct l7vs_conn*, char*, size_t* );
    int   ( *analyze_rsdata )( struct l7vs_service*, struct l7vs_conn*, char*, size_t* );
    int   ( *destroy )( uint32_t );
    void  ( *fini )();
    void* ( *create_sa )( struct l7vs_service_arg* );
    int   ( *service_arg )( struct l7vs_service_arg_multi*, uint32_t );
    int   ( *parse )( void*, int, char** );
    void  ( *destroy_sa )( void** );
    int   ( *initialize )( struct l7vs_service*, struct l7vs_conn*, char*, size_t, struct l7vs_dest** );
    int   ( *finalize )( struct l7vs_service*, struct l7vs_conn*, char*, size_t, struct l7vs_dest**, int );
    enum LOG_LEVEL_TAG ( *get_log_level )( const enum LOG_CATEGORY_TAG );
    void  ( *put_log_debug )( const enum LOG_CATEGORY_TAG, const unsigned int, char*, int, const char* );
    void  ( *put_log_info )( const enum LOG_CATEGORY_TAG, const unsigned int, char*, int, const char* );
    void  ( *put_log_warn )( const enum LOG_CATEGORY_TAG, const unsigned int, char*, int, const char* );
    void  ( *put_log_error )( const enum LOG_CATEGORY_TAG, const unsigned int, char*, int, const char* );
    void  ( *put_log_fatal )( const enum LOG_CATEGORY_TAG, const unsigned int, char*, int, const char* );
    void* ( *replication_pay_memory )( char*, unsigned int* );
};

extern char l7vs_module_path[];

//
// functions
//

extern struct l7vs_protomod* l7vs_protomod_get( char* );
extern void  l7vs_protomod_put( struct l7vs_protomod* );
extern struct l7vs_protomod* l7vs_protomod_lookup( char* );
extern int   l7vs_module_init( char* );
extern void  l7vs_module_fini();
extern void* l7vs_module_load( char*, char* );
extern void  l7vs_module_unload( void* );
extern void  l7vs_module_register( GList**, void* );
extern void  l7vs_module_remove( GList**, void* );
extern void* l7vs_module_lookup( GList*, void*, GCompareFunc );

#define PUT_LOG_FATAL(mod, cat, message_id, message, arg...) { \
        if (mod.get_log_level != NULL && mod.put_log_fatal != NULL && \
        LOG_LV_FATAL >= mod.get_log_level(cat)) { \
        char buf[BUF_LEN]; \
        snprintf(buf, BUF_LEN, message, ##arg); \
        mod.put_log_fatal(cat, message_id, __FILE__, __LINE__, buf); }}
        
#define PUT_LOG_ERROR(mod, cat, message_id, message, arg...) { \
        if (mod.get_log_level != NULL && mod.put_log_error != NULL && \
        LOG_LV_ERROR >= mod.get_log_level(cat)) { \
        char buf[BUF_LEN]; \
        snprintf(buf, BUF_LEN, message, ##arg); \
        mod.put_log_error(cat, message_id, __FILE__, __LINE__, buf); }}

#define PUT_LOG_WARN(mod, cat, message_id, message, arg...) { \
        if (mod.get_log_level != NULL && mod.put_log_warn != NULL && \
        LOG_LV_WARN >= mod.get_log_level(cat)) { \
        char buf[BUF_LEN]; \
        snprintf(buf, BUF_LEN, message, ##arg); \
        mod.put_log_warn(cat, message_id, __FILE__, __LINE__, buf); }}

#define PUT_LOG_INFO(mod, cat, message_id, message, arg...) { \
        if (mod.get_log_level != NULL && mod.put_log_info != NULL && \
        LOG_LV_INFO >= mod.get_log_level(cat)) { \
        char buf[BUF_LEN]; \
        snprintf(buf, BUF_LEN, message, ##arg); \
        mod.put_log_info(cat, message_id, __FILE__, __LINE__, buf); }}

#define PUT_LOG_DEBUG(mod, cat, message_id, message, arg...) { \
        if (mod.put_log_debug != NULL) { \
        char buf[BUF_LEN]; \
        snprintf(buf, BUF_LEN, message, ##arg); \
        mod.put_log_debug(cat, message_id, __FILE__, __LINE__, buf); }}

template <typename T> enum LOG_LEVEL_TAG IS_DEBUG( const T &obj, enum LOG_CATEGORY_TAG tag ){
    if( obj.get_log_level ){
        return obj.get_log_level(tag);
    }
    if( obj.put_log_debug ){
        obj.put_log_debug( tag, 0 , __FILE__, __LINE__, "oops pointer is NULL!!" );
    }
    return LOG_LV_DEBUG;
}

/*!
 * Extract struct l7vs_protomod to strings for debug log.
 * @param  string
 * @return l7vs_protomod struct
 */
inline void l7vs_protomod_c_str(char* buf, struct l7vs_protomod* proto) {
    if (proto == NULL) {
        snprintf(buf, DEBUG_STR_LEN, "NULL");
    }
    else {
        snprintf(buf, DEBUG_STR_LEN, "handle: %p, modname: %s, refcnt: %d, create: %p, compare: %p, "
            "select_dest: %p, analyze_cldata: %p, analyze_rsdata: %p, destroy: %p, fini: %p, create_sa: %p, "
            "service_arg: %p, parse: %p, destroy_sa: %p, initialize: %p, finalize: %p, get_log_level: %p, "
            "put_log_debug: %p, put_log_info: %p, put_log_warn: %p, put_log_error: %p, put_log_fatal: %p",
            proto->handle, proto->modname, proto->refcnt, proto->create, proto->compare, proto->select_dest,
            proto->analyze_cldata, proto->analyze_rsdata, proto->destroy, proto->fini, proto->create_sa,
            proto->service_arg, proto->parse, proto->destroy_sa, proto->initialize, proto->finalize,
            proto->get_log_level, proto->put_log_debug, proto->put_log_info, proto->put_log_warn,
            proto->put_log_error, proto->put_log_fatal);
    }
}

#endif //L7VS_MODULE_H
