/*
 * @file  l7vsd.c
 * @brief l7vsd main module.
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2008  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#ifndef _GNU_SOURCE
#define _GNU_SOURCE     /* for getopt_long() */
#endif /* _GNU_SOURCE */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <getopt.h>
#include <glib.h>
#include <errno.h>
#include "logger_wrapper.h"
#include "parameter_wrapper.h"
#include "l7vs.h"
#include "l7vs_module.h"
#include "l7vs_config.h"
#include "l7vs_iomuxlist.h"
#include "l7vs_config.h"
#include "l7vs_lsock.h"

static void sig_exit_handler(int sig);
static int set_sighandler(int sig, void (*handler)(int));
static int set_sighandlers(void);
static void usage(FILE *fp, char *ident);

static int exit_requested = 0;
static int received_sig = 0;

l7vs::Logger l7vs::Logger::instance;
l7vs::Parameter l7vs::Parameter::instance;

/*!
 * exit signal handler
 *
 * @param[in]   sig signal
 * @return      void
 */
static void
sig_exit_handler(int sig)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,1,
            "in_function: static void sig_exit_handler(int sig) "
            "sig=%d",
            sig);
    }
    /*------ DEBUG LOG END ------*/

    received_sig = sig;
    exit_requested++;

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,2,
            "out_function: static void sig_exit_handler(int sig) "
            "return_value: void");
    }
    /*------ DEBUG LOG END ------*/
}

/*!
 * setup signal handler
 *
 * @param[in] sig      signal
 * @param[in] handler  signal handler
 * @retval     0       succeed
 * @retval    -1       failed
 */
static int
set_sighandler(int sig, void (*handler)(int))
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,3,
            "in_function: static int set_sighandler(int sig, void (*handler)(int)) "
            "sig=%d: "
            "handler=%p",
            sig, handler);
    }
    /*------ DEBUG LOG END ------*/

    struct sigaction act;
    int ret;

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,4,
            "sigaction : sig=%d: act=NULL",
            sig);
    }
    /*------ DEBUG LOG END ------*/
    ret = sigaction(sig, NULL, &act);
    if (ret < 0) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SIGNAL,1, "sigaction on signal %d failed", sig);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,5,
                "out_function: static int set_sighandler(int sig, void (*handler)(int)) "
                "return_value: %d",
                ret);
        }
        /*------ DEBUG LOG END ------*/
        return ret;
    }
    act.sa_flags &= ~SA_RESETHAND;
    act.sa_handler = handler;

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,6,
            "sigaction : sig=%d: act.sa_flags=%d, act.sa_handler=%p",
            sig, act.sa_flags, act.sa_handler);
    }
    /*------ DEBUG LOG END ------*/
    ret = sigaction(sig, &act, NULL);
    if (ret < 0) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SIGNAL,2, "sigaction on signal %d failed", sig);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,7,
                "out_function: static int set_sighandler(int sig, void (*handler)(int)) "
                "return_value: %d",
                ret);
        }
        /*------ DEBUG LOG END ------*/
        return ret;
    }

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,8,
            "out_function: static int set_sighandler(int sig, void (*handler)(int)) "
            "return_value: 0");
    }
    /*------ DEBUG LOG END ------*/
    return 0;
}

/*!
 * setup all signal handlers
 *
 * @param[in]   void
 * @retval      0   succeed
 * @retval      -1  failed
 */
static int
set_sighandlers(void)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,9,
            "in_function: static int set_sighandlers(void) ");
    }
    /*------ DEBUG LOG END ------*/
    int ret;

#define SET_SIGHANDLER(sig, handler)                                            \
    do {                                                                        \
        ret = set_sighandler((sig), (handler));                                 \
        if (ret < 0) {                                                          \
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {  \
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,10,                  \
                    "out_function: static int set_sighandlers(void) "           \
                    "return_value: %d",                                         \
                    ret);                                                       \
            }                                                                   \
            return ret;                                                         \
        }                                                                       \
    } while (0)

    SET_SIGHANDLER(SIGHUP,  sig_exit_handler);
    SET_SIGHANDLER(SIGINT,  sig_exit_handler);
    SET_SIGHANDLER(SIGQUIT, sig_exit_handler);
    SET_SIGHANDLER(SIGTERM, sig_exit_handler);
    SET_SIGHANDLER(SIGUSR1, SIG_IGN);
    SET_SIGHANDLER(SIGUSR2, SIG_IGN);
    SET_SIGHANDLER(SIGALRM, SIG_IGN);
    SET_SIGHANDLER(SIGCHLD, SIG_IGN);

#undef SET_SIGHANDLER

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,11,
            "out_function: static int set_sighandlers(void) "
            "return_value: 0");
    }
    /*------ DEBUG LOG END ------*/
    return 0;
}

/*!
 * show usage
 *
 * @param[in]   fp      filepointer
 * @param[in]   ident   program identifier
 * @return      void
 */
static void
usage(FILE *fp, char *ident)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
        if (!fp) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,12,
                "in_function: static void usage(FILE *fp, char *ident) "
                "fp=NULL, ident=%s",
                ident);
        }
        else {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,13,
                "in_function: static void usage(FILE *fp, char *ident) "
                "fp->_fileno=%d, ident=%s",
                fp->_fileno, ident);
        }
    }
    /*------ DEBUG LOG END ------*/

    if (!fp) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,1, "fp is NULL.");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,14,
                "out_function: static void usage(FILE *fp, char *ident) "
                "return_value: void");
        }
        /*------ DEBUG LOG END ------*/
        return;
    }
    if (!ident) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,2, "ident is NULL.");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,15,
                "out_function: static void usage(FILE *fp, char *ident) "
                "return_value: void");
        }
        /*------ DEBUG LOG END ------*/
        return;
    }
    fprintf(fp,
        "Usage: %s [-d] [-h]\n"
        "   -d    --debug        run in debug mode (in foreground)\n"
        "   -h    --help         print this help messages and exit\n"
        "   -b    --blocking     blocking mode enable( default non-blocking )\n",
        ident);

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,16,
            "out_function: static void usage(FILE *fp, char *ident) "
            "return_value: void");
    }
    /*------ DEBUG LOG END ------*/
}

/*!
 * entry point
 *
 * @param[in]   argc    number of arguments
 * @param[in]   argv    arguments
 * @return      int     exit status
 */
int
//l7vsd_main(int argc, char *argv[])
main(int argc, char *argv[])
{
    //parameter set function register
    parameter_register_function_pointer( PARAM_COMP_SNMPAGENT, l7vs_snmpbridge_reload_config );
    parameter_register_function_pointer( PARAM_COMP_REPLICATION, l7vs_replication_reset );

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
        char argv_str[DEBUG_STR_LEN] = {0};
        size_t pos = 0;
        for (int i = 0; i < argc; ++i) {
            if (DEBUG_STR_LEN <= pos + strlen(argv[i]) + 1) {
                break;
            }
            strcpy(argv_str + pos, argv[i]);
            pos += strlen(argv[i]);
            *(argv_str + pos) = ' ';
            ++pos;
        }
        *(argv_str + pos) = '\0';
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,17,
            "in_function: int main(int argc, char *argv[]) "
            "argc=%d, argv=%s",
            argc, argv_str);
    }
    /*------ DEBUG LOG END ------*/

    int c;
    int ret;
    int debug = 0;
    int blocking = 0;

    static struct option options[] = {
        {"debug",       no_argument,    NULL,   'd'},
        {"help",        no_argument,    NULL,   'h'},
        {"blocking",    no_argument,    NULL,   'b'},
        {NULL,          0,              NULL,   0  }
    };

    while ((c = getopt_long(argc, argv, "dhb", options, NULL)) != -1) {
        switch (c) {
        case 'd':
            debug = 1;
            break;
        case 'b':
            blocking = -1;
            break;
        case '?':
        default:
            fprintf(stderr, "%s: unknown option: %s\n",
                argv[0], argv[optind - 1]);
            usage(stderr, argv[0]);
            fflush(stdout);
            fsync(1);
            _exit(1);
            break;
        case 'h':
            usage(stdout, argv[0]);
            fflush(stdout);
            fsync(1);
            _exit(0);
            break;
        }
    }

    if (!debug) {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,18,
                "deamon : nochdir=0: noclose=0");
        }
        /*------ DEBUG LOG END ------*/
        ret = daemon(0, 0);
        if (ret < 0) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,3, "daemon() failed: %s", strerror(errno));
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,19,
                    "out_function: int main(int argc, char *argv[]) "
                    "return_value: 1");
            }
            /*------ DEBUG LOG END ------*/
            fflush(stdout);
            fsync(1);
            _exit(1);
        }
    }

    LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_START_STOP,1, "Starting.");

    set_sighandlers();

    sprintf(l7vs_module_path, L7VS_MODULE_PATH);
    ret = l7vs_iomux_init();
    if (ret < 0) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,5, "iomux_init failed. Exitting.");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,20,
                "out_function: int main(int argc, char *argv[]) "
                "return_value: 1");
        }
        /*------ DEBUG LOG END ------*/
        fflush(stdout);
        fsync(1);
        _exit(1);
    }

    //snmpbridge initialize
    LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_PROGRAM,1, "snmpbridge initialize start.");
    ret = l7vs_snmpbridge_initialize();
    if( ret < 0 ){
        //debug log output
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,4, "snmpbridge initialize failure.");
    }
    LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_PROGRAM,2, "snmpbridge initialize success.");

    ret = l7vs_config_init();
    if (ret < 0) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,6, "config_init failed. Exiting.");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,21,
                "out_function: int main(int argc, char *argv[]) "
                "return_value: 1");
        }
        /*------ DEBUG LOG END ------*/
        fflush(stdout);
        fsync(1);
        _exit(1);
    }

    l7vs_module_init(NULL);
    l7vs_lsock_init();
    l7vs_replication_init();
    l7vs_conn_init();
    while (!exit_requested) {
        ret = l7vs_iomux_poll(NULL, blocking);
        if (ret < 0) {
            if (errno == EINTR) {
                /* stop_requested will be checked... */
                continue;
            }
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,1, "poll: %s", strerror(errno));
            break;
        }
    }

    if (received_sig) {
        LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_SYSTEM_SIGNAL,1, "signal %d received.", received_sig);
    }

    l7vs_replication_fini();
    l7vs_lsock_fini();
    l7vs_module_fini();
    l7vs_config_fini();
    l7vs_iomux_fini();

    LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_START_STOP,2, "Exiting.");

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_PROGRAM)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,22,
            "out_function: int main(int argc, char *argv[]) "
            "return_value: 0");
    }
    /*------ DEBUG LOG END ------*/
    fflush(stdout);
    fsync(1);
    _exit(0);
    return 0;  /* this instruction never called ;-) */
}
