/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;

namespace Yubeshi.Nmea
{
    /// <summary>
    ///     Latitude and Longitude, with Time of Position Fix and Status
    /// </summary>
    public class GpGll : Packet
    {
        #region type definitions
        public enum Mode
        { 
            Unknown = -1,
            None,
            Autonomous, 
            Differential,
            DeadReckoning,
        }
        #endregion

        #region fields
        private static Dictionary<string, Mode> modes;
        private static readonly byte[] header;
        private const int elementNum = -1; // 6 or 7


        #endregion

        #region constructors
        
        static GpGll()
        {
            header = Encoding.ASCII.GetBytes("$GPGLL,");

            modes = new Dictionary<string, Mode>();
            modes["N"] = Mode.None;
            modes["A"] = Mode.Autonomous;
            modes["D"] = Mode.Differential;
            modes["E"] = Mode.DeadReckoning;
        }
        
        public GpGll()
        {
        }

        public GpGll(byte[] sentence)
            : this(sentence, GetElements(sentence, elementNum))
        {
        }

        private GpGll(byte[] sentence, Elements elements)
        {
            string[] v = elements.Values;
            if (v.Length < 6 || v.Length > 7)
            {
                throw new ArgumentException();
            }
            Position = new GeodeticCoordinate(
                                    Degree.FromNmeaFormat(v[0], v[1]),
                                    Degree.FromNmeaFormat(v[2], v[3]));
            CurrentTime = ParseTime(v[4]);
            DataStatus = ParseStatus(v[5]);
            PositioningMode = (v.Length == 7) ? 
                                        GetPositioningMode(v[6]): Mode.Unknown;
            CheckSum = elements.CheckSum;
        }

        #endregion

        #region properties

        public GeodeticCoordinate Position
        {
            get;
            private set;
        }

        public TimeSpan CurrentTime
        {
            get;
            private set;
        }

        public Status DataStatus
        {
            get;
            private set;
        }

        public Mode PositioningMode
        {
            get;
            private set;
        }

        #endregion

        #region public method

        public static bool TryParse(byte[] sentence, out UnknownPacket packet)
        {
            return TryParse(sentence, out packet, header, elementNum, Build);
        }

        #endregion

        #region private method

        private static Packet Build(byte[] sentence, Elements elements)
        {
            return new GpGll(sentence, elements);
        }

        private Mode GetPositioningMode(string mode)
        {
            Mode m;
            if (modes.TryGetValue(mode, out m))
            {
                return m;
            }
            return Mode.Unknown;
        }
        #endregion
    }
}