package jp.nekoteki.android.demowatcher;

import android.app.ActivityManager;
import android.app.ActivityManager.RunningAppProcessInfo;
import android.app.AlarmManager;
import android.app.IntentService;
import android.app.KeyguardManager;
import android.app.PendingIntent;
import android.app.KeyguardManager.KeyguardLock;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.net.Uri;
import android.os.Handler;
import android.os.PowerManager;
import android.preference.PreferenceManager;
import android.provider.BaseColumns;
import android.util.Log;
import android.view.Window;
import android.view.WindowManager;
import android.webkit.MimeTypeMap;
import android.widget.Toast;

public class WatchService extends IntentService {
	final static String NAME = "DemoWatchService";
	final public static String DEFAULT_WATCH = "com.mxtech.videoplayer.pro";
	final public static String DEFAULT_OPEN = "file:///sdcard/test/test.mp4";
	final public static int DEFAULT_INTERVAL = 30;
	public Handler handler;
	
	public WatchService() {
		super(NAME);
		this.handler = new Handler();
	}

	public WatchService(String name) {
		super(name);
		this.handler = new Handler();
	}

	public PendingIntent getPendingIntent(Context context) {
		return this.getPendingIntent(context, false);
	}
	
	public PendingIntent getPendingIntent(Context context, boolean create) {
		int flag;
		if (create) {
			flag = PendingIntent.FLAG_UPDATE_CURRENT;
		} else {
			flag = PendingIntent.FLAG_UPDATE_CURRENT | PendingIntent.FLAG_NO_CREATE;
		}
		Intent intent = new Intent(context, WatchService.class);
		intent.putExtra("open", this.getOpenTarget(context));
		intent.putExtra("watch", this.getWatchTarget(context));
		intent.putExtra("method", this.getLaunchMethod(context));
		return PendingIntent.getService(context, 0, intent, flag);
	}
	
	public SharedPreferences getPrefs(Context c) {
		return PreferenceManager.getDefaultSharedPreferences(c);
	}
	
	public String getWatchTarget(Context c) {
		return getPrefs(c).getString("watch", DEFAULT_WATCH);
	}
	
	public String getOpenTarget(Context c) {
		return getPrefs(c).getString("open", DEFAULT_OPEN);
	}
	
	public String getLaunchMethod(Context c) {
		return getPrefs(c).getString("method", "package");
	}
	
	public int getInterval(Context c) {
		return getPrefs(c).getInt("interval", DEFAULT_INTERVAL);
	}
	

	public boolean cancelSchedule(Context context) {
		Editor peditor = getPrefs(context).edit();
		peditor.putBoolean("enabled", false);
		peditor.commit();

		PendingIntent intent = this.getPendingIntent(context);
		if (intent == null) return false;
		Log.d(this.getClass().getSimpleName(), "Disable watch schedule");
		AlarmManager manager = (AlarmManager) context.getSystemService(ALARM_SERVICE);
		manager.cancel(intent);
		intent.cancel();
		
		killWatchTarget(context);
		return true;
	}
	
	public boolean schedule(Context context) {
		Editor peditor = getPrefs(context).edit();
		peditor.putBoolean("enabled", true);
		peditor.commit();
		
		if (this.getPendingIntent(context) != null)
			return false;
		AlarmManager manager = (AlarmManager) context.getSystemService(ALARM_SERVICE);
		manager.setInexactRepeating(AlarmManager.RTC_WAKEUP,
				System.currentTimeMillis(), this.getInterval(context) * 1000,
				this.getPendingIntent(context, true));
		return true;
	}

	@Override
	protected void onHandleIntent(Intent intent) {
		Log.d(NAME, "Run watch handle!");

		this.forceWake();
		// TODO: unlock support.

		if (checkTopActivity(intent.getStringExtra("watch")))
			return;
		
		try {
			if (intent.getStringExtra("method").equals("url")) {
				launchWithURL(intent);
			} else {
				launchWithPackage(intent);
			}
		} catch (Exception e) {
			Log.d(NAME, "Faild to Start: "+e.getMessage());
			handler.post(new Runnable() {
				public String message = "";
				public Runnable setMessage(String message) {
					this.message = message;
					return this;
				}
				@Override
				public void run() {
					Toast.makeText(getApplicationContext(),
							"[DemoWatcher] Failed to start target!\n"+this.message,
							Toast.LENGTH_LONG).show();
				}
			}.setMessage(e.getMessage()));
		}
	}
	
	protected void launchWithPackage(Intent intent) {
		String watch = intent.getStringExtra("watch");
		Intent openintent = getPackageManager().getLaunchIntentForPackage(watch);
		openintent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_BROUGHT_TO_FRONT | Intent.FLAG_ACTIVITY_RESET_TASK_IF_NEEDED);
		
		Log.d(NAME, "Sending package "+watch);

		startActivity(openintent);
	}
	
	protected void launchWithURL(Intent intent) {
		String url = intent.getStringExtra("open");
		Intent openintent = new Intent(Intent.ACTION_VIEW);
		openintent.setData(Uri.parse(url));
		openintent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_BROUGHT_TO_FRONT | Intent.FLAG_ACTIVITY_RESET_TASK_IF_NEEDED);

		Log.d(NAME, "Sending view intent with "+url);
		startActivity(openintent);
	}
	
	public void killWatchTarget(Context context) {
		ActivityManager am = (ActivityManager)context.getSystemService(ACTIVITY_SERVICE);
		String watch = getWatchTarget(context); 
		Log.d(NAME, "Killing "+watch);
		am.killBackgroundProcesses(watch);
		for (RunningAppProcessInfo pi : am.getRunningAppProcesses()) {
	    	if (!watch.equals(pi.processName))
	    		continue;
			Log.d(NAME, "Kill pid="+String.valueOf(pi.pid));
	    	android.os.Process.killProcess(pi.pid);
	    }
	}
	
	public boolean checkTopActivity(String watch) {
		ActivityManager am = (ActivityManager)this.getSystemService(ACTIVITY_SERVICE);
		String front = am.getRunningTasks(1).get(0).topActivity.getPackageName();
		Log.d(NAME, "Current top task: "+front);
		if (front.equals(watch)) {
			Log.d(NAME, watch+" is running on top.");
			return true;
		}

		Log.i(NAME, "Watch target '"+watch+"' is not top activitiy.");
	    return false;
	}
	
	public void forceWake() {
	    // force wake
	    PowerManager pm = (PowerManager) getSystemService(POWER_SERVICE);
	    if (!pm.isScreenOn()) {
	    	Log.i(NAME, "Screen is off, forcing awake.");
	    	PowerManager.WakeLock wakelock = pm.newWakeLock(PowerManager.FULL_WAKE_LOCK
	    			| PowerManager.ACQUIRE_CAUSES_WAKEUP | PowerManager.ON_AFTER_RELEASE, NAME);
	    	wakelock.acquire();
	    	wakelock.release();
	    }
	}
}
