import React from 'react';
import PropTypes from 'prop-types';
import { Button } from 'reactstrap';
import StepZilla from 'react-stepzilla';

class ThemeWizard extends React.Component {
  static propTypes = {
    draft_key: PropTypes.string.isRequired,
    url: PropTypes.string.isRequired,
  }

  constructor (props) {
    super(props);
    this.state = {
      //draft: new DraftStore(props.draft),
      theme: {
        name: "",
        author: "",
        url: "",
        short_desc: "",
        allow_orig_dl: true,
        rights_confirmed: false,
        draft_key: props.draft_key,
        license_cc: null,
        license_pcl: false,
        license_yukari: false,
        license_other: '',
        license_other_url: '',
      },
    };
  }

  updateTheme(theme_props) {
    this.setState({theme: Object.assign(this.state.theme, theme_props)});
  }

  save() {
    let data = {};
    for (let key in this.state.theme) {
      if (key == "rights_confirmed")
        continue;
      data[`theme[${key}]`] = this.state.theme[key];
    }
    
    jQuery.ajax({
      url: this.props.url,
      method: this.props.method ? this.props.method : 'POST',
      data: data,
      success: (data, status, xhr) => {
        if (Turbolinks && Turbolinks.visit) {
          Turbolinks.visit(data.path);
        } else {
          location.href = data.path;
        }
      },
      error: () => {
        alert(I18n.t('save_theme_error'));
      },
    });
  }

  render() {
    const theme = this.state.theme;
    let steps = [];
    if (this.props.login_buttons) {
      steps.push({
        name: I18n.t('theme_wizard_steps.login'),
        component: <TW_Login buttons={this.props.login_buttons} />
      });
    }
    steps = steps.concat([
      {
        name: I18n.t('theme_wizard_steps.confirm_rights'),
        component: <TW_ConfirmRights theme={theme} updateTheme={this.updateTheme.bind(this)} />
      },
      {
        name: I18n.t('theme_wizard_steps.theme_info'),
       component: <TW_ThemeInfo theme={theme} updateTheme={this.updateTheme.bind(this)} />
      },
      {
        name: I18n.t('theme_wizard_steps.choose_license'),
        component: <TW_License theme={theme} updateTheme={this.updateTheme.bind(this)} />
      },
      {
        name: I18n.t('theme_wizard_steps.publish'),
        component: <TW_Publish theme={theme} save={this.save.bind(this)} />
      },
    ]);
    return(
      <div className="theme-wizard">
        <StepZilla steps={steps}
                   backButtonCls="btn btn-lg btn-secondary pull-left"
                   backButtonText={I18n.t('prev', {default: 'Previous'})}
                   nextButtonText={I18n.t('next', {default: 'Next'})}
                   />
      </div>
    );
  }
}


class TW_Login extends React.Component {
  render() {
    return(
      <div>
        <div className="card">
          <div className="card-body">
            <h3>ログインの確認</h3>
            <p>現在ログインしていません。このまま進めた場合、テーマを匿名で公開できますが、公開されたテーマの修正・削除などは一切できません。
            </p>
            <p>{I18n.t('login_desc')}</p>
            <p>{I18n.t('login_privacy_notice')}</p>
            <div className="text-center" dangerouslySetInnerHTML={{__html: this.props.buttons}}>
            </div>
          </div>
        </div>
      </div>
    );
  }
}
class TW_ConfirmRights extends React.Component {
  constructor (props) {
    super(props);
    this.state = {
      valid: true,
    };
  }

  isValidated() {
    if (this.props.theme.rights_confirmed) {
      this.setState({valid: true});
      return true;
    }
    this.setState({valid: false});
    return false;
  }

  render() {
    return(
      <div>
        <div className="card">
          <div className="card-body">
            <h3>最初に権利について確認してください</h3>
            <p>
              公開テーマとして登録すると、データは誰でもダウンロードできるようになります。
              登録した全ての画像に関して、あなたが公開する権利を間違いなく持っている事を再度確認してください。
            </p>
            <ul>
              <li>画像は自分でつくったものですか？</li>
              <li>許可されていない他人の著作物を無断で加工/利用していませんか？</li>
              <li>キャラクターの場合は、そのキャラクターの利用が許可されているか確認しましょう。<br />
                <small className="text-muted">（例えば「初音ミク」およびクリプトン社のキャラクターの場合は<a target="license" href="http://piapro.jp/license/pcl/summary">ピアプロ・キャラクター・ライセンス(PCL)</a>に沿っていなくてはなりません）</small>
              </li>
            </ul>
            <div className="form-group">
              <div className="form-check">
                <label className={`form-check-label ${this.state.valid ? '' : 'text-danger'}`}>
                  <input className="form-check-input" type="checkbox" 
                         checked={this.props.theme.rights_confirmed}
                         onChange={(e) => this.props.updateTheme({rights_confirmed: e.target.checked})} />
                    &nbsp;確認しました。全ての画像の著作権を自分が有しているか、著作権者から許諾を得ています。
                </label>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  }
}

class TW_ThemeInfo extends React.Component {
  constructor (props) {
    super(props);
    this.state = {
      valid_name: true,
      valid_author: true,
      valid_short_desc: true,
      valid_url: true,
    };
  }

  isValidated() {
    const theme = this.props.theme;
    let ret = true;
    const name = theme.name.replace(/^\s+|\s+$/g, '')
    if (name.length > 0 && name.length < 60) {
      this.setState({valid_name: true});
    } else {
      this.setState({valid_name: false});
      ret = false;
    }
    if (theme.url.length == 0 || theme.url.match(/^https?:\/\/[0-9a-zA-Z-]+(\.[0-9a-zA-Z-]+)+/)) {
      this.setState({valid_url: true});
    } else {
      this.setState({valid_url: false});
      ret = false;
    }
    const author = theme.author.replace(/^\s+|\s+$/g, '')
    if (author.length > 0 && author.length < 255) {
      this.setState({valid_author: true});
    } else {
      this.setState({valid_author: false});
      ret = false;
    }
    const short_desc = theme.short_desc.replace(/^\s+|\s+$/g, '')
    if (short_desc.length < 140) {
      this.setState({valid_short_desc: true});
    } else {
      this.setState({valid_short_desc: false});
      ret = false;
    }
    return ret;
  }

  render() {
    const inputs = [];
    ['name', 'author', 'url', 'short_desc'].forEach((attr) => {
      let required = null;
      if (attr == 'name' || attr == 'author')
        required = <span><abbr title="required">*</abbr>&nbsp;</span>
      inputs.push(
        <div className="form-group row">
          <label className="col-md-2 form-control-label string" htmlFor={`theme_${attr}`}>
            {required}
            {I18n.t(`activerecord.attributes.theme.${attr}`)}</label>
          <div className="col-md-10">
            <input className={`form-control string ${this.state['valid_'+attr] ? '' : 'is-invalid'}`} key={`input-group-${attr}`}
                   placeholder={I18n.t(`simple_formn.placeholders.theme.${attr}`, {default: null})}
                   name={`theme[${attr}]`} id={`theme_${attr}`} type="text" 
                   value={this.props.theme[attr]}
                   onChange={(e) => this.props.updateTheme({[attr]: e.target.value})}
              />
          </div>
        </div>
      );
    });
    
    return(
      <div>
        <div className="card">
          <div className="card-body">
            <h3>テーマ情報を入力して下さい</h3>
            {inputs}
          </div>
        </div>
      </div>
    );
  }
}

class TW_License extends React.Component {
  constructor (props) {
    super(props);
    this.state = {
      valid: true,
      valid_cc: true,
      valid_other: true,
      valid_other_url: true,
      other_checked: !!props.theme.license_other,
    };
  }

  isValidated() {
    const theme = this.props.theme;
    let ret = true;
    if (theme.license_cc && theme.license_cc == '_tbd') {
      this.setState({valid_cc: false});
      ret = false;
    } else {
      this.setState({valid_cc: true});
    }

    if (this.state.other_checked) {
      if (theme.license_other) {
        this.setState({valid_other: true});
      } else {
        this.setState({valid_other: false});
        ret = false;
      }
      if (theme.license_other_url && theme.license_other_url.match(/^https?:\/\/[a-zA-Z0-9-]+(\.[0-9a-zA-Z-]+)+/)) {
        this.setState({valid_other_url: true});
      } else {
        this.setState({valid_other_url: false});
        ret = false;
      }
    }
    return ret;
  }

  render() {
    const cc_conds = {
      'by': '再配布時や派生作品の公開時には、作者名を表示する必要があります (BY)',
      'by-sa': '作者名を表示し、派生・改変作品を同一条件で公開する必要があります (BY-SA)',
      'by-nd': '作者名を表示した再配布は可能ですが、派生・改変作品に利用しないでください (BY-ND)',
      'by-nc': '作者名を表示し、目的は非商用利用に限ります (BY-NC)',
      'by-nc-sa': '作者名を表示し、非商用利用かつ派生作品を同一条件で公開する必要があります (BY-NC-SA)',
      'by-nc-nd': '作者名を表示した非商用目的でのみ利用できますが、一切の改変を許可しません (BY-NC-ND)',
    };

    const char_conds = {
      'pcl': <span><a target="license" href="http://piapro.jp/license/pcl/summary">ピアプロ・キャラクター・ライセンス</a>に基づく</span>,
      'yukari': <span><a target="license" href="https://www.ah-soft.com/vocaloid/yukari/charactor_guide.html">結月ゆかりキャラクター使用ガイドライン</a>に基づく</span>,
    };

    const char_choosers = [];
    for (let key in char_conds) {
      char_choosers.push(
        <div className="form-group" key={`char-check-${key}`}>
          <div className="form-check">
            <label className="form-check-label">
              <input className="form-check-input" type="checkbox" 
                     checked={this.props.theme[`license_${key}`]}
                     onChange={(e) => this.props.updateTheme({[`license_${key}`]: e.target.checked})} />
                &nbsp;{char_conds[key]}
            </label>
          </div>
        </div>
      );
    }

    const cc_choosers = [];
    for (let key in cc_conds) {
      cc_choosers.push(
        <div className="form-group" key={`cc-choice-${key}`}>
          <div className="form-check">
            <label className={`form-check-label ${this.state.valid_cc ? '' : 'text-danger'}`}>
              <input className="form-check-input" type="radio" name="cc" value={key}
                     checked={this.props.theme.license_cc == key}
                     onChange={(e) => this.props.updateTheme({license_cc: key})}
                     />
              &nbsp;{cc_conds[key]}
            </label>
          </div>
        </div>
      );
    }

    let cc_chooser_block = null;
    if (this.props.theme.license_cc) {
      cc_chooser_block = <div className="cc-choosers ml-4">
        {cc_choosers}
      </div>;
    }

    let other_block = null;
    if (this.state.other_checked) {
      other_block = <div className="other-block">
        <div className="form-group row">
          <label className="col-2 pl-4 form-control-label string" htmlFor="license_oather">
            ライセンス名</label>
          <div className="col-10">
            <input className={`form-control string ${this.state.valid_other ? '' : 'is-invalid'}`}
               type="text"  id="license_other"
               placeholder="名称を入力して下さい" value={this.props.theme.license_other}
               onChange={(e) => this.props.updateTheme({license_other: e.target.value})}
            />
          </div>
        </div>
        <div className="form-group row">
          <label className="col-2 pl-4 form-control-label string" htmlFor="license_other_url">
            ライセンス本文URL</label>
          <div className="col-10">
          <input className={`form-control string ${this.state.valid_other_url ? '' : 'is-invalid'}`}
                   type="text" id="license_other_url"
                   placeholder="ライセンスの紹介・本文・など一次情報のURLを入力して下さい"
                   value={this.props.theme.license_other_url}
                   onChange={(e) => this.props.updateTheme({license_other_url: e.target.value})}
              />
          </div>
        </div>
      </div>;
    }

    return(
      <div>
        <div className="card">
          <div className="card-body">
            <h3>ライセンスの選択</h3>
            <p>公開したテーマを利用者がどのように扱えるか選択して下さい。</p>
            
            <div className="form-group">
              <div className="form-check">
                <label className={`form-check-label ${this.state.valid ? '' : 'text-danger'}`}>
                  <input className="form-check-input" type="checkbox" 
                         checked={this.props.theme.allow_orig_dl}
                         onChange={(e) => this.props.updateTheme({allow_orig_dl: e.target.checked})} />
                    &nbsp;テーマの元画像ファイルをダウンロード可能にする
                </label>
              </div>
            </div>

            <div className="form-group">
              <div className="form-check">
                <label className="form-check-label">
                  <input className="form-check-input" type="checkbox" 
                         checked={this.props.theme.license_cc}
                         onChange={(e) => this.props.updateTheme({license_cc: (e.target.checked ? '_tbd' : '')})}
                         />
                  &nbsp;テーマを<a target="license" href="http://www.creativecommons.org/">クリエイティブコモンズ</a>の元で配布する
                </label>
              </div>
            </div>
            {cc_chooser_block}

            {char_choosers}

            <div className="form-group">
              <div className="form-check">
                <label className="form-check-label">
                  <input className="form-check-input" type="checkbox" 
                         checked={this.state.other_checked}
                         onChange={(e) => {this.setState({other_checked: e.target.checked}); if (!e.target.checked) { this.props.updateTheme({license_other: '', license_other_url: ''}); };}} />
                    &nbsp;その他のライセンスに基づく
                </label>
              </div>
            </div>
            {other_block}
          </div>
        </div>
      </div>
    );
  }
}

class TW_Publish extends React.Component {
  constructor (props) {
    super(props);
    this.state = {
      wip: false,
    };
  }

  save() {
    this.setState({wip: true});
    this.props.save();
  }

  render() {
    const theme = this.props.theme;
    return (
      <div>
        <div className="card">
          <div className="card-body">
            <h4>最後のステップです</h4>
            <p>公開の準備ができました。以下の情報が間違いない事を確認して、公開するボタンを押して下さい。</p>
            <ul>
              <li><b>タイトル:</b> {theme.name}</li>
              <li><b>作者名:</b> {theme.author}</li>
              {theme.url ? <li><b>リンク先:</b> {theme.url}</li> : null}
              {theme.shoft_desc ? <li><b>短い説明:</b> {theme.short_desc}</li> : null}
              <li><b>元画像のダウンロード:</b> {theme.allow_orig_dl ? '許可する': '許可しない'}</li>
              {theme.license_cc ? <li>クリエイティブコモンズ ({theme.license_cc}) に基づきます</li> : null}
              {theme.license_pcl ? <li>ピアプロ・キャラクター・ライセンスに基づきます</li> : null}
              {theme.license_yukari ? <li>結月ゆかりキャラクター使用ガイドラインに基づきます</li> : null}
              {theme.license_other ? <li>{theme.license_other}に基づきます</li> : null}
            </ul>
          </div>
          <div className="text-center mb-4">
            <button className="btn btn-lg btn-primary"
                    disabled={this.state.wip ? 'disabled' : null}
                    onClick={(e) => this.save()}>
              {this.state.wip ? '登録中です...' : 'この内容で登録する!'}
            </button>
          </div>
        </div>
      </div>
    );
  }
}
  
export default ThemeWizard;
