class NaviVoice < ActiveRecord::Base
  default_scope -> { order('rating desc, id desc') }

  has_attached_file :archive, url: "/bin/:class/:attachment/:id_partition/:style.:extension",
    styles: {preview: ""}, :processors => [:voice_preview]
  attr_accessor :raw_delete_key

  validates_attachment :archive, presence: true, size: {less_than: 5120.kilobytes}
  validates :archive, voice_archive: true
  validates :title, presence: true, length: {maximum: 80}
  validates :description, length: {maximum: 280}
  validates :unit, inclusion: {in: %w(metric imperial)}
  validates :archive_fingerprint, uniqueness: true
  validates :lang, length: {is: 2, allow_nil: true}

  before_save :hash_delete_key
  after_validation :dup_errormsg

  def delete_key_valid?(key)
    delete_key? or return false
    salt, hash = delete_key.split('/', 2)
    hash == Digest::SHA1.hexdigest("#{salt}#{key}")
  end

  private
  def hash_delete_key
    @raw_delete_key.blank? and return true
    salt = SecureRandom.hex
    self[:delete_key] = "#{salt}/" + Digest::SHA1.hexdigest("#{salt}#{@raw_delete_key}")
    true
  end

  def dup_errormsg
    self.errors[:archive_fingerprint].blank? and return true
    self.errors[:archive_fingerprint].each do |e|
      self.errors[:archive] << e
    end
  end
end
