#include <pebble.h>

#define ANIME_TIME 5000
#define CHAR_IMAGES 3
#define CHAR_FRAME_MS 250
static Window *s_main_window;
static TextLayer *s_time_layer;
static TextLayer *s_date_layer;
static TextLayer *s_wday_layer;
static Layer *s_battery_layer;

static GFont s_time_font;
static GFont s_date_font;
static BitmapLayer *s_noconn_layer;
static GBitmap *s_noconn_bitmap;
static BitmapLayer *s_logo_layer;
static GBitmap *s_logo_bitmap;
static BitmapLayer *s_char_layer;
static GBitmap *s_char_bitmap[CHAR_IMAGES+1];
static uint8_t s_char_cur_frame = 0;
static uint8_t s_battery_per = 0;
static uint32_t anime_time = ANIME_TIME;

static void update_time() {
  // Get a tm structure
  time_t temp = time(NULL); 
  struct tm *tick_time = localtime(&temp);

  // Create a long-lived buffer
  static char tbuffer[] = "00:00";

  // Write the current hours and minutes into the buffer
  if(clock_is_24h_style() == true) {
    //Use 2h hour format
    strftime(tbuffer, sizeof("00:00"), "%H:%M", tick_time);
  } else {
    //Use 12 hour format
    strftime(tbuffer, sizeof("00:00"), "%I:%M", tick_time);
  }

  // Display this time on the TextLayer
  text_layer_set_text(s_time_layer, tbuffer);
}

static void update_date() {
  // Get a tm structure
  time_t temp = time(NULL); 
  struct tm *tick_time = localtime(&temp);

  // Create a long-lived buffer
  static char dbuffer[] = "Jun  01";
  static char wbuffer[] = "Mon";

  strftime(dbuffer, sizeof("Jun  01"), "%b  %d", tick_time);
  strftime(wbuffer, sizeof("Mon"), "%a", tick_time);

  if (tick_time->tm_wday == 0) {
    text_layer_set_text_color(s_wday_layer, GColorSunsetOrange);
  } else if (tick_time->tm_wday == 6) {
    text_layer_set_text_color(s_wday_layer, GColorElectricBlue);
  } else {
    text_layer_set_text_color(s_wday_layer, GColorWhite);
  }
  
  // Display this date on the TextLayer
  text_layer_set_text(s_date_layer, dbuffer);
  text_layer_set_text(s_wday_layer, wbuffer);
}

static void battery_handler(BatteryChargeState charge) {
  APP_LOG(APP_LOG_LEVEL_DEBUG, "Battery handler called.");
  s_battery_per = charge.charge_percent;
  layer_mark_dirty(s_battery_layer);
}

static void connection_handler(bool connected) {
  APP_LOG(APP_LOG_LEVEL_DEBUG, "Connection handler called.");
  layer_set_hidden(bitmap_layer_get_layer(s_noconn_layer), connected);
}

static void render_battery(Layer *layer, GContext *ctx) {
  uint8_t i;
  graphics_context_set_fill_color(ctx, GColorWhite);
  APP_LOG(APP_LOG_LEVEL_DEBUG, "Render battery: percent %d", (int) s_battery_per);
  for (i = 0; s_battery_per >= (i+1) * 5; i++) {
    graphics_fill_circle(ctx, GPoint(i*7+1, 2), 1);
  }
}

static void anime_handler(void *context) {
  s_char_cur_frame++;
  if (s_char_cur_frame > CHAR_IMAGES)
    s_char_cur_frame = 0;
  bitmap_layer_set_bitmap(s_char_layer, s_char_bitmap[s_char_cur_frame]);
  layer_mark_dirty(bitmap_layer_get_layer(s_char_layer));
  anime_time -= CHAR_FRAME_MS;
  APP_LOG(APP_LOG_LEVEL_DEBUG, "Char anime: frame %d, remain: %d", (int) s_char_cur_frame, (int)anime_time);
  if (anime_time > 0) 
    app_timer_register(CHAR_FRAME_MS, anime_handler, NULL);
  return;
}

static void tap_handler(AccelAxisType axis, int32_t direction) {
  uint32_t need_to_run = 0;
  if (anime_time <= 0) {
    need_to_run = 1;
  }
  anime_time = ANIME_TIME;
  if (need_to_run)
    app_timer_register(1, anime_handler, NULL);
}

static void main_window_load(Window *window) {
  window_set_background_color(window, GColorBlack);

  s_char_layer = bitmap_layer_create(GRect(29, 0, 86, 80));
  s_char_bitmap[0] = gbitmap_create_with_resource(RESOURCE_ID_CHAR_0);
  s_char_bitmap[1] = gbitmap_create_with_resource(RESOURCE_ID_CHAR_1);
  s_char_bitmap[2] = gbitmap_create_with_resource(RESOURCE_ID_CHAR_2);
  s_char_bitmap[3] = s_char_bitmap[1];
  bitmap_layer_set_bitmap(s_char_layer, s_char_bitmap[0]);
  layer_add_child(window_get_root_layer(window), bitmap_layer_get_layer(s_char_layer));

  s_logo_layer = bitmap_layer_create(GRect(4, 88, 136, 11));
  s_logo_bitmap = gbitmap_create_with_resource(RESOURCE_ID_VOCALENDAR_LOGO);
  bitmap_layer_set_bitmap(s_logo_layer, s_logo_bitmap);
  layer_add_child(window_get_root_layer(window), bitmap_layer_get_layer(s_logo_layer));

  s_noconn_layer = bitmap_layer_create(GRect(124, 4, 16, 16));
  s_noconn_bitmap = gbitmap_create_with_resource(RESOURCE_ID_NO_CONNECTION);
  bitmap_layer_set_bitmap(s_noconn_layer, s_noconn_bitmap);
  layer_set_hidden(bitmap_layer_get_layer(s_noconn_layer), connection_service_peek_pebble_app_connection());
  layer_add_child(window_get_root_layer(window), bitmap_layer_get_layer(s_noconn_layer));
  
  // Create time TextLayer
  s_time_layer = text_layer_create(GRect(4, 110, 136, 36));
  text_layer_set_background_color(s_time_layer, GColorWhite);
  text_layer_set_text_color(s_time_layer, GColorBlack);
  text_layer_set_text(s_time_layer, "00:00");

  // Create date TextLayer
  s_date_layer = text_layer_create(GRect(4, 149, 136, 18));
  text_layer_set_background_color(s_date_layer, GColorClear);
  text_layer_set_text_color(s_date_layer, GColorWhite);

  // Create wday TextLayer
  s_wday_layer = text_layer_create(GRect(4, 149, 136, 18));
  text_layer_set_background_color(s_wday_layer, GColorClear);
  text_layer_set_text_color(s_wday_layer, GColorWhite);

  // Create battery Layer
  s_battery_layer = layer_create(GRect(4, 102, 136, 4));
  layer_set_update_proc(s_battery_layer, render_battery);

  // Create GFont
  s_time_font = fonts_load_custom_font(resource_get_handle(RESOURCE_ID_FONT_NOKIA_33));
  s_date_font = fonts_load_custom_font(resource_get_handle(RESOURCE_ID_FONT_NOKIA_17));

  // Apply to TextLayer
  text_layer_set_font(s_time_layer, s_time_font);
  text_layer_set_text_alignment(s_time_layer, GTextAlignmentCenter);
  text_layer_set_font(s_date_layer, s_date_font);
  text_layer_set_font(s_wday_layer, s_date_font);
  text_layer_set_text_alignment(s_wday_layer, GTextAlignmentRight);

  // Add it as a child layer to the Window's root layer
  layer_add_child(window_get_root_layer(window), text_layer_get_layer(s_time_layer));
  layer_add_child(window_get_root_layer(window), text_layer_get_layer(s_date_layer));
  layer_add_child(window_get_root_layer(window), text_layer_get_layer(s_wday_layer));
  layer_add_child(window_get_root_layer(window), s_battery_layer);

  // Make sure the time is displayed from the start
  app_timer_register(1, anime_handler, NULL);
  battery_handler(battery_state_service_peek());
  update_time();
  update_date();
}

static void main_window_unload(Window *window) {
  int i;

  //Unload GFont
  fonts_unload_custom_font(s_time_font);
  fonts_unload_custom_font(s_date_font);
  
  //Destroy GBitmap
  gbitmap_destroy(s_logo_bitmap);
  gbitmap_destroy(s_noconn_bitmap);
  for (i = 0; i < CHAR_IMAGES; i++) {
    gbitmap_destroy(s_char_bitmap[i]);
  }
  
  //Destroy BitmapLayer
  bitmap_layer_destroy(s_logo_layer);
  bitmap_layer_destroy(s_noconn_layer);
  bitmap_layer_destroy(s_char_layer);
  
  // Destroy TextLayer
  text_layer_destroy(s_time_layer);
  text_layer_destroy(s_date_layer);
  text_layer_destroy(s_wday_layer);

  layer_destroy(s_battery_layer);
}

static void tick_handler(struct tm *tick_time, TimeUnits units_changed) {
  update_time();
  if (units_changed & DAY_UNIT) {
    update_date();
  }
}
  
static void init() {
  // Create main Window element and assign to pointer
  s_main_window = window_create();

  // Set handlers to manage the elements inside the Window
  window_set_window_handlers(s_main_window, (WindowHandlers) {
    .load = main_window_load,
    .unload = main_window_unload
  });

  // Show the Window on the watch, with animated=true
  window_stack_push(s_main_window, true);

  // Register with TickTimerService
  tick_timer_service_subscribe(MINUTE_UNIT, tick_handler);
  accel_tap_service_subscribe(tap_handler);
  battery_state_service_subscribe(battery_handler);

  connection_service_subscribe((ConnectionHandlers) {
    .pebble_app_connection_handler = connection_handler
  });
}

static void deinit() {
  // Destroy Window
  window_destroy(s_main_window);

  /*
  tick_timer_service_unsubscribe();
  accel_tap_service_unsubscribe();
  battery_state_service_unsubscribe();
  connection_service_unsubscribe();
  */
}

int main(void) {
  init();
  app_event_loop();
  deinit();
}
