
#include "base64.h"
#include <string.h>

static unsigned char const PAD = '=';

static const unsigned char _encode_table[] = {
	0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50,
	0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
	0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76,
	0x77, 0x78, 0x79, 0x7a, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x2b, 0x2f,
};

static const unsigned char _decode_table[] = {
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x3e, 0xff, 0xff, 0xff, 0x3f,
	0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e,
	0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
	0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0xff, 0xff, 0xff, 0xff, 0xff,
};

static inline unsigned char enc(int c)
{
	return _encode_table[c & 63];
}

static inline unsigned char dec(int c)
{
	return _decode_table[c & 127];
}

void base64_encode(char const *src, size_t length, std::vector<char> *out)
{
	size_t srcpos, dstlen, dstpos;

	dstlen = (length + 2) / 3 * 4;
	out->resize(dstlen);
	if (dstlen == 0) {
		return;
	}
	char *dst = &out->at(0);
	dstpos = 0;
	for (srcpos = 0; srcpos < length; srcpos += 3) {
		int v = (unsigned char)src[srcpos] << 16;
		if (srcpos + 1 < length) {
			v |= (unsigned char)src[srcpos + 1] << 8;
			if (srcpos + 2 < length) {
				v |= (unsigned char)src[srcpos + 2];
				dst[dstpos + 3] = enc(v);
			} else {
				dst[dstpos + 3] = PAD;
			}
			dst[dstpos + 2] = enc(v >> 6);
		} else {
			dst[dstpos + 2] = PAD;
			dst[dstpos + 3] = PAD;
		}
		dst[dstpos + 1] = enc(v >> 12);
		dst[dstpos] = enc(v >> 18);
		dstpos += 4;
	}
}

void base64_decode(char const *src, size_t length, std::vector<char> *out)
{
	unsigned char const *begin = (unsigned char const *)src;
	unsigned char const *end = begin + length;
	unsigned char const *ptr = begin;
	out->clear();
	out->reserve(length * 3 / 4);
	int count = 0;
	int bits = 0;
	while (1) {
		if (isspace(*ptr)) {
			ptr++;
		} else {
			unsigned char c = 0xff;
			if (ptr < end && *ptr < 0x80) {
				c = dec(*ptr);
			}
			if (c < 0x40) {
				bits = (bits << 6) | c;
				count++;
			} else {
				if (count < 4) {
					bits <<= (4 - count) * 6;
				}
				c = 0xff;
			}
			if (count == 4 || c == 0xff) {
				if (count >= 2) {
					out->push_back(bits >> 16);
					if (count >= 3) {
						out->push_back(bits >> 8);
						if (count == 4) {
							out->push_back(bits);
						}
					}
				}
				count = 0;
				bits = 0;
				if (c == 0xff) {
					break;
				}
			}
			ptr++;
		}
	}
}

void base64_encode(std::vector<char> const *src, std::vector<char> *out)
{
	base64_encode(&src->at(0), src->size(), out);
}

void base64_decode(std::vector<char> const *src, std::vector<char> *out)
{
	base64_decode(&src->at(0), src->size(), out);
}

void base64_encode(char const *src, std::vector<char> *out)
{
	base64_encode((char const *)src, strlen(src), out);
}

void base64_decode(char const *src, std::vector<char> *out)
{
	base64_decode((char const *)src, strlen(src), out);
}

