/* pgx -- low level driver for X11 graphics */
#include <stdio.h>
#include <math.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

/* externals to communicate with main program */
extern FILE *pgraf;        /* file pointer to metacode file */
extern int argcglobal;     /* command line stuff from main program */
extern char **argvglobal;
extern float zoomfactor,deltheta,delphi,radfactor;
                           /* for changes in perspective */
extern int viewinfo;       /* set to plot view info on window */
extern char *tempfilename; /* name of temporary metacode file */

/* various random defines */

/* increments for perspective changes */
#define ZOOMINCR 1.1
#define THETAINCR 0.05
#define PHIINCR 0.1
#define RADINCR 1.414

/* color definitions */
#define RED "red"
#define GREEN "forest green"
#define BLUE "blue"

/* parameter for line drawing */
#define DEL 0.08

/* default pixel dimensions of pgraf window */
#define XLO 1
#define XHI 865
#define YLO 1
#define YHI 577
#define XMAX 15.1
#define YMAX 10.1
#define XFUDGE 3.0
#define YFUDGE 3.0
#define MAXPLOTS 501


/* X windows stuff */
#define DEF_FONT "9x15"
static char xfont[80];              /* storage for font name */
static Display *dpy;                /* X server connection */
static XFontStruct *fontstruct;     /* Font descriptor */
static unsigned long fg,bg,bd,fgred,fggreen,fgblue;  /* pixel values */
static XColor red,green,blue,exact; /* color structures */
static Colormap cmap;               /* the color map */
static Window win,rootwin;          /* Main window id, root window id */
static XSizeHints xsh;              /* Size hints for window manager */
static XWMHints xwmh =
    {(InputHint|StateHint),False,NormalState,0,0,0,0,0,0,}; /* Other hints */
static XEvent event;                /* event received */
static GC gc,gcred,gcgreen,gcblue,mygc;  /* graphics context */
static XGCValues gcv;               /* structure for making graphics context */
static int xsize = XHI - XLO;       /* size of x dimension of window */
static int ysize = YHI - YLO;       /* size of y dimension of window */
static int xret,yret,border,depth;  /* dummies */
static float charht,charwdth;       /* character dimensions */
static int linewidth = 0;           /* line width */
static int bnumber;                 /* number of button pressed (1, 2, or 3) */

/* variables to handle pgraf connection */

static char dash[] = {6, 3};
static char dot[] = {3, 6};
static char dot_dash[] = {9, 3, 3, 3};
static int dsegs[] = {2, 2, 4};
static int dpixels[] = {9, 9, 18};
static float xold,yold,xnew,ynew;
static int firsttime = 1;           /* initialize window only once */
static long pgrafoffset[MAXPLOTS];  /* offset of start of current graph */
static int plotnum = 0;             /* number of current plot */
static int plotread = 1;            /* number of plots read */

pginit()
{
  int fxsize;
  if (firsttime) {

/* find the offset at the head of the file */
    pgrafoffset[plotnum] = ftell(pgraf);

/* optional second command line argument is line width */
    if (argcglobal == 3) {
      argcglobal = 2;
      linewidth = atoi(argvglobal[2]);
    }

/* optional first command line argument is font name */
    if (argcglobal == 2) {
      argcglobal = 1;
      if (strlen(argvglobal[1]) > 79) argvglobal[1][79] = '\0';
      strcpy(xfont,argvglobal[1]);
    }
    else strcpy(xfont,DEF_FONT);

/* error message */
    if (argcglobal != 1) {
      fprintf(stderr,"Usage: pgx [font [linewidth]]\n");
      cexit(1);
    }

/* open X11 pgraf window on default display */
    if ((dpy = XOpenDisplay("")) == NULL) {
      fprintf(stderr,"pgx: can't open display %s\n",argvglobal[1]);
      cexit(1);
    }

/* get font */
    if ((fontstruct = XLoadQueryFont(dpy,xfont)) == NULL) {
      fprintf(stderr,"pgx: can't find font %s\n",xfont);
      cexit(1);
    }

/* Select colors */
    cmap = DefaultColormap(dpy,DefaultScreen(dpy));
    bd = WhitePixel(dpy,DefaultScreen(dpy));
    bg = WhitePixel(dpy,DefaultScreen(dpy));
    fg = BlackPixel(dpy,DefaultScreen(dpy));
    if (XAllocNamedColor(dpy,cmap,RED,&exact,&red) == 0) fgred = fg;
    else fgred = red.pixel;
    if (XAllocNamedColor(dpy,cmap,GREEN,&exact,&green) == 0)
         fggreen = fg;
    else fggreen = green.pixel;
    if (XAllocNamedColor(dpy,cmap,BLUE,&exact,&blue) == 0) fgblue = fg;
    else fgblue = blue.pixel;

/* set size hints for window manager */
    xsh.flags = (PPosition | PSize);
    xsh.height = ysize;
    xsh.width = xsize;
    xsh.x = XLO;
    xsh.y = YLO;

/* create a simple window */
    win = XCreateSimpleWindow(dpy,DefaultRootWindow(dpy),
        XLO,YLO,xsize,ysize,2,bd,bg);

/* set standard properties */
    XSetStandardProperties(dpy,win,"pgx","pgx",
         None,argvglobal,argcglobal,&xsh);
    XSetWMHints(dpy,win,&xwmh);

/* create graphics contexts for each color */
    gcv.foreground = fg;
    gcv.background = bg;
    gcv.line_width = linewidth;
    gcv.line_style = LineSolid;
    gcv.font = fontstruct->fid;
    gc = XCreateGC(dpy,win,(GCForeground|GCBackground|GCLineWidth|
         GCLineStyle|GCFont),&gcv);
    gcv.foreground = fgred;
    gcred = XCreateGC(dpy,win,(GCForeground|GCBackground|GCLineWidth|
         GCLineStyle|GCFont),&gcv);
    gcv.foreground = fggreen;
    gcgreen = XCreateGC(dpy,win,(GCForeground|GCBackground|GCLineWidth|
         GCLineStyle|GCFont),&gcv);
    gcv.foreground = fgblue;
    gcblue = XCreateGC(dpy,win,(GCForeground|GCBackground|GCLineWidth|
         GCLineStyle|GCFont),&gcv);

/* only look at button press and exposure events */
    XSelectInput(dpy,win,(OwnerGrabButtonMask|
         ButtonPressMask|ButtonReleaseMask|ExposureMask));

/* map the window */
    XMapWindow(dpy,win);

/* wait for an exposure event */
    while (1) {
      XNextEvent(dpy,&event);
      if ((event.type == Expose) && (event.xexpose.count == 0)) break;
    }

/* get size information */
    XGetGeometry(dpy,win,&rootwin,&xret,&yret,&xsize,&ysize,&border,&depth);

/* rationalize it */
    fxsize = 1.5*ysize;
    if (xsize > fxsize) xsize = 1.5*ysize + 0.5;
    else ysize = xsize/1.5 + 0.5;
    if (xsize < 1) xsize = 1;
    if (ysize < 1) ysize = 1;

/* set character sizes */
    charwdth = 0.5*(float)((fontstruct->max_bounds.width + 
         fontstruct->min_bounds.width)*XMAX)/xsize;
    charht = 0.9*(float)((fontstruct->max_bounds.ascent
         + fontstruct->max_bounds.descent)*YMAX)/ysize;

/* clear the window */
    XClearWindow(dpy,win);

/* don't do this more than once */
    firsttime = 0;
  }
}

pgflsh()
{

/* don't proceed without having called pginit */
  if (firsttime) {
    fprintf(stderr,"pgx: pginit must be called first\n");
    exit(1);
  }

/* wait for a mouse button push or expose */
  while (1) {
    XNextEvent(dpy,&event);

/* on to release if button push */
    if (event.type == ButtonPress) break;

/* expose event */
    if ((event.type == Expose) && (event.xexpose.count == 0)) {

/* get size information */
      XGetGeometry(dpy,win,&rootwin,&xret,&yret,&xsize,&ysize,&border,&depth);

/* rationalize it */
      if (xsize > (unsigned int)(1.5*ysize)) xsize = 1.5*ysize + 0.5;
      else ysize = xsize/1.5 + 0.5;
      if (xsize < 1) xsize = 1;
      if (ysize < 1) ysize = 1;

/* reset character sizes */
      charwdth = 0.5*(float)((fontstruct->max_bounds.width + 
           fontstruct->min_bounds.width)*XMAX)/xsize;
      charht = 0.9*(float)((fontstruct->max_bounds.ascent
           + fontstruct->max_bounds.descent)*YMAX)/ysize;

/* clear and return */
      XClearWindow(dpy,win);

/* seek to start of current plot */
      if (fseek(pgraf,pgrafoffset[plotnum],0) < 0) {
        fprintf(stderr,"pgx: seek to plot %d at %d failed\n",
             plotnum,pgrafoffset[plotnum]);
        cexit(1);
      }
      return(0);
    }
  }

/* find offset if we are at the frontier */
  if (plotnum == plotread - 1) {
    pgrafoffset[plotread++] = ftell(pgraf);
  }
  if (plotread >= MAXPLOTS) {
    fprintf(stderr,"pgx: too many plots\n");
    cexit(1);
  }

/* get button number */
  bnumber = event.xbutton.button;

/* middle button */
  if (bnumber == 2) {
    if (nokeypressed(bnumber)) {

/* go forward one plot */
      plotnum++;
      if (plotnum >= plotread) plotnum = plotread - 1;
    }
  }

/* left button */
  else if (bnumber == 1) {
    if (nokeypressed(bnumber)) {

/* go backward one plot */
      plotnum--;
      if (plotnum < 0) plotnum = 0;
    }
  }

/* bye! */
  else cexit(0);

/* seek to start of next desired plot */
  if (fseek(pgraf,pgrafoffset[plotnum],0) < 0) {
    fprintf(stderr,"pgx: seek to plot %d at %d failed\n",
         plotnum,pgrafoffset[plotnum]);
    cexit(1);
  }

/* button release assumed at this point */
  XNextEvent(dpy,&event);

/* clear the window */
  XClearWindow(dpy,win);
}

/* check to see if a key is pressed and take appropriate action */
nokeypressed(button)
int button;
{
  char key_vector[32];

/* find the state of the keyboard */
  XQueryKeymap(dpy,key_vector);

/* check for various keys and take action */
  if (iskey(XK_a,key_vector)) {           /* azimuth */
    if (button == 2) delphi += PHIINCR;
    else delphi -= PHIINCR;
    viewinfo = 1;
  }
  else if (iskey(XK_e,key_vector)) {      /* elevation */
    if (button == 2) deltheta += THETAINCR;
    else deltheta -= THETAINCR;
    viewinfo = 1;
  }
  else if (iskey(XK_z,key_vector)) {      /* zoom */
    if (button == 2) zoomfactor *= ZOOMINCR;
    else zoomfactor /= ZOOMINCR;
    viewinfo = 1;
  }
  else if (iskey(XK_r,key_vector)) {      /* radius */
    if (button == 2) radfactor *= RADINCR;
    else radfactor /= RADINCR;
    viewinfo = 1;
  }
  else {                       /* nothing of interest */
    return(1);
  }
  return(0);
}

/* determine which key is pressed */
iskey(sym,key_vector)
KeySym sym;
char key_vector[32];
{
  KeyCode code;
  int keybit;
  unsigned int byteindex,bitindex;
  code = XKeysymToKeycode(dpy,sym);
  if (code == NoSymbol) return(0);
  byteindex = code/8;
  bitindex = code & 7;
  keybit = 1 & (key_vector[byteindex] >> bitindex);
  return(keybit);
}

pgmove(x,y)
float x,y;
{

/* don't proceed without having called pginit */
  if (firsttime) {
    fprintf(stderr,"pgx: pginit must be called first\n");
    exit(1);
  }

  xnew = x;
  ynew = y;
}

pgdraw(x,y,ltype)
float x,y;
int ltype;
{
  int temp,ixold,ixnew,iyold,iynew,phasepos,color;

/* don't proceed without having called pginit */
  if (firsttime) {
    fprintf(stderr,"pgx: pginit must be called first\n");
    exit(1);
  }

/* four colors -- black, red, green, blue */
  color = 0;
  while (ltype < 1) {
    ltype += 4;
    color--;
  }
  while (ltype > 4) {
    ltype -= 4;
    color++;
  }
  if (color == 3) mygc = gcblue;
  else if (color == 2) mygc = gcgreen;
  else if (color == 1) mygc = gcred;
  else mygc = gc;

/* push coordinates */
  xold = xnew;
  yold = ynew;
  xnew = x;
  ynew = y;

/* set up integer coordinates */
  ixold = (xsize)*xold/XMAX + XFUDGE;
  ixnew = (xsize)*xnew/XMAX + XFUDGE;
  iyold = (ysize)*(1. - yold/YMAX) - YFUDGE;
  iynew = (ysize)*(1. - ynew/YMAX) - YFUDGE;

/* find phase position and make stroke in positive direction */
  if (abs(ixnew - ixold) > abs(iynew - iyold)) {
    if (ixnew < ixold) {
      temp = ixnew;
      ixnew = ixold;
      ixold = temp;
      temp = iynew;
      iynew = iyold;
      iyold = temp;
    }
    phasepos = ixold;
  }
  else {
    if (iynew < iyold) {
      temp = ixnew;
      ixnew = ixold;
      ixold = temp;
      temp = iynew;
      iynew = iyold;
      iyold = temp;
    }
    phasepos = iyold;
  }
  if (ltype == 1) {
    XSetLineAttributes(dpy,mygc,linewidth,LineSolid,CapButt,JoinMiter);
  }
  else if (ltype == 2) {
    XSetLineAttributes(dpy,mygc,linewidth,LineOnOffDash,CapButt,JoinMiter);
    XSetDashes(dpy,mygc,mod(phasepos,dpixels[0]),dash,dsegs[0]);
  }
  else if (ltype == 3) {
    XSetLineAttributes(dpy,mygc,linewidth,LineOnOffDash,CapButt,JoinMiter);
    XSetDashes(dpy,mygc,mod(phasepos,dpixels[1]),dot,dsegs[1]);
  }
  else {
    XSetLineAttributes(dpy,mygc,linewidth,LineOnOffDash,CapButt,JoinMiter);
    XSetDashes(dpy,mygc,mod(phasepos,dpixels[2]),dot_dash,dsegs[2]);
  }

/* draw the vector */
  XDrawLine(dpy,win,mygc,ixold,iyold,ixnew,iynew);
}

pgtext(x,y,text)
float x,y;
char *text;
{
  int loop,nbytes,ix,iy;
  char buffer[101];

/* don't proceed without having called pginit */
  if (firsttime) {
    fprintf(stderr,"pgx: pginit must be called first\n");
    exit(1);
  }

/* find length of text */
  nbytes = strlen(text);
  if (nbytes > 100) nbytes = 100;

/* copy text into padded buffer */
  strncpy(buffer,text,100);
  buffer[nbytes] = '\0';

/* convert newline into null */
  for (loop = 0; loop < 100; loop++) if (buffer[loop] == '\n')
       buffer[loop] = '\0';
  nbytes = strlen(buffer);

/* compute integer coordinates for left middle of text */
  ix = (xsize)*(x + 0.5*charwdth)/XMAX + XFUDGE;
  iy = (ysize)*(1. - (y - 0.25*charht)/YMAX) - YFUDGE;

/* write the text */
  XDrawImageString(dpy,win,gc,ix,iy,buffer,nbytes);
}

pginfo(dcx,dcy)
float *dcx,*dcy;
{

/* don't proceed without having called pginit */
  if (firsttime) {
    fprintf(stderr,"pgx: pginit must be called first\n");
    exit(1);
  }

/* return information on character cell size */
  *dcx = charwdth;
  *dcy = charht;
}
