/*
 * nasd_cheops_mgr_dce.c
 */
/*
 * DCE Callbacks and Mechanisms for Cheops Manager
 */
/*
 * Copyright (c) 1996,1997,1998,1999 Carnegie Mellon University.
 * All rights reserved.
 *
 * Author: Khalil Amiri, CMU SCS/ECE, July 18 1997
 *
 * Permission to use, copy, modify and distribute this software and
 * its documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND
 * FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * Carnegie Mellon requests users of this software to return to
 *
 *  Software Distribution Coordinator  or  Software.Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 * any improvements or extensions that they make and grant Carnegie Mellon the
 * rights to redistribute these changes.
 */

/*
 * cheops storage manager interface
 */

#include <nasd/nasd_options.h>

#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE

# undef NASD_PTHREAD_EXC
# define NASD_PTHREAD_EXC 1

# include <stdio.h>
# include <stdlib.h>
# include <dce/dce_error.h>
# include <rpc/rpc.h>
# include <dce/stubbase.h>
# include <errno.h>
# include <errno.h>
# include <fcntl.h>
# include <unistd.h>
# include <malloc.h>

# include <nasd/nasd_general.h>
# include <nasd/nasd_cheops_types.h>
# include <nasd/nasd_cheops_common.h>
# include <nasd/nasd_cheops_mgr_common.h>
# include <nasd/nasd_cheops_mgr_structs.h>
# include <nasd/nasd_cheops_mgr.h>
# include <nasd/nasd_cheops_mgr_internal.h>

int nasd_cheops_dce_protseq_installed = 0;
int nasd_cheops_active = 0;

# define DOBEGIN(_opname_) {                                            \
  nasd_cheops_mgr_begin_rpc();                                          \
}
# define DORETURN(_opname_) {                                           \
  nasd_cheops_mgr_end_rpc();                                            \
  return;                                                               \
}

/* debugging */
#define NASD_CHEOPS_OP_DEBUG 1

/*
 * RPC Callbacks
 */

void 
nasd_cheops_null_cm(
  handle_t                               client_handle, 
  nasd_res_otw_t                         out_res_otw,
  error_status_t                        *op_status)
{
  nasd_res_t res;

  DOBEGIN(null);
  res.nasd_status = NASD_SUCCESS;
  nasd_res_t_marshall(&res, out_res_otw);
  *op_status = 0;          /* always successful */
#if NASD_CHEOPS_OP_DEBUG > 0
  (cheops_Msg "Cheops storage manager: Handled %s\n","nasd_cheops_NULL_cm");
#endif /* CHEOPS_OP_DEBUG */
  DORETURN(null);
}

void
nasd_cheops_bs_create_cm(
  handle_t                               client_handle,
  nasd_cheops_bs_cr_args_otw_t       in_args_otw,
  nasd_cheops_bs_cr_res_otw_t        out_res_otw,
  error_status_t                        *op_status)
{
  nasd_cookie_t                in_cookie;
  nasd_cheops_bs_cr_args_t args;
  nasd_cheops_bs_cr_res_t res;
	int rc=0;                   
	nasd_cheops_qos_req_t qos_req; /* XXX default qos struct */

  DOBEGIN(bs_create);

  nasd_cheops_bs_cr_args_t_unmarshall(in_args_otw, &args);
  in_cookie = args.in_cookie;

  TRY {
    rc = nasd_cheops_mgr_check_cookie(client_handle, in_cookie, 0,
                                      NASD_CHEOPS_OP_BS_CREATE); 
    if (rc) {
      res.out_nasd_status = NASD_AUTHCHECK_FAILED;
      *op_status = 0;
    } else {
      bzero((char *)&res.out_bs_handle, sizeof(nasd_cheops_bs_handle_t));
      rc = nasd_cheops_bs_create(NASD_FALSE, &qos_req, &res.out_bs_handle); 
      *op_status = 0;
      res.out_nasd_status = rc;
    }
  } CATCH_ALL {
    res.out_nasd_status = NASD_REMOTE_TRAP;
#if NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0
    nasd_printf("CHEOPS MGR: caught exception performing bs_create\n");
#endif /* NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0 */
  } ENDTRY

#if NASD_CHEOPS_OP_DEBUG > 0
  (cheops_Msg "Cheops storage manager: Handled %s (new id=0x%"
   NASD_ID_FMT ")\n",
	 "nasd_cheops_BS_CREATE_cm", res.out_bs_handle.bsid);
#endif /* CHEOPS_OP_DEBUG */

  nasd_cheops_bs_cr_res_t_marshall(&res, out_res_otw);

  DORETURN(bs_create);
}

void
nasd_cheops_bs_qos_create_cm(
  handle_t                               client_handle,
  nasd_cheops_bs_cr_args_otw_t       in_args_otw,
  nasd_cheops_bs_cr_res_otw_t        out_res_otw,
  error_status_t                        *op_status)
{
  nasd_cookie_t in_cookie;
  int  failures = 0;
	int  rc=0;                   
  nasd_cheops_bs_qc_args_t args;
  nasd_cheops_bs_qc_res_t res;

  DOBEGIN(bs_qos_create);

  nasd_cheops_bs_qc_args_t_unmarshall(in_args_otw, &args);
  in_cookie = args.in_cookie;

  TRY {
    rc = nasd_cheops_mgr_check_cookie (client_handle, in_cookie, 0,
                                       NASD_CHEOPS_OP_BS_CREATE); 
    if (rc) {
      res.out_nasd_status = NASD_AUTHCHECK_FAILED;
      *op_status = 0;
    } else {
      bzero((char *)&res.out_bs_handle, sizeof(nasd_cheops_bs_handle_t));
      rc = nasd_cheops_bs_create(NASD_TRUE, &args.in_qos_req,
                                 &res.out_bs_handle);
      res.out_nasd_status = rc;
      *op_status = 0;
    } 
  } CATCH_ALL {
#if NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0
    nasd_printf("CHEOPS MGR: caught exception performing bs_qos_create\n");
#endif /* NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0 */
  } ENDTRY

#if NASD_CHEOPS_OP_DEBUG > 0
  (cheops_Msg "Cheops storage manager: Handled %s\n",
   "nasd_cheops_BS_QOSCREATE_cm");
#endif /* CHEOPS_OP_DEBUG */

  nasd_cheops_bs_qc_res_t_marshall(&res, out_res_otw);

  DORETURN(bs_qos_create);
}

void 
nasd_cheops_bs_lookup_cm(
  handle_t                               client_handle,
  nasd_cheops_bs_lu_args_otw_t       in_args_otw,
  nasd_cheops_bs_lu_res_otw_t        out_res_otw,
  error_status_t                        *op_status)
{
  int rc;
  nasd_cheops_bs_lu_args_t args;
  nasd_cheops_bs_lu_res_t res;
  nasd_cookie_t in_cookie;
  nasd_identifier_t in_bsid;

  DOBEGIN(bs_lookup);

  nasd_cheops_bs_lu_args_t_unmarshall(in_args_otw, &args);
  in_cookie = args.in_cookie;
  in_bsid = args.in_bsid;

  TRY {
    rc = nasd_cheops_mgr_check_cookie(client_handle, in_cookie, in_bsid,
                                      NASD_CHEOPS_OP_BS_LOOKUP);
    if (rc) {
      res.out_nasd_status = NASD_AUTHCHECK_FAILED;
      *op_status = 0;
    } else {
      rc = nasd_cheops_bs_lookup(in_bsid, &res.out_bs_handle); 
      res.out_nasd_status = rc;
      *op_status = 0;
    }
  } CATCH_ALL {
#if NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0
    nasd_printf("CHEOPS MGR: caught exception performing bs_lookup\n");
#endif /* NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0 */
  } ENDTRY

#if NASD_CHEOPS_OP_DEBUG > 0
  (cheops_Msg "Cheops storage manager: Handled %s(rc=%d)\n",
   "nasd_cheops_BS_LOOKUP_cm",rc);
#endif /* CHEOPS_OP_DEBUG */

  nasd_cheops_bs_lu_res_t_marshall(&res, out_res_otw);

  DORETURN(bs_lookup);
}

void
nasd_cheops_bs_remove_cm(
  handle_t                               client_handle,
  nasd_cheops_bs_rm_args_otw_t       in_args_otw,
  nasd_cheops_bs_rm_res_otw_t        out_res_otw,
  error_status_t                        *op_status)
{
  nasd_cheops_bs_rm_args_t args;
  nasd_cheops_bs_rm_res_t res;
  nasd_cookie_t in_cookie;
  nasd_identifier_t in_bsid;
  int  rc;

  DOBEGIN(bs_remove);

	nasd_cheops_bs_rm_args_t_unmarshall(in_args_otw, &args);
  in_cookie = args.in_cookie;
  in_bsid = args.in_bsid;

  TRY {
    rc = nasd_cheops_mgr_check_cookie(client_handle, in_cookie, in_bsid,
                                      NASD_CHEOPS_OP_BS_REMOVE);
    if (rc) {
      res.out_nasd_status = NASD_AUTHCHECK_FAILED;
      *op_status = 0;
    } else {
      rc = nasd_cheops_bs_remove(in_bsid); 
      res.out_nasd_status = rc;
      *op_status = 0;
    }
  } CATCH_ALL {
#if NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0
    nasd_printf("CHEOPS MGR: caught exception performing bs_remove\n");
#endif /* NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0 */
  } ENDTRY

#if NASD_CHEOPS_OP_DEBUG > 0
  (cheops_Msg "Cheops storage manager: Handled %s\n",
   "nasd_cheops_BS_REMOVE_cm");
#endif /* CHEOPS_OP_DEBUG */

  nasd_cheops_bs_rm_res_t_marshall(&res, out_res_otw);

  DORETURN(bs_remove);
}

void
nasd_cheops_bs_getattr_cm(
  handle_t                               client_handle,
  nasd_cheops_bs_ga_args_otw_t      in_args_otw,
  nads_cheops_bs_ga_res_otw_t       out_res_otw,
  error_status_t                        *op_status)
{
  nasd_cheops_bs_ga_args_t args;
  nads_cheops_bs_ga_res_t res;
  nasd_cookie_t         in_cookie;
  nasd_identifier_t     in_bsid;
  int rc=0;

  DOBEGIN(bs_getattr);

	nasd_cheops_bs_ga_args_t_unmarshall(in_args_otw, &args);
  in_cookie = args.in_cookie;
  in_bsid = args.in_bsid;

  TRY {
    rc = nasd_cheops_mgr_check_cookie(client_handle, in_cookie, in_bsid,
                                      NASD_CHEOPS_OP_BS_GETATTR);
    if (rc) {
      res.out_nasd_status = NASD_AUTHCHECK_FAILED;
      *op_status = 0;
    } else {
      rc = nasd_cheops_bs_getattr(in_bsid, &res.out_attr); 
      res.out_nasd_status = rc;
      *op_status = 0;
    }
  } CATCH_ALL {
#if NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0
    nasd_printf("CHEOPS MGR: caught exception performing bs_getattr\n");
#endif /* NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0 */
  } ENDTRY

#if NASD_CHEOPS_OP_DEBUG > 0
  (cheops_Msg "Cheops storage manager: Handled %s\n",
   "nasd_cheops_BS_GETATTR_cm");
#endif /* CHEOPS_OP_DEBUG */

  nads_cheops_bs_ga_res_t_marshall(&res, out_res_otw);

  DORETURN(bs_getattr);
}

void
nasd_cheops_bs_setattr_cm(
  handle_t                               client_handle,
  nasd_cheops_bs_sa_args_otw_t      in_args_otw,
  nads_cheops_bs_sa_res_otw_t       out_res_otw,
  error_status_t                        *op_status)
{
  nasd_cheops_bs_sa_args_t args;
  nads_cheops_bs_sa_res_t res;
  nasd_cookie_t         in_cookie;
  nasd_identifier_t     in_bsid;
  nasd_attribute_t      in_attr;
  int rc=0;

  DOBEGIN(bs_setattr);

	nasd_cheops_bs_sa_args_t_unmarshall(in_args_otw, &args);
  in_cookie = args.in_cookie;
  in_bsid = args.in_bsid;
  in_attr = args.in_attr;

  TRY {
    rc = nasd_cheops_mgr_check_cookie(client_handle, in_cookie, in_bsid,
                                      NASD_CHEOPS_OP_BS_SETATTR);
    if (rc) {
      res.out_nasd_status = NASD_AUTHCHECK_FAILED;
      *op_status = 0;
    } else {
      rc = nasd_cheops_bs_setattr(in_bsid, in_attr); 
      res.out_nasd_status = rc;
      *op_status = 0;
    }
  } CATCH_ALL {
#if NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0
    nasd_printf("CHEOPS MGR: caught exception performing bs_setattr\n");
#endif /* NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0 */
  } ENDTRY

#if NASD_CHEOPS_OP_DEBUG > 0
  (cheops_Msg "Cheops storage manager: Handled %s\n",
   "nasd_cheops_BS_SETATTR_cm");
#endif /* CHEOPS_OP_DEBUG */

  nads_cheops_bs_sa_res_t_marshall(&res, out_res_otw);

  DORETURN(bs_setattr);
}

void
nasd_cheops_bs_refresh_cm(
  handle_t                               client_handle,
  nasd_cheops_bs_re_args_otw_t      in_args_otw,
  nasd_cheops_bs_re_res_otw_t       out_res_otw,
  error_status_t                        *op_status)
{
  nasd_cheops_bs_re_args_t args;
  nasd_cheops_bs_re_res_t res;
  nasd_cookie_t in_cookie;
  nasd_identifier_t in_bsid;
  int  rc;

  DOBEGIN(bs_refresh);

	nasd_cheops_bs_re_args_t_unmarshall(in_args_otw, &args);
  in_cookie = args.in_cookie;
  in_bsid = args.in_bsid;

  TRY {
    rc = nasd_cheops_mgr_check_cookie(client_handle, in_cookie, in_bsid,
                                      NASD_CHEOPS_OP_BS_LOOKUP);
    if (rc) {
      res.out_nasd_status = NASD_AUTHCHECK_FAILED;
      *op_status = 0;
    } else {
      bzero((char *)&res.out_bs_handle, sizeof(nasd_cheops_bs_handle_t));
      rc = nasd_cheops_bs_lookup(in_bsid, &res.out_bs_handle); 
      res.out_nasd_status = rc;
      *op_status = 0;
    }
  } CATCH_ALL {
#if NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0
    nasd_printf("CHEOPS MGR: caught exception performing bs_refresh\n");
#endif /* NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0 */
  } ENDTRY

#if NASD_CHEOPS_OP_DEBUG > 0
  (cheops_Msg "Cheops storage manager: Handled %s\n",
   "nasd_cheops_BS_REFRESH_cm");
#endif /* CHEOPS_OP_DEBUG */

  nasd_cheops_bs_re_res_t_marshall(&res, out_res_otw);
  DORETURN(bs_refresh);
}

void
nasd_cheops_dr_lookup_cm(
  handle_t                              client_handle,
  nasd_cheops_dr_lu_args_otw_t      in_args_otw,
  nasd_cheops_dr_lu_res_otw_t       out_res_otw,
  error_status_t                        *op_status)
{
  nasd_cheops_dr_lu_args_t args;
  nasd_cheops_dr_lu_res_t res;
  nasd_cookie_t in_cookie;
  nasd_disk_ident_t in_drive_id;
  int rc=0;

  DOBEGIN(dr_lookup);

  nasd_cheops_dr_lu_args_t_unmarshall(in_args_otw, &args);
  in_cookie = args.in_cookie;
  in_drive_id = args.in_drive_id;

  TRY {
    rc = nasd_cheops_mgr_check_cookie(client_handle, in_cookie, in_drive_id,
                                      NASD_CHEOPS_OP_DR_LOOKUP);
    if (rc) {
      res.out_nasd_status = NASD_AUTHCHECK_FAILED;
      *op_status = 0;
    } else {
      bzero((char *)&res.out_dr_info, sizeof(nasd_cheops_cl_dinfo_t));
      rc = nasd_cheops_dr_lookup(in_drive_id, &res.out_dr_info); 
      res.out_nasd_status = rc;
      *op_status = 0;
    }
	} CATCH_ALL {
#if NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0
    nasd_printf("CHEOPS MGR: caught exception performing dr_lookup\n");
#endif /* NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0 */
  } ENDTRY

#if NASD_CHEOPS_OP_DEBUG > 0
  (cheops_Msg "Cheops storage manager: Handled %s\n",
   "nasd_cheops_DR_LOOKUP_cm");
#endif /* CHEOPS_OP_DEBUG */

  nasd_cheops_dr_lu_res_t_marshall(&res, out_res_otw);

  DORETURN(dr_lookup);
}

void
nasd_cheops_mgr_lookup_cm(
  handle_t                              client_handle,
  nasd_cheops_mgr_lu_args_otw_t     in_args_otw,
  nasd_cheops_mgr_lu_res_otw_t      out_res_otw,
  error_status_t                        *op_status)
{
  nasd_cheops_mgr_lu_args_t args;
  nasd_cheops_mgr_lu_res_t res;
  nasd_cookie_t in_cookie;
  nasd_identifier_t in_bsid;
  int rc=0;

  DOBEGIN(mgr_lookup);

  nasd_cheops_mgr_lu_args_t_unmarshall(in_args_otw, &args);
  in_cookie = args.in_cookie;
  in_bsid = args.in_bsid;

  TRY {
    rc = nasd_cheops_mgr_check_cookie(client_handle, in_cookie, in_bsid,
                                      NASD_CHEOPS_OP_MGR_LOOKUP);
    if (rc) {
      res.out_nasd_status = NASD_AUTHCHECK_FAILED;
      *op_status = 0;
    } else {
      bzero((char *)&res.out_mgr_info, sizeof(nasd_cheops_mgr_info_t));
      rc = nasd_cheops_mgr_lookup(in_bsid, &res.out_mgr_info); 
      res.out_nasd_status = rc;
      *op_status = 0;
    }
	} CATCH_ALL {
#if NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0
    nasd_printf("CHEOPS MGR: caught exception performing mgr_lookup\n");
#endif /* NASD_CHEOPS_RPC_DEBUG_DCE_EXCEPTION > 0 */
  } ENDTRY

#if CHEOPS_OP_DEBUG > 0
  (cheops_Msg "Cheops storage manager: Handled %s\n",
   "nasd_cheops_MGR_LOOKUP_cm");
#endif /* CHEOPS_OP_DEBUG */

  nasd_cheops_mgr_lu_res_t_marshall(&res, out_res_otw);

  DORETURN(mgr_lookup);
}

/*
 * RPC Mechanism
 */

/*
 * !!! Do not hand-modify! Generate this stanza with "make mepv"
 */

static nasd_cheops_mgr_v0_2_epv_t nasd_cheops_mgr_epv = {
nasd_cheops_null_cm
,nasd_cheops_bs_lookup_cm
,nasd_cheops_mgr_lookup_cm
,nasd_cheops_dr_lookup_cm
,nasd_cheops_bs_refresh_cm
,nasd_cheops_bs_qos_create_cm
,nasd_cheops_bs_create_cm
,nasd_cheops_bs_remove_cm
,nasd_cheops_bs_getattr_cm
,nasd_cheops_bs_setattr_cm
};

nasd_status_t
nasd_cheops_mgr_rpc_specific_init(void)
{
  return NASD_SUCCESS;
}

nasd_status_t
nasd_cheops_mgr_rpc_specific_startup(void)
{
  rpc_binding_vector_p_t binding_vector;
  rpc_if_handle_t nasd_cheops_ifspec;
  rpc_if_rep_p_t nasd_cheops_ifrep;
  dce_error_string_t err_str;
  error_status_t status;
  int ps;
  nasd_status_t rc;

  nasd_cheops_ifrep = (rpc_if_rep_p_t)nasd_cheops_mgr_v0_2_s_ifspec;
  nasd_cheops_ifrep->manager_epv = (rpc_mgr_epv_t)&nasd_cheops_mgr_epv;
  nasd_cheops_ifspec = nasd_cheops_mgr_v0_2_s_ifspec;

  if (nasd_cheops_dce_protseq_installed == 0) {
    rpc_server_use_protseq_if((unsigned_char_t *)"ncadg_ip_udp",
      rpc_c_protseq_max_reqs_default, nasd_cheops_ifspec, &status);
    if (status != rpc_s_ok) {
      dce_error_inq_text(status, err_str, &ps);
      if (ps)
        strcpy(err_str, "unknown");
      nasd_printf("CHEOPS MGR ERROR: cannot register protocol "
                  "sequence 0x%x (%s)\n",
        (unsigned)status, err_str);
      return(NASD_FAIL);
    }
    nasd_cheops_dce_protseq_installed = 1;
  }

  rpc_server_register_if(nasd_cheops_ifspec, NULL, NULL, &status);
  if (status != rpc_s_ok) {
    dce_error_inq_text(status, err_str, &ps);
    if (ps)
      strcpy(err_str, "unknown");
    nasd_printf("EDRFS ERROR: cannot register interface 0x%x (%s)\n",
      (unsigned)status, err_str);
    return(NASD_FAIL);
  }

  return(NASD_SUCCESS);
}

void
nasd_cheops_mgr_rpc_specific_stop(void)
{
  rpc_if_handle_t nasd_cheops_ifspec;
  dce_error_string_t err_str;
  error_status_t status;
  int ps;

  nasd_cheops_active = 0;

  nasd_cheops_ifspec = nasd_cheops_mgr_v0_2_s_ifspec;
  rpc_server_unregister_if(nasd_cheops_ifspec, NULL, &status);
  if (status) {
    dce_error_inq_text(status, err_str, &ps);
    nasd_printf("CHEOPS MGR: cannot unregister server_if status %d (%s)\n",
                status, err_str);
    NASD_PANIC();
  }
}

nasd_status_t
nasd_cheops_mgr_rpc_specific_listen(
  int          service_threads,
  nasd_uint16  ipport)
{
  char err_str[dce_c_error_string_len];
  error_status_t status;
  int dce_ps;

  rpc_server_listen(service_threads, &status);
  if (status) {
#ifdef KERNEL
    if (status == rpc_s_already_listening)
      return(NASD_SUCCESS);
#endif /* KERNEL */

    dce_error_inq_text(status, err_str, &dce_ps);

    if (dce_ps) {
      strcpy(err_str, "unknown");
    }

    nasd_printf("CHEOPS MGR ERROR: DCE listener aborted! status=0x%x (%s)\n",
      status, err_str);

    return(NASD_FAIL);
  }

  return(NASD_SUCCESS);
}

nasd_status_t
nasd_cheops_mgr_rpc_specific_set_stacksize(
  int  stacksize)
{
#if defined(DEC_OSF) && defined(KERNEL)
  return(NASD_OP_NOT_SUPPORTED);
#else /* defined(DEC_OSF) && defined(KERNEL) */
  error_status_t status;

  /* XXX better error handling */

  rpc_mgmt_set_server_stack_size(stacksize, &status);
  if (status)
    return(NASD_FAIL);

  return(NASD_SUCCESS);
#endif /* defined(DEC_OSF) && defined(KERNEL) */
}
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE */

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
