/*
 * nasd_security.h
 *
 * Definitions for common security module.
 *
 * Author: Marc Unangst
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1997,1998,1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#ifndef _NASD_SECURITY_H_
#define _NASD_SECURITY_H_ 

#include <nasd/nasd_options.h>
#include <nasd/nasd_types.h>
#include <nasd/nasd_common.h>
#include <cryptolib/libcrypt.h>

#if NASD_SECURITY_PRINT_STATS > 0
nasd_status_t nasd_SHA1_init_stats();
nasd_status_t nasd_HMAC_init_stats();

nasd_status_t nasd_SHA1_zero_stats();
nasd_status_t nasd_HMAC_zero_stats();

void nasd_HMAC_print_stats();
void nasd_SHA1_print_stats();
#endif /* NASD_SECURITY_PRINT_STATS > 0 */

extern nasd_uint16 nasd_security_levels[];
#if NASD_SECURE_RPCS_ENABLE > 0
#define NASD_MAX_SECURITY_LEVEL 3
#else /* NASD_SECURE_RPCS_ENABLE > 0 */
#define NASD_MAX_SECURITY_LEVEL 0
#endif /* NASD_SECURE_RPCS_ENABLE > 0 */


#define NASD_SEC_IS_CAPABILITY(_t_)	((_t_) == NASD_RED_CAPABILITY || (_t_) == NASD_BLACK_CAPABILITY)

/* number of bytes to calculate a digest over when doing heirarchical
   MACs.  Same as NASD_OD_BASIC_BLOCKSIZE for now.  If you change this
   to something other than NASD_OD_BASIC_BLOCKSIZE, you will probably
   have to change a lot of the piping/integrity code. */
#define NASD_SEC_DIGEST_BLOCKSIZE	8192
/* number of digests to include per HMAC */
#define NASD_SEC_HMAC_NUMBLOCKS		4
#define NASD_SEC_HMAC_BLOCKSIZE		(NASD_SEC_HMAC_NUMBLOCKS*NASD_SEC_DIGEST_BLOCKSIZE)

typedef struct nasd_security_context_s		nasd_security_context_t;
typedef struct nasd_sec_keyring_s       	nasd_sec_keyring_t;

struct nasd_security_context_s {
  HMAC_SHA1_CTX HMAC_context;
  SHA_CTX SHA1_ctx;
  nasd_uint16 protection;
  int pending_hmac;
  nasd_offset_t cur_off;
  nasd_len_t remain;
  nasd_offset_t last_commit_off;
};

struct nasd_sec_keyring_s {
  nasd_key_t master_key;
  nasd_key_t drive_key;
  nasd_partnum_t partnum;
  nasd_key_t part_key;
  nasd_key_t red_key;
  nasd_key_t black_key;
};

/* common/nasd_security.c */
nasd_status_t nasd_sec_password_to_keys(char *password,
                                        nasd_partnum_t partnum,
                                        nasd_sec_keyring_t *keys);
void nasd_sec_generate_key(char *password, nasd_byte constant,
                           nasd_key_t key);
nasd_status_t nasd_sec_privacy_key(nasd_key_t cred_key, nasd_key_t out_key);
nasd_status_t nasd_sec_integrity_key(nasd_key_t cred_key, nasd_key_t out_key);
void nasd_sec_fill_nonce(nasd_security_param_t *sec_param,
                         nasd_key_t op_key,
                         nasd_key_t integrity_key,
                         nasd_iv_t icv,
                         int is_reply,
                         void *args_otw,
                         int args_otw_len,
                         nasd_timespec_t *timestamp,
                         nasd_digest_nonce_otw_t digest_otw,
                         nasd_security_context_t *context);
void nasd_sec_print_cookie(nasd_key_t in_key, nasd_capability_t *cap);
void nasd_sec_print_key(nasd_key_t in_key);
void nasd_sec_build_capability(nasd_partnum_t partnum,
                               nasd_identifier_t ni,
                               nasd_access_rights_t rights,
                               nasd_audit_t AuditID,
                               nasd_int32 expiration,
                               nasd_uint16 min_protection,
                               nasd_uint16 type,
                               nasd_offset_t region_start,
                               nasd_offset_t region_end,
                               nasd_av_t av,
                               nasd_key_t basis_key,
                               nasd_cookie_t *out_cookie);
nasd_status_t nasd_sec_seclevel_to_protection(int sec_level,
                                              nasd_uint16 *protection);
char *nasd_sec_level_string(int sec_level);

#endif /* !_NASD_SECURITY_H_ */

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
