/*
 * pdrivename.c
 *
 * Fetch drive's build string.  Not strictly a test, but an important
 * part of the regression process
 *
 * Author: David Rochberg, Jim Zelenka
*/
/*
 * Copyright (c) of Carnegie Mellon University, 1996,1997,1998,1999,2000.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#include <nasd/nasd_options.h>
#include <nasd/nasd_getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <nasd/nasd_pdrive.h>
#include <nasd/nasd_pdrive_client.h>
#include <nasd/nasd_pdrive_client_kpdev.h>
#include <nasd/nasd_security.h>

char *progname;

int                        binding_type;
int                        binding_args_len;
void                      *binding_args;
char                      *binding_port = NASD_PDRIVE_PORT;
nasd_drive_param_kpdev_t   kpdev_args;
int                        nondefault_binding = 0;

nasd_error_string_t error_text;

void
usage()
{
  int i;
  fprintf(stderr, "USAGE: %s [options] servername master_password\n", progname);
  fprintf(stderr, "Options:\n");
  fprintf(stderr, "  -k use kernel device\n");
  fprintf(stderr, "  -l use colocated drive\n");
  fprintf(stderr, "  -M use message queues\n");
  fprintf(stderr, "  -s sec_level\n");
  fprintf(stderr, "  where sec_level is one of\n");
  for(i = 0; i <= NASD_MAX_SECURITY_LEVEL; i++) {
    fprintf(stderr, "     %d %s\n", i, nasd_sec_level_string(i));
  }
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE
  fprintf(stderr, "  -T use DCE-TCP\n");
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE */
  fflush(stderr);
  exit(1);
}

int
main(
  int     argc,
  char  **argv)
{
  nasd_ctrl_drive_info_t res;
  nasd_security_param_t sec_param;
  nasd_cookie_t cookie;
  nasd_sec_keyring_t keys;
  nasd_rpc_status_t status;
  nasd_drive_handle_t h;
  char *server_name;
  nasd_status_t rc;
  int p;
  char *master_password;
  nasd_timespec_t tm;
  char c;
  int sec_level = 0;
  nasd_uint16 protection;

  progname = argv[0];

  binding_type = NASD_BIND_DEFAULT;
  binding_args = NULL;
  binding_args_len = 0;

  while (nasd_getopt(argc, argv, "klMs:T", &c)) {
    switch(c) {
      case 'k':
        if (nondefault_binding)
          usage();
        nondefault_binding = 1;
        binding_type = NASD_BIND_KPDEV_DEFAULT;
        binding_args = &kpdev_args;
        binding_args_len = sizeof(kpdev_args);
        strcpy(kpdev_args.devname, "/dev/nasdkp0");
        break;
      case 'l':
        if (nondefault_binding)
          usage();
        nondefault_binding = 1;
        binding_type = NASD_BIND_COLOCATE;
        binding_args = &kpdev_args;
        binding_args_len = sizeof(kpdev_args);
        strcpy(kpdev_args.devname, "/dev/nasdkp0");
        break;
      case 'M':
        if (nondefault_binding)
          usage();
        nondefault_binding = 1;
        binding_type = NASD_BIND_MSGQ;
        break;
      case 's':
        if (sscanf(nasd_optarg, "%d", &sec_level) != 1) {
          usage();
        }
        break;
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE
      case 'T':
        if (nondefault_binding)
          usage();
        nondefault_binding = 1;
        binding_type = NASD_BIND_DCE_DIRECT_TCP;
        break;
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE */
      default:
        fprintf(stderr, "Unknown option '%c'\n", nasd_optopt);
        usage();
    }
  }

  if (nasd_optind >= argc)
    usage();
  server_name = argv[nasd_optind];
  nasd_optind++;

  if (nasd_optind >= argc)
    usage();
  master_password=argv[nasd_optind];
  nasd_optind++;

  if (nasd_optind < argc)
    usage();

  rc = nasd_sec_seclevel_to_protection(sec_level, &protection);
  if (rc) {
    fprintf(stderr, "ERROR: invalid security level %d\n", sec_level);
    usage();
  }

  rc = nasd_cl_p_init();
  if (rc) {
    fprintf(stderr, "ERROR: cannot init client library, rc=0x%x (%s)\n",
      rc, nasd_error_string(rc));
    exit(1);
  }

  rc = nasd_bind_to_drive(server_name, binding_port,
    binding_type, binding_args, binding_args_len, &h);
  if (rc) {
    fprintf(stderr, "ERROR: cannot bind to server %s error 0x%x (%s)\n",
      server_name, rc, nasd_error_string(rc));
    fflush(stderr);
    exit(1);
  }

  /* generate capability */
  
  if (sec_level > 0 ) {
    fprintf(stderr,"Warning.  This code untested under security\n");
  }

  nasd_sec_password_to_keys(master_password, 0 , &keys);
  nasd_drive_handle_get_time(h, &tm);
  tm.ts_sec += (60*60);
  nasd_sec_build_capability(0, NASD_CTRL_DRIVE_INFO,
                            NASD_ACCESS_RIGHTS_GETATTR, /* I have no idea what to put here */
			    0, tm.ts_sec,
                            protection, NASD_BLACK_CAPABILITY, 0, 0, 7,
                            keys.black_key, &cookie);

  /* fill in security param */
  sec_param.type = NASD_BLACK_CAPABILITY;
  sec_param.partnum = 0,
  sec_param.actual_protection = protection;

  /* do RPC */
  rc = nasd_cl_p_ctrl_get_drive_info(h, cookie.key, &sec_param, &cookie.capability,
                       &res);

  if (rc || status) {
    fprintf(stderr, "ERROR: nasd_status=0x%x (%s) status=0x%x (%s)\n",
      rc, nasd_error_string(rc),
      status, nasd_cl_error_string(h, status, error_text));
    exit(1);
  }

  nasd_printf("Drive on %s: %s\n",server_name,res.build_stamp);

  rc = nasd_unbind_drive(&h);
  if (rc) {
    fprintf(stderr, "ERROR: got 0x%x (%s) from nasd_unbind_drive()\n",
      rc, nasd_error_string(rc));
    fflush(stderr);
    exit(2);
  }

  nasd_cl_p_shutdown();

  exit(0);
}

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
