/*
 * nasd_rm_msgq.c
 *
 * Unconditionally remove the NASD drive's and/or EDRFS file manager's
 * message queues.
 *
 * Author: Marc Unangst
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#include <nasd/nasd_options.h>
#include <nasd/nasd_getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <nasd/nasd_msg.h>
#include <nasd/nasd_mq.h>
#include <nasd/nasd_edrfs_mq.h>

char *progname;

void
usage()
{
  fprintf(stderr, "USAGE: %s [options]\n", progname);
  fprintf(stderr, "Options:\n");
  fprintf(stderr, "  -d delete drive message queue\n");
  fprintf(stderr, "  -e delete EDRFS server message queue\n");
  fflush(stderr);
  exit(1);
}

void
nasd_remove_msgq(int key)
{
  int qid;
  int rc;

  qid = nasd_msg_msgget(key, 0);
  if(qid < 0) {
    fprintf(stderr, "ERROR: can't get message queue ID for key 0x%x: errno=%d (%s)\n",
            key, errno, strerror(errno));
    return;
  }
  rc = msgctl(qid, IPC_RMID, NULL);
  if(rc) {
    fprintf(stderr, "ERROR: can't remove message queue ID %d (key 0x%x): errno=%d (%s)\n",
	    qid, key, errno, strerror(errno));
    return;
  }
}

int
main(int argc, char **argv)
{
  int do_edrfs = 0;
  int do_drive = 0;
  char c;

  progname = argv[0];

  while(nasd_getopt(argc, argv, "de", &c)) {
    switch(c) {
    case 'd':
      do_drive = 1;
      break;
    case 'e':
      do_edrfs = 1;
      break;
    default:
      fprintf(stderr, "Unknown option '%c'\n", nasd_optopt);
      usage();
    }
  }

  if(!do_drive && !do_edrfs) {
    fprintf(stderr, "ERROR: must specify at least one of -d or -e\n");
    usage();
  }

  if(do_drive) {
    nasd_remove_msgq(NASD_MSGQ_KEY);
  }

  if(do_edrfs) {
    nasd_remove_msgq(NASD_EDRFS_MQ_KEY);
  }

  exit(0);
}

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
